"use strict";
// Copyright (c) Jupyter Development Team.
// Distributed under the terms of the Modified BSD License.
Object.defineProperty(exports, "__esModule", { value: true });
exports.WorkspaceManager = void 0;
const coreutils_1 = require("@jupyterlab/coreutils");
const statedb_1 = require("@jupyterlab/statedb");
const serverconnection_1 = require("../serverconnection");
/**
 * The url for the lab workspaces service.
 */
const SERVICE_WORKSPACES_URL = 'api/workspaces';
/**
 * The workspaces API service manager.
 */
class WorkspaceManager extends statedb_1.DataConnector {
    /**
     * Create a new workspace manager.
     */
    constructor(options = {}) {
        var _a;
        super();
        this.serverSettings =
            (_a = options.serverSettings) !== null && _a !== void 0 ? _a : serverconnection_1.ServerConnection.makeSettings();
    }
    /**
     * Fetch a workspace.
     *
     * @param id - The workspace's ID.
     *
     * @returns A promise that resolves if successful.
     */
    async fetch(id) {
        const { serverSettings } = this;
        const { baseUrl, appUrl } = serverSettings;
        const { makeRequest, ResponseError } = serverconnection_1.ServerConnection;
        const base = baseUrl + appUrl;
        const url = Private.url(base, id);
        const response = await makeRequest(url, {}, serverSettings);
        if (response.status !== 200) {
            const err = await ResponseError.create(response);
            throw err;
        }
        return response.json();
    }
    /**
     * Fetch the list of workspace IDs that exist on the server.
     *
     * @returns A promise that resolves if successful.
     */
    async list() {
        const { serverSettings } = this;
        const { baseUrl, appUrl } = serverSettings;
        const { makeRequest, ResponseError } = serverconnection_1.ServerConnection;
        const base = baseUrl + appUrl;
        const url = Private.url(base, '');
        const response = await makeRequest(url, {}, serverSettings);
        if (response.status !== 200) {
            const err = await ResponseError.create(response);
            throw err;
        }
        const result = await response.json();
        return result.workspaces;
    }
    /**
     * Remove a workspace from the server.
     *
     * @param id - The workspaces's ID.
     *
     * @returns A promise that resolves if successful.
     */
    async remove(id) {
        const { serverSettings } = this;
        const { baseUrl, appUrl } = serverSettings;
        const { makeRequest, ResponseError } = serverconnection_1.ServerConnection;
        const base = baseUrl + appUrl;
        const url = Private.url(base, id);
        const init = { method: 'DELETE' };
        const response = await makeRequest(url, init, serverSettings);
        if (response.status !== 204) {
            const err = await ResponseError.create(response);
            throw err;
        }
    }
    /**
     * Save a workspace.
     *
     * @param id - The workspace's ID.
     *
     * @param workspace - The workspace being saved.
     *
     * @returns A promise that resolves if successful.
     */
    async save(id, workspace) {
        const { serverSettings } = this;
        const { baseUrl, appUrl } = serverSettings;
        const { makeRequest, ResponseError } = serverconnection_1.ServerConnection;
        const base = baseUrl + appUrl;
        const url = Private.url(base, id);
        const init = { body: JSON.stringify(workspace), method: 'PUT' };
        const response = await makeRequest(url, init, serverSettings);
        if (response.status !== 204) {
            const err = await ResponseError.create(response);
            throw err;
        }
    }
}
exports.WorkspaceManager = WorkspaceManager;
/**
 * A namespace for private data.
 */
var Private;
(function (Private) {
    /**
     * Get the url for a workspace.
     */
    function url(base, id) {
        return coreutils_1.URLExt.join(base, SERVICE_WORKSPACES_URL, id);
    }
    Private.url = url;
})(Private || (Private = {}));
//# sourceMappingURL=index.js.map