(function (global, factory) {
    typeof exports === 'object' && typeof module !== 'undefined' ? factory(exports, require('@lumino/coreutils'), require('@lumino/signaling')) :
    typeof define === 'function' && define.amd ? define(['exports', '@lumino/coreutils', '@lumino/signaling'], factory) :
    (global = typeof globalThis !== 'undefined' ? globalThis : global || self, factory(global.lumino_polling = {}, global.lumino_coreutils, global.lumino_signaling));
})(this, (function (exports, coreutils, signaling) { 'use strict';

    // Copyright (c) Jupyter Development Team.
    /**
     * A class that wraps an asynchronous function to poll at a regular interval
     * with exponential increases to the interval length if the poll fails.
     *
     * @typeparam T - The resolved type of the factory's promises.
     * Defaults to `any`.
     *
     * @typeparam U - The rejected type of the factory's promises.
     * Defaults to `any`.
     *
     * @typeparam V - An optional type to extend the phases supported by a poll.
     * Defaults to `standby`, which already exists in the `Phase` type.
     */
    class Poll {
        /**
         * Instantiate a new poll with exponential backoff in case of failure.
         *
         * @param options - The poll instantiation options.
         */
        constructor(options) {
            var _a;
            this._disposed = new signaling.Signal(this);
            this._lingered = 0;
            this._tick = new coreutils.PromiseDelegate();
            this._ticked = new signaling.Signal(this);
            this._factory = options.factory;
            this._linger = (_a = options.linger) !== null && _a !== void 0 ? _a : Private.DEFAULT_LINGER;
            this._standby = options.standby || Private.DEFAULT_STANDBY;
            this._state = { ...Private.DEFAULT_STATE, timestamp: new Date().getTime() };
            // Normalize poll frequency `max` to be the greater of
            // default `max`, `options.frequency.max`, or `options.frequency.interval`.
            const frequency = options.frequency || {};
            const max = Math.max(frequency.interval || 0, frequency.max || 0, Private.DEFAULT_FREQUENCY.max);
            this.frequency = { ...Private.DEFAULT_FREQUENCY, ...frequency, ...{ max } };
            this.name = options.name || Private.DEFAULT_NAME;
            if ('auto' in options ? options.auto : true) {
                setTimeout(() => this.start());
            }
        }
        /**
         * A signal emitted when the poll is disposed.
         */
        get disposed() {
            return this._disposed;
        }
        /**
         * The polling frequency parameters.
         */
        get frequency() {
            return this._frequency;
        }
        set frequency(frequency) {
            if (this.isDisposed || coreutils.JSONExt.deepEqual(frequency, this.frequency || {})) {
                return;
            }
            let { backoff, interval, max } = frequency;
            interval = Math.round(interval);
            max = Math.round(max);
            if (typeof backoff === 'number' && backoff < 1) {
                throw new Error('Poll backoff growth factor must be at least 1');
            }
            if ((interval < 0 || interval > max) && interval !== Poll.NEVER) {
                throw new Error('Poll interval must be between 0 and max');
            }
            if (max > Poll.MAX_INTERVAL && max !== Poll.NEVER) {
                throw new Error(`Max interval must be less than ${Poll.MAX_INTERVAL}`);
            }
            this._frequency = { backoff, interval, max };
        }
        /**
         * Whether the poll is disposed.
         */
        get isDisposed() {
            return this.state.phase === 'disposed';
        }
        /**
         * Indicates when the poll switches to standby.
         */
        get standby() {
            return this._standby;
        }
        set standby(standby) {
            if (this.isDisposed || this.standby === standby) {
                return;
            }
            this._standby = standby;
        }
        /**
         * The poll state, which is the content of the current poll tick.
         */
        get state() {
            return this._state;
        }
        /**
         * A promise that resolves when the poll next ticks.
         */
        get tick() {
            return this._tick.promise;
        }
        /**
         * A signal emitted when the poll ticks and fires off a new request.
         */
        get ticked() {
            return this._ticked;
        }
        /**
         * Return an async iterator that yields every tick.
         */
        async *[Symbol.asyncIterator]() {
            while (!this.isDisposed) {
                yield this.state;
                await this.tick.catch(() => undefined);
            }
        }
        /**
         * Dispose the poll.
         */
        dispose() {
            if (this.isDisposed) {
                return;
            }
            this._state = {
                ...Private.DISPOSED_STATE,
                timestamp: new Date().getTime()
            };
            this._tick.promise.catch(_ => undefined);
            this._tick.reject(new Error(`Poll (${this.name}) is disposed.`));
            this._disposed.emit(undefined);
            signaling.Signal.clearData(this);
        }
        /**
         * Refreshes the poll. Schedules `refreshed` tick if necessary.
         *
         * @returns A promise that resolves after tick is scheduled and never rejects.
         *
         * #### Notes
         * The returned promise resolves after the tick is scheduled, but before
         * the polling action is run. To wait until after the poll action executes,
         * await the `poll.tick` promise: `await poll.refresh(); await poll.tick;`
         */
        refresh() {
            return this.schedule({
                cancel: ({ phase }) => phase === 'refreshed',
                interval: Poll.IMMEDIATE,
                phase: 'refreshed'
            });
        }
        /**
         * Schedule the next poll tick.
         *
         * @param next - The next poll state data to schedule. Defaults to standby.
         *
         * @param next.cancel - Cancels state transition if function returns `true`.
         *
         * @returns A promise that resolves when the next poll state is active.
         *
         * #### Notes
         * This method is not meant to be invoked by user code typically. It is public
         * to allow poll instances to be composed into classes that schedule ticks.
         */
        async schedule(next = {}) {
            if (this.isDisposed) {
                return;
            }
            // Check if the phase transition should be canceled.
            if (next.cancel && next.cancel(this.state)) {
                return;
            }
            // Update poll state.
            const pending = this._tick;
            const scheduled = new coreutils.PromiseDelegate();
            const state = {
                interval: this.frequency.interval,
                payload: null,
                phase: 'standby',
                timestamp: new Date().getTime(),
                ...next
            };
            this._state = state;
            this._tick = scheduled;
            // Clear the schedule if possible.
            clearTimeout(this._timeout);
            // Emit ticked signal, resolve pending promise, and await its settlement.
            this._ticked.emit(this.state);
            pending.resolve(this);
            await pending.promise;
            if (state.interval === Poll.NEVER) {
                this._timeout = undefined;
                return;
            }
            // Schedule next execution and cache its timeout handle.
            const execute = () => {
                if (this.isDisposed || this.tick !== scheduled.promise) {
                    return;
                }
                this._execute();
            };
            // Cache the handle in case it needs to be unscheduled.
            this._timeout = setTimeout(execute, state.interval);
        }
        /**
         * Starts the poll. Schedules `started` tick if necessary.
         *
         * @returns A promise that resolves after tick is scheduled and never rejects.
         */
        start() {
            return this.schedule({
                cancel: ({ phase }) => phase !== 'constructed' && phase !== 'standby' && phase !== 'stopped',
                interval: Poll.IMMEDIATE,
                phase: 'started'
            });
        }
        /**
         * Stops the poll. Schedules `stopped` tick if necessary.
         *
         * @returns A promise that resolves after tick is scheduled and never rejects.
         */
        stop() {
            return this.schedule({
                cancel: ({ phase }) => phase === 'stopped',
                interval: Poll.NEVER,
                phase: 'stopped'
            });
        }
        /**
         * Whether the poll is hidden.
         *
         * #### Notes
         * This property is only relevant in a browser context.
         */
        get hidden() {
            return Private.hidden;
        }
        /**
         * Execute a new poll factory promise or stand by if necessary.
         */
        _execute() {
            let standby = typeof this.standby === 'function' ? this.standby() : this.standby;
            // Check if execution should proceed, linger, or stand by.
            if (standby === 'never') {
                standby = false;
            }
            else if (standby === 'when-hidden') {
                if (this.hidden) {
                    standby = ++this._lingered > this._linger;
                }
                else {
                    this._lingered = 0;
                    standby = false;
                }
            }
            // If in standby mode schedule next tick without calling the factory.
            if (standby) {
                void this.schedule();
                return;
            }
            const pending = this.tick;
            this._factory(this.state)
                .then((resolved) => {
                if (this.isDisposed || this.tick !== pending) {
                    return;
                }
                void this.schedule({
                    payload: resolved,
                    phase: this.state.phase === 'rejected' ? 'reconnected' : 'resolved'
                });
            })
                .catch((rejected) => {
                if (this.isDisposed || this.tick !== pending) {
                    return;
                }
                void this.schedule({
                    interval: Private.sleep(this.frequency, this.state),
                    payload: rejected,
                    phase: 'rejected'
                });
            });
        }
    }
    /**
     * A namespace for `Poll` types, interfaces, and statics.
     */
    (function (Poll) {
        /**
         * An interval value in ms that indicates the poll should tick immediately.
         */
        Poll.IMMEDIATE = 0;
        /**
         * Delays are 32-bit integers in many browsers so intervals need to be capped.
         *
         * #### Notes
         * https://developer.mozilla.org/en-US/docs/Web/API/WindowOrWorkerGlobalScope/setTimeout#Maximum_delay_value
         */
        Poll.MAX_INTERVAL = 2147483647;
        /**
         * An interval value that indicates the poll should never tick.
         */
        Poll.NEVER = Infinity;
    })(Poll || (Poll = {}));
    /**
     * A namespace for private module data.
     */
    var Private;
    (function (Private) {
        /**
         * The default backoff growth rate if `backoff` is `true`.
         */
        Private.DEFAULT_BACKOFF = 3;
        /**
         * The default polling frequency.
         */
        Private.DEFAULT_FREQUENCY = {
            backoff: true,
            interval: 1000,
            max: 30 * 1000
        };
        /**
         * The default number of times to `linger` when a poll is hidden.
         */
        Private.DEFAULT_LINGER = 1;
        /**
         * The default poll name.
         */
        Private.DEFAULT_NAME = 'unknown';
        /**
         * The default poll standby behavior.
         */
        Private.DEFAULT_STANDBY = 'when-hidden';
        /**
         * The first poll tick state's default values superseded in constructor.
         */
        Private.DEFAULT_STATE = {
            interval: Poll.NEVER,
            payload: null,
            phase: 'constructed',
            timestamp: new Date(0).getTime()
        };
        /**
         * The disposed tick state values.
         */
        Private.DISPOSED_STATE = {
            interval: Poll.NEVER,
            payload: null,
            phase: 'disposed',
            timestamp: new Date(0).getTime()
        };
        /**
         * Returns the number of milliseconds to sleep before the next tick.
         *
         * @param frequency - The poll's base frequency.
         * @param last - The poll's last tick.
         */
        function sleep(frequency, last) {
            const { backoff, interval, max } = frequency;
            if (interval === Poll.NEVER) {
                return interval;
            }
            const growth = backoff === true ? Private.DEFAULT_BACKOFF : backoff === false ? 1 : backoff;
            const random = getRandomIntInclusive(interval, last.interval * growth);
            return Math.min(max, random);
        }
        Private.sleep = sleep;
        /**
         * Keep track of whether the document is hidden. This flag is only relevant in
         * a browser context.
         *
         * Listen to `visibilitychange` event to set the `hidden` flag.
         *
         * Listening to `pagehide` is also necessary because Safari support for
         * `visibilitychange` events is partial, cf.
         * https://developer.mozilla.org/docs/Web/API/Document/visibilitychange_event
         */
        Private.hidden = (() => {
            if (typeof document === 'undefined') {
                return false;
            }
            document.addEventListener('visibilitychange', () => {
                Private.hidden = document.visibilityState === 'hidden';
            });
            document.addEventListener('pagehide', () => {
                Private.hidden = document.visibilityState === 'hidden';
            });
            return document.visibilityState === 'hidden';
        })();
        /**
         * Get a random integer between min and max, inclusive of both.
         *
         * #### Notes
         * From
         * https://developer.mozilla.org/docs/Web/JavaScript/Reference/Global_Objects/Math/random#Getting_a_random_integer_between_two_values_inclusive
         *
         * From the MDN page: It might be tempting to use Math.round() to accomplish
         * that, but doing so would cause your random numbers to follow a non-uniform
         * distribution, which may not be acceptable for your needs.
         */
        function getRandomIntInclusive(min, max) {
            min = Math.ceil(min);
            max = Math.floor(max);
            return Math.floor(Math.random() * (max - min + 1)) + min;
        }
    })(Private || (Private = {}));

    // Copyright (c) Jupyter Development Team.
    /**
     * A base class to implement rate limiters with different invocation strategies.
     *
     * @typeparam T - The resolved type of the underlying function.
     *
     * @typeparam U - The rejected type of the underlying function.
     *
     * @typeparam V - Arguments for the underlying function.
     */
    class RateLimiter {
        /**
         * Instantiate a rate limiter.
         *
         * @param fn - The function to rate limit.
         *
         * @param limit - The rate limit; defaults to 500ms.
         */
        constructor(fn, limit = 500) {
            /**
             * Arguments for the underlying function.
             */
            this.args = undefined;
            /**
             * A promise that resolves on each successful invocation.
             */
            this.payload = null;
            this.limit = limit;
            this.poll = new Poll({
                auto: false,
                factory: async () => {
                    const { args } = this;
                    this.args = undefined;
                    return fn(...args);
                },
                frequency: { backoff: false, interval: Poll.NEVER, max: Poll.NEVER },
                standby: 'never'
            });
            this.payload = new coreutils.PromiseDelegate();
            this.poll.ticked.connect((_, state) => {
                const { payload } = this;
                if (state.phase === 'resolved') {
                    this.payload = new coreutils.PromiseDelegate();
                    payload.resolve(state.payload);
                    return;
                }
                if (state.phase === 'rejected' || state.phase === 'stopped') {
                    this.payload = new coreutils.PromiseDelegate();
                    payload.promise.catch(_ => undefined);
                    payload.reject(state.payload);
                    return;
                }
            }, this);
        }
        /**
         * Whether the rate limiter is disposed.
         */
        get isDisposed() {
            return this.payload === null;
        }
        /**
         * Disposes the rate limiter.
         */
        dispose() {
            if (this.isDisposed) {
                return;
            }
            this.args = undefined;
            this.payload = null;
            this.poll.dispose();
        }
        /**
         * Stop the function if it is mid-flight.
         */
        async stop() {
            return this.poll.stop();
        }
    }
    /**
     * Wraps and debounces a function that can be called multiple times and only
     * executes the underlying function one `interval` after the last invocation.
     *
     * @typeparam T - The resolved type of the underlying function. Defaults to any.
     *
     * @typeparam U - The rejected type of the underlying function. Defaults to any.
     *
     * @typeparam V - Arguments for the underlying function. Defaults to any[].
     */
    class Debouncer extends RateLimiter {
        /**
         * Invokes the function and only executes after rate limit has elapsed.
         * Each invocation resets the timer.
         */
        invoke(...args) {
            this.args = args;
            void this.poll.schedule({ interval: this.limit, phase: 'invoked' });
            return this.payload.promise;
        }
    }
    /**
     * Wraps and throttles a function that can be called multiple times and only
     * executes the underlying function once per `interval`.
     *
     * @typeparam T - The resolved type of the underlying function. Defaults to any.
     *
     * @typeparam U - The rejected type of the underlying function. Defaults to any.
     *
     * @typeparam V - Arguments for the underlying function. Defaults to any[].
     */
    class Throttler extends RateLimiter {
        /**
         * Instantiate a throttler.
         *
         * @param fn - The function being throttled.
         *
         * @param options - Throttling configuration or throttling limit in ms.
         *
         * #### Notes
         * The `edge` defaults to `leading`; the `limit` defaults to `500`.
         */
        constructor(fn, options) {
            super(fn, typeof options === 'number' ? options : options && options.limit);
            this._trailing = false;
            if (typeof options !== 'number' && options && options.edge === 'trailing') {
                this._trailing = true;
            }
            this._interval = this._trailing ? this.limit : Poll.IMMEDIATE;
        }
        /**
         * Throttles function invocations if one is currently in flight.
         */
        invoke(...args) {
            const idle = this.poll.state.phase !== 'invoked';
            if (idle || this._trailing) {
                this.args = args;
            }
            if (idle) {
                void this.poll.schedule({ interval: this._interval, phase: 'invoked' });
            }
            return this.payload.promise;
        }
    }

    exports.Debouncer = Debouncer;
    exports.Poll = Poll;
    exports.RateLimiter = RateLimiter;
    exports.Throttler = Throttler;

}));
//# sourceMappingURL=index.js.map
