"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.findConfigFile = findConfigFile;
exports.readDefaultConfig = readDefaultConfig;
var _debug = _interopRequireDefault(require("debug"));
var _fs = _interopRequireDefault(require("fs"));
var _lodash = _interopRequireDefault(require("lodash"));
var _path = _interopRequireDefault(require("path"));
var _core = require("@verdaccio/core");
var _configUtils = require("./config-utils");
function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }
const CONFIG_FILE = 'config.yaml';
const XDG = 'xdg';
const WIN = 'win';
const WIN32 = 'win32';
// eslint-disable-next-line
const pkgJSON = {
  name: 'verdaccio'
};
const debug = (0, _debug.default)('verdaccio:config');

/**
 * Find and get the first config file that match.
 * @return {String} the config file path
 */
function findConfigFile(configPath) {
  if (typeof configPath !== 'undefined') {
    return _path.default.resolve(configPath);
  }
  const configPaths = getConfigPaths();
  debug('%o posible locations found', configPaths.length);
  if (_lodash.default.isEmpty(configPaths)) {
    // this should never happens
    throw new Error('no configuration files can be processed');
  }

  // find the first location that already exist
  const primaryConf = _lodash.default.find(configPaths, configLocation => (0, _configUtils.fileExists)(configLocation.path));
  if (typeof primaryConf !== 'undefined') {
    debug('previous location exist already %s', primaryConf === null || primaryConf === void 0 ? void 0 : primaryConf.path);
    return primaryConf.path;
  }

  // @ts-ignore
  return createConfigFile(_lodash.default.head(configPaths)).path;
}
function createConfigFile(configLocation) {
  createConfigFolder(configLocation);
  const defaultConfig = updateStorageLinks(configLocation, readDefaultConfig());
  _fs.default.writeFileSync(configLocation.path, defaultConfig);
  return configLocation;
}
function readDefaultConfig() {
  const pathDefaultConf = _path.default.resolve(__dirname, 'conf/default.yaml');
  try {
    debug('default configuration file %s', pathDefaultConf);
    _fs.default.accessSync(pathDefaultConf, _fs.default.constants.R_OK);
  } catch {
    throw new TypeError('configuration file does not have enough permissions for reading');
  }
  // @ts-ignore
  return _fs.default.readFileSync(pathDefaultConf, _core.CHARACTER_ENCODING.UTF8);
}
function createConfigFolder(configLocation) {
  _fs.default.mkdirSync(_path.default.dirname(configLocation.path), {
    recursive: true
  });
  debug(`Creating default config file in %o`, configLocation === null || configLocation === void 0 ? void 0 : configLocation.path);
}
function updateStorageLinks(configLocation, defaultConfig) {
  if (configLocation.type !== XDG) {
    return defaultConfig;
  }

  // $XDG_DATA_HOME defines the base directory relative to which user specific data
  // files should be stored, If $XDG_DATA_HOME is either not set or empty, a default
  // equal to $HOME/.local/share should be used.
  let dataDir = process.env.XDG_DATA_HOME || _path.default.join(process.env.HOME, '.local', 'share');
  if ((0, _configUtils.folderExists)(dataDir)) {
    debug(`previous storage located`);
    debug(`update storage links to %s`, dataDir);
    dataDir = _path.default.resolve(_path.default.join(dataDir, pkgJSON.name, 'storage'));
    return defaultConfig.replace(/^storage: .\/storage$/m, `storage: ${dataDir}`);
  }
  debug(`could not find a previous storage location, skip override`);
  return defaultConfig;
}

/**
 * Return a list of configuration locations by platform.
 * @returns
 */
function getConfigPaths() {
  const listPaths = [getXDGDirectory(), getWindowsDirectory(), getRelativeDefaultDirectory(), getOldDirectory()];
  return listPaths.reduce(function (acc, currentValue) {
    if (typeof currentValue !== 'undefined') {
      debug('directory detected path %s for type %s', currentValue === null || currentValue === void 0 ? void 0 : currentValue.path, currentValue.type);
      acc.push(currentValue);
    }
    return acc;
  }, []);
}

/**
 * Get XDG_CONFIG_HOME or HOME location (usually unix)
 * @returns
 */
const getXDGDirectory = () => {
  const xDGConfigPath = process.env.XDG_CONFIG_HOME || process.env.HOME && _path.default.join(process.env.HOME, '.config');
  if (xDGConfigPath && (0, _configUtils.folderExists)(xDGConfigPath)) {
    debug('XDGConfig folder path %s', xDGConfigPath);
    return {
      path: _path.default.join(xDGConfigPath, pkgJSON.name, CONFIG_FILE),
      type: XDG
    };
  }
};

/**
 * Detect windows location, APPDATA
 * usually something like C:\User\<Build User>\AppData\Local
 * @returns
 */
const getWindowsDirectory = () => {
  if (process.platform === WIN32 && process.env.APPDATA && (0, _configUtils.folderExists)(process.env.APPDATA)) {
    debug('is windows appdata: %s', process.env.APPDATA);
    return {
      path: _path.default.resolve(_path.default.join(process.env.APPDATA, pkgJSON.name, CONFIG_FILE)),
      type: WIN
    };
  }
};

/**
 * Return relative directory, this is the default.
 * It will cretate config in your {currentLocation/verdaccio/config.yaml}
 * @returns
 */
const getRelativeDefaultDirectory = () => {
  return {
    path: _path.default.resolve(_path.default.join('.', pkgJSON.name, CONFIG_FILE)),
    type: 'def'
  };
};

/**
 * This should never happens, consider it DEPRECATED
 * @returns
 */
const getOldDirectory = () => {
  return {
    path: _path.default.resolve(_path.default.join('.', CONFIG_FILE)),
    type: 'old'
  };
};
//# sourceMappingURL=config-path.js.map