import { ISignal, Signal } from '@lumino/signaling';
import { IAttachmentsModel } from '@jupyterlab/attachments';
import { CodeEditor } from '@jupyterlab/codeeditor';
import { IChangedArgs } from '@jupyterlab/coreutils';
import * as nbformat from '@jupyterlab/nbformat';
import { ObservableValue } from '@jupyterlab/observables';
import { IOutputAreaModel } from '@jupyterlab/outputarea';
import { IMapChange, ISharedCell, ISharedCodeCell, ISharedMarkdownCell, ISharedRawCell } from '@jupyter/ydoc';
/**
 * The definition of a model object for a cell.
 */
export interface ICellModel extends CodeEditor.IModel {
    /**
     * The type of the cell.
     */
    readonly type: nbformat.CellType;
    /**
     * A unique identifier for the cell.
     */
    readonly id: string;
    /**
     * A signal emitted when the content of the model changes.
     */
    readonly contentChanged: ISignal<ICellModel, void>;
    /**
     * A signal emitted when a model state changes.
     */
    readonly stateChanged: ISignal<ICellModel, IChangedArgs<boolean, boolean, any>>;
    /**
     * Whether the cell is trusted.
     */
    trusted: boolean;
    /**
     * The metadata associated with the cell.
     *
     * ### Notes
     * This is a copy of the metadata. Changing a part of it
     * won't affect the model.
     * As this returns a copy of all metadata, it is advised to
     * use `getMetadata` to speed up the process of getting a single key.
     */
    readonly metadata: Omit<nbformat.IBaseCellMetadata, 'trusted'>;
    /**
     * Signal emitted when cell metadata changes.
     */
    readonly metadataChanged: ISignal<ICellModel, IMapChange>;
    /**
     * The cell shared model.
     */
    readonly sharedModel: ISharedCell;
    /**
     * Delete a metadata.
     *
     * @param key Metadata key
     */
    deleteMetadata(key: string): void;
    /**
     * Get a metadata
     *
     * ### Notes
     * This returns a copy of the key value.
     *
     * @param key Metadata key
     */
    getMetadata(key: string): any;
    /**
     * Set a metadata
     *
     * @param key Metadata key
     * @param value Metadata value
     */
    setMetadata(key: string, value: any): void;
    /**
     * Serialize the model to JSON.
     */
    toJSON(): nbformat.ICell;
}
/**
 * The definition of a model cell object for a cell with attachments.
 */
export interface IAttachmentsCellModel extends ICellModel {
    /**
     * The cell attachments
     */
    readonly attachments: IAttachmentsModel;
}
/**
 * The definition of a code cell.
 */
export interface ICodeCellModel extends ICellModel {
    /**
     * The type of the cell.
     *
     * #### Notes
     * This is a read-only property.
     */
    readonly type: 'code';
    /**
     * Whether the code cell has been edited since the last run.
     */
    readonly isDirty: boolean;
    /**
     * Serialize the model to JSON.
     */
    toJSON(): nbformat.ICodeCell;
    /**
     * The code cell's prompt number. Will be null if the cell has not been run.
     */
    executionCount: nbformat.ExecutionCount;
    /**
     * The cell outputs.
     */
    readonly outputs: IOutputAreaModel;
    /**
     * Clear execution, outputs, and related metadata
     */
    clearExecution(): void;
    /**
     * The code cell shared model
     */
    readonly sharedModel: ISharedCodeCell;
}
/**
 * The definition of a markdown cell.
 */
export interface IMarkdownCellModel extends IAttachmentsCellModel {
    /**
     * The type of the cell.
     */
    readonly type: 'markdown';
    /**
     * Serialize the model to JSON.
     */
    toJSON(): nbformat.IMarkdownCell;
}
/**
 * The definition of a raw cell.
 */
export interface IRawCellModel extends IAttachmentsCellModel {
    /**
     * The type of the cell.
     */
    readonly type: 'raw';
    /**
     * Serialize the model to JSON.
     */
    toJSON(): nbformat.IRawCell;
}
export declare function isCodeCellModel(model: ICellModel): model is ICodeCellModel;
export declare function isMarkdownCellModel(model: ICellModel): model is IMarkdownCellModel;
export declare function isRawCellModel(model: ICellModel): model is IRawCellModel;
/**
 * An implementation of the cell model.
 */
export declare abstract class CellModel extends CodeEditor.Model implements ICellModel {
    constructor(options?: CellModel.IOptions<ISharedCell>);
    readonly sharedModel: ISharedCell;
    /**
     * The type of cell.
     */
    abstract get type(): nbformat.CellType;
    /**
     * A signal emitted when the state of the model changes.
     */
    readonly contentChanged: Signal<this, void>;
    /**
     * Signal emitted when cell metadata changes.
     */
    get metadataChanged(): ISignal<ICellModel, IMapChange>;
    /**
     * A signal emitted when a model state changes.
     */
    readonly stateChanged: Signal<this, IChangedArgs<any, any, "trusted" | "isDirty" | "executionCount">>;
    /**
     * The id for the cell.
     */
    get id(): string;
    /**
     * The metadata associated with the cell.
     */
    get metadata(): Omit<nbformat.IBaseCellMetadata, 'trusted'>;
    /**
     * The trusted state of the model.
     */
    get trusted(): boolean;
    set trusted(newValue: boolean);
    /**
     * Dispose of the resources held by the model.
     */
    dispose(): void;
    /**
     * Handle a change to the trusted state.
     *
     * The default implementation is a no-op.
     */
    onTrustedChanged(trusted: CellModel, args: ObservableValue.IChangedArgs): void;
    /**
     * Delete a metadata
     *
     * @param key Metadata key
     */
    deleteMetadata(key: string): any;
    /**
     * Get a metadata
     *
     * ### Notes
     * This returns a copy of the key value.
     *
     * @param key Metadata key
     */
    getMetadata(key: string): any;
    /**
     * Set a metadata
     *
     * @param key Metadata key
     * @param value Metadata value
     */
    setMetadata(key: string, value: any): void;
    /**
     * Serialize the model to JSON.
     */
    toJSON(): nbformat.ICell;
    /**
     * Handle a change to the observable value.
     */
    protected onGenericChange(): void;
    private _onMetadataChanged;
    private _metadataChanged;
    private _trusted;
}
/**
 * The namespace for `CellModel` statics.
 */
export declare namespace CellModel {
    /**
     * The options used to initialize a `CellModel`.
     */
    interface IOptions<T extends ISharedCell> {
        /**
         * A unique identifier for the model.
         */
        id?: string;
        /**
         * The cell shared model.
         */
        sharedModel?: T;
        /**
         * The cell type
         */
        cell_type?: string;
        /**
         * Whether the cell is trusted or not.
         */
        trusted?: boolean;
    }
}
/**
 * A base implementation for cell models with attachments.
 */
export declare abstract class AttachmentsCellModel extends CellModel {
    /**
     * Construct a new cell with optional attachments.
     */
    constructor(options: AttachmentsCellModel.IOptions<ISharedCell>);
    /**
     * Get the attachments of the model.
     */
    get attachments(): IAttachmentsModel;
    /**
     * Dispose of the resources held by the model.
     */
    dispose(): void;
    /**
     * Serialize the model to JSON.
     */
    toJSON(): nbformat.IRawCell | nbformat.IMarkdownCell;
    /**
     * Handle a change to the cell outputs modelDB and reflect it in the shared model.
     */
    private _onAttachmentsChange;
    /**
     * Handle a change to the code cell value.
     */
    private _onSharedModelChanged;
    private _attachments;
}
/**
 * The namespace for `AttachmentsCellModel` statics.
 */
export declare namespace AttachmentsCellModel {
    /**
     * The options used to initialize a `AttachmentsCellModel`.
     */
    interface IOptions<T extends ISharedCell> extends CellModel.IOptions<T> {
        /**
         * The factory for attachment model creation.
         */
        contentFactory?: IContentFactory;
    }
    /**
     * A factory for creating code cell model content.
     */
    interface IContentFactory {
        /**
         * Create an output area.
         */
        createAttachmentsModel(options: IAttachmentsModel.IOptions): IAttachmentsModel;
    }
    /**
     * The default implementation of an `IContentFactory`.
     */
    class ContentFactory implements IContentFactory {
        /**
         * Create an attachments model.
         */
        createAttachmentsModel(options: IAttachmentsModel.IOptions): IAttachmentsModel;
    }
    /**
     * The shared `ContentFactory` instance.
     */
    const defaultContentFactory: ContentFactory;
}
/**
 * An implementation of a raw cell model.
 */
export declare class RawCellModel extends AttachmentsCellModel {
    /**
     * Construct a raw cell model from optional shared model.
     */
    constructor(options?: Omit<AttachmentsCellModel.IOptions<ISharedRawCell>, 'cell_type'>);
    /**
     * The type of the cell.
     */
    get type(): 'raw';
    /**
     * Serialize the model to JSON.
     */
    toJSON(): nbformat.IRawCell;
}
/**
 * An implementation of a markdown cell model.
 */
export declare class MarkdownCellModel extends AttachmentsCellModel {
    /**
     * Construct a markdown cell model from optional shared model.
     */
    constructor(options?: Omit<AttachmentsCellModel.IOptions<ISharedMarkdownCell>, 'cell_type'>);
    /**
     * The type of the cell.
     */
    get type(): 'markdown';
    /**
     * Serialize the model to JSON.
     */
    toJSON(): nbformat.IMarkdownCell;
}
/**
 * An implementation of a code cell Model.
 */
export declare class CodeCellModel extends CellModel implements ICodeCellModel {
    /**
     * Construct a new code cell with optional original cell content.
     */
    constructor(options?: CodeCellModel.IOptions);
    /**
     * The type of the cell.
     */
    get type(): 'code';
    /**
     * The execution count of the cell.
     */
    get executionCount(): nbformat.ExecutionCount;
    set executionCount(newValue: nbformat.ExecutionCount);
    /**
     * Whether the cell is dirty or not.
     *
     * A cell is dirty if it is output is not empty and does not
     * result of the input code execution.
     */
    get isDirty(): boolean;
    /**
     * The cell outputs.
     */
    get outputs(): IOutputAreaModel;
    readonly sharedModel: ISharedCodeCell;
    clearExecution(): void;
    /**
     * Dispose of the resources held by the model.
     */
    dispose(): void;
    /**
     * Handle a change to the trusted state.
     */
    onTrustedChanged(trusted: CellModel, args: ObservableValue.IChangedArgs): void;
    /**
     * Serialize the model to JSON.
     */
    toJSON(): nbformat.ICodeCell;
    /**
     * Handle a change to the cell outputs modelDB and reflect it in the shared model.
     */
    protected onOutputsChange(sender: IOutputAreaModel, event: IOutputAreaModel.ChangedArgs): void;
    /**
     * Handle a change to the code cell value.
     */
    private _onSharedModelChanged;
    /**
     * Set whether the cell is dirty or not.
     */
    private _setDirty;
    private _executedCode;
    private _isDirty;
    private _outputs;
}
/**
 * The namespace for `CodeCellModel` statics.
 */
export declare namespace CodeCellModel {
    /**
     * The options used to initialize a `CodeCellModel`.
     */
    interface IOptions extends Omit<CellModel.IOptions<ISharedCodeCell>, 'cell_type'> {
        /**
         * The factory for output area model creation.
         */
        contentFactory?: IContentFactory;
    }
    /**
     * A factory for creating code cell model content.
     */
    interface IContentFactory {
        /**
         * Create an output area.
         */
        createOutputArea(options: IOutputAreaModel.IOptions): IOutputAreaModel;
    }
    /**
     * The default implementation of an `IContentFactory`.
     */
    class ContentFactory implements IContentFactory {
        /**
         * Create an output area.
         */
        createOutputArea(options: IOutputAreaModel.IOptions): IOutputAreaModel;
    }
    /**
     * The shared `ContentFactory` instance.
     */
    const defaultContentFactory: ContentFactory;
}
