// Copyright (c) Jupyter Development Team.
// Distributed under the terms of the Modified BSD License.
/**
 * @packageDocumentation
 * @module launcher-extension
 */
import { ILabShell } from '@jupyterlab/application';
import { ICommandPalette, MainAreaWidget } from '@jupyterlab/apputils';
import { IDefaultFileBrowser, IFileBrowserFactory } from '@jupyterlab/filebrowser';
import { ILauncher, Launcher, LauncherModel } from '@jupyterlab/launcher';
import { ITranslator } from '@jupyterlab/translation';
import { addIcon, launcherIcon } from '@jupyterlab/ui-components';
import { find } from '@lumino/algorithm';
/**
 * The command IDs used by the launcher plugin.
 */
var CommandIDs;
(function (CommandIDs) {
    CommandIDs.create = 'launcher:create';
})(CommandIDs || (CommandIDs = {}));
/**
 * A service providing an interface to the the launcher.
 */
const plugin = {
    activate,
    id: '@jupyterlab/launcher-extension:plugin',
    requires: [ITranslator],
    optional: [
        ILabShell,
        ICommandPalette,
        IDefaultFileBrowser,
        IFileBrowserFactory
    ],
    provides: ILauncher,
    autoStart: true
};
/**
 * Export the plugin as default.
 */
export default plugin;
/**
 * Activate the launcher.
 */
function activate(app, translator, labShell, palette, defaultBrowser, factory) {
    const { commands, shell } = app;
    const trans = translator.load('jupyterlab');
    const model = new LauncherModel();
    commands.addCommand(CommandIDs.create, {
        label: trans.__('New Launcher'),
        icon: args => (args.toolbar ? addIcon : undefined),
        execute: (args) => {
            var _a, _b;
            const cwd = (_b = (_a = args['cwd']) !== null && _a !== void 0 ? _a : defaultBrowser === null || defaultBrowser === void 0 ? void 0 : defaultBrowser.model.path) !== null && _b !== void 0 ? _b : '';
            const id = `launcher-${Private.id++}`;
            const callback = (item) => {
                // If widget is attached to the main area replace the launcher
                if (find(shell.widgets('main'), w => w === item)) {
                    shell.add(item, 'main', { ref: id });
                    launcher.dispose();
                }
            };
            const launcher = new Launcher({
                model,
                cwd,
                callback,
                commands,
                translator
            });
            launcher.model = model;
            launcher.title.icon = launcherIcon;
            launcher.title.label = trans.__('Launcher');
            const main = new MainAreaWidget({ content: launcher });
            // If there are any other widgets open, remove the launcher close icon.
            main.title.closable = !!Array.from(shell.widgets('main')).length;
            main.id = id;
            shell.add(main, 'main', {
                activate: args['activate'],
                ref: args['ref']
            });
            if (labShell) {
                labShell.layoutModified.connect(() => {
                    // If there is only a launcher open, remove the close icon.
                    main.title.closable = Array.from(labShell.widgets('main')).length > 1;
                }, main);
            }
            if (defaultBrowser) {
                const onPathChanged = (model) => {
                    launcher.cwd = model.path;
                };
                defaultBrowser.model.pathChanged.connect(onPathChanged);
                launcher.disposed.connect(() => {
                    defaultBrowser.model.pathChanged.disconnect(onPathChanged);
                });
            }
            return main;
        }
    });
    if (labShell) {
        void Promise.all([app.restored, defaultBrowser === null || defaultBrowser === void 0 ? void 0 : defaultBrowser.model.restored]).then(() => {
            function maybeCreate() {
                // Create a launcher if there are no open items.
                if (labShell.isEmpty('main')) {
                    void commands.execute(CommandIDs.create);
                }
            }
            // When layout is modified, create a launcher if there are no open items.
            labShell.layoutModified.connect(() => {
                maybeCreate();
            });
        });
    }
    if (palette) {
        palette.addItem({
            command: CommandIDs.create,
            category: trans.__('Launcher')
        });
    }
    if (labShell) {
        labShell.addButtonEnabled = true;
        labShell.addRequested.connect((sender, arg) => {
            var _a;
            // Get the ref for the current tab of the tabbar which the add button was clicked
            const ref = ((_a = arg.currentTitle) === null || _a === void 0 ? void 0 : _a.owner.id) ||
                arg.titles[arg.titles.length - 1].owner.id;
            return commands.execute(CommandIDs.create, { ref });
        });
    }
    return model;
}
/**
 * The namespace for module private data.
 */
var Private;
(function (Private) {
    /**
     * The incrementing id used for launcher widgets.
     */
    Private.id = 0;
})(Private || (Private = {}));
//# sourceMappingURL=index.js.map