"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ErrorCode = void 0;
exports.addGravatarSupport = addGravatarSupport;
exports.addScope = addScope;
Object.defineProperty(exports, "buildToken", {
  enumerable: true,
  get: function () {
    return _utils.buildToken;
  }
});
exports.convertPayloadToBase64 = convertPayloadToBase64;
exports.deleteProperties = deleteProperties;
exports.encodeScopedUri = encodeScopedUri;
exports.fileExists = fileExists;
exports.folderExists = folderExists;
exports.formatAuthor = formatAuthor;
exports.getVersion = getVersion;
exports.hasDiffOneKey = hasDiffOneKey;
exports.hasLogin = hasLogin;
exports.isHost = isHost;
exports.isObject = void 0;
exports.isObjectOrArray = isObjectOrArray;
exports.isRelatedToDeprecation = isRelatedToDeprecation;
exports.isVersionValid = isVersionValid;
exports.mask = mask;
exports.normalizeDistTags = normalizeDistTags;
exports.pad = pad;
exports.parseAddress = parseAddress;
Object.defineProperty(exports, "parseConfigFile", {
  enumerable: true,
  get: function () {
    return _config.parseConfigFile;
  }
});
exports.parseInterval = parseInterval;
exports.parseReadme = parseReadme;
exports.semverSort = semverSort;
exports.sortByName = sortByName;
exports.tagVersion = tagVersion;
exports.validateURL = validateURL;
var _fs = _interopRequireDefault(require("fs"));
var _lodash = _interopRequireDefault(require("lodash"));
var _semver = _interopRequireDefault(require("semver"));
var _url = require("url");
var _validator = _interopRequireDefault(require("validator"));
var _config = require("@verdaccio/config");
var _core = require("@verdaccio/core");
var _utils = require("@verdaccio/utils");
var _constants = require("./constants");
var _logger = require("./logger");
function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }
function ownKeys(object, enumerableOnly) { var keys = Object.keys(object); if (Object.getOwnPropertySymbols) { var symbols = Object.getOwnPropertySymbols(object); enumerableOnly && (symbols = symbols.filter(function (sym) { return Object.getOwnPropertyDescriptor(object, sym).enumerable; })), keys.push.apply(keys, symbols); } return keys; }
function _objectSpread(target) { for (var i = 1; i < arguments.length; i++) { var source = null != arguments[i] ? arguments[i] : {}; i % 2 ? ownKeys(Object(source), !0).forEach(function (key) { _defineProperty(target, key, source[key]); }) : Object.getOwnPropertyDescriptors ? Object.defineProperties(target, Object.getOwnPropertyDescriptors(source)) : ownKeys(Object(source)).forEach(function (key) { Object.defineProperty(target, key, Object.getOwnPropertyDescriptor(source, key)); }); } return target; }
function _defineProperty(obj, key, value) { key = _toPropertyKey(key); if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }
function _toPropertyKey(arg) { var key = _toPrimitive(arg, "string"); return typeof key === "symbol" ? key : String(key); }
function _toPrimitive(input, hint) { if (typeof input !== "object" || input === null) return input; var prim = input[Symbol.toPrimitive]; if (prim !== undefined) { var res = prim.call(input, hint || "default"); if (typeof res !== "object") return res; throw new TypeError("@@toPrimitive must return a primitive value."); } return (hint === "string" ? String : Number)(input); }
const {
  getBadData,
  getBadRequest,
  getCode,
  getConflict,
  getForbidden,
  getInternalError,
  getNotFound,
  getServiceUnavailable,
  getUnauthorized
} = _core.errorUtils;
const validProtocols = ['https', 'http'];
function convertPayloadToBase64(payload) {
  return Buffer.from(payload, 'base64');
}

/**
 * Check whether an element is an Object
 * @param {*} obj the element
 * @return {Boolean}
 */
const isObject = _core.validatioUtils.isObject;

/**
 * @deprecated not used un v6
 */
exports.isObject = isObject;
function isObjectOrArray(obj) {
  return _lodash.default.isObject(obj) && _lodash.default.isNull(obj) === false;
}
function tagVersion(data, version, tag) {
  if (tag && data[_constants.DIST_TAGS][tag] !== version && _semver.default.parse(version, true)) {
    // valid version - store
    data[_constants.DIST_TAGS][tag] = version;
    return true;
  }
  return false;
}

/**
 * Gets version from a package object taking into account semver weirdness.
 * @return {String} return the semantic version of a package
 */
function getVersion(pkg, version) {
  // this condition must allow cast
  if (_lodash.default.isNil(pkg.versions[version]) === false) {
    return pkg.versions[version];
  }
  try {
    version = _semver.default.parse(version, true);
    for (const versionItem in pkg.versions) {
      if (version.compare(_semver.default.parse(versionItem, true)) === 0) {
        return pkg.versions[versionItem];
      }
    }
  } catch (err) {
    return undefined;
  }
}

/**
 * Parse an internet address
 * Allow:
 - https:localhost:1234        - protocol + host + port
 - localhost:1234              - host + port
 - 1234                        - port
 - http::1234                  - protocol + port
 - https://localhost:443/      - full url + https
 - http://[::1]:443/           - ipv6
 - unix:/tmp/http.sock         - unix sockets
 - https://unix:/tmp/http.sock - unix sockets (https)
 * @param {*} urlAddress the internet address definition
 * @return {Object|Null} literal object that represent the address parsed
 */
function parseAddress(urlAddress) {
  //
  // TODO: refactor it to something more reasonable?
  //
  //        protocol :  //      (  host  )|(    ipv6     ):  port  /
  let urlPattern = /^((https?):(\/\/)?)?((([^\/:]*)|\[([^\[\]]+)\]):)?(\d+)\/?$/.exec(urlAddress);
  if (urlPattern) {
    return {
      proto: urlPattern[2] || _constants.DEFAULT_PROTOCOL,
      host: urlPattern[6] || urlPattern[7] || _constants.DEFAULT_DOMAIN,
      port: urlPattern[8] || _constants.DEFAULT_PORT
    };
  }
  urlPattern = /^((https?):(\/\/)?)?unix:(.*)$/.exec(urlAddress);
  if (urlPattern) {
    return {
      proto: urlPattern[2] || _constants.DEFAULT_PROTOCOL,
      path: urlPattern[4]
    };
  }
  return null;
}

/**
 * Function filters out bad semver versions and sorts the array.
 * @return {Array} sorted Array
 */
function semverSort(listVersions) {
  return listVersions.filter(function (x) {
    if (!_semver.default.parse(x, true)) {
      _logger.logger.warn({
        ver: x
      }, 'ignoring bad version @{ver}');
      return false;
    }
    return true;
  })
  // FIXME: it seems the @types/semver do not handle a legitimate method named 'compareLoose'
  // @ts-ignore
  .sort(_semver.default.compareLoose).map(String);
}

/**
 * Flatten arrays of tags.
 * @param {*} data
 */
function normalizeDistTags(pkg) {
  let sorted;
  if (!pkg[_constants.DIST_TAGS].latest) {
    // overwrite latest with highest known version based on semver sort
    sorted = semverSort(Object.keys(pkg.versions));
    if (sorted && sorted.length) {
      pkg[_constants.DIST_TAGS].latest = sorted.pop();
    }
  }
  for (const tag in pkg[_constants.DIST_TAGS]) {
    if (_lodash.default.isArray(pkg[_constants.DIST_TAGS][tag])) {
      if (pkg[_constants.DIST_TAGS][tag].length) {
        // sort array
        // FIXME: this is clearly wrong, we need to research why this is like this.
        // @ts-ignore
        sorted = semverSort(pkg[_constants.DIST_TAGS][tag]);
        if (sorted.length) {
          // use highest version based on semver sort
          pkg[_constants.DIST_TAGS][tag] = sorted.pop();
        }
      } else {
        delete pkg[_constants.DIST_TAGS][tag];
      }
    } else if (_lodash.default.isString(pkg[_constants.DIST_TAGS][tag])) {
      if (!_semver.default.parse(pkg[_constants.DIST_TAGS][tag], true)) {
        // if the version is invalid, delete the dist-tag entry
        delete pkg[_constants.DIST_TAGS][tag];
      }
    }
  }
}
const parseIntervalTable = {
  '': 1000,
  ms: 1,
  s: 1000,
  m: 60 * 1000,
  h: 60 * 60 * 1000,
  d: 86400000,
  w: 7 * 86400000,
  M: 30 * 86400000,
  y: 365 * 86400000
};

/**
 * Parse an internal string to number
 * @param {*} interval
 * @return {Number}
 */
function parseInterval(interval) {
  if (typeof interval === 'number') {
    return interval * 1000;
  }
  let result = 0;
  let last_suffix = Infinity;
  interval.split(/\s+/).forEach(function (x) {
    if (!x) {
      return;
    }
    const m = x.match(/^((0|[1-9][0-9]*)(\.[0-9]+)?)(ms|s|m|h|d|w|M|y|)$/);
    if (!m || parseIntervalTable[m[4]] >= last_suffix || m[4] === '' && last_suffix !== Infinity) {
      throw Error('invalid interval: ' + interval);
    }
    last_suffix = parseIntervalTable[m[4]];
    result += Number(m[1]) * parseIntervalTable[m[4]];
  });
  return result;
}
const ErrorCode = {
  getConflict,
  getBadData,
  getBadRequest,
  getInternalError,
  getUnauthorized,
  getForbidden,
  getServiceUnavailable,
  getNotFound,
  getCode
};

/**
 * Check whether the path already exist.
 * @param {String} path
 * @return {Boolean}
 */
exports.ErrorCode = ErrorCode;
function folderExists(path) {
  try {
    const stat = _fs.default.statSync(path);
    return stat.isDirectory();
  } catch (_) {
    return false;
  }
}

/**
 * Check whether the file already exist.
 * @param {String} path
 * @return {Boolean}
 */
function fileExists(path) {
  try {
    const stat = _fs.default.statSync(path);
    return stat.isFile();
  } catch (_) {
    return false;
  }
}
function sortByName(packages, orderAscending = true) {
  return packages.slice().sort(function (a, b) {
    const comparatorNames = a.name.toLowerCase() < b.name.toLowerCase();
    return orderAscending ? comparatorNames ? -1 : 1 : comparatorNames ? 1 : -1;
  });
}
function addScope(scope, packageName) {
  return `@${scope}/${packageName}`;
}
function deleteProperties(propertiesToDelete, objectItem) {
  _lodash.default.forEach(propertiesToDelete, property => {
    delete objectItem[property];
  });
  return objectItem;
}
function addGravatarSupport(pkgInfo, online = true) {
  const pkgInfoCopy = _objectSpread({}, pkgInfo);
  const author = _lodash.default.get(pkgInfo, 'latest.author', null);
  const contributors = (0, _utils.normalizeContributors)(_lodash.default.get(pkgInfo, 'latest.contributors', []));
  const maintainers = _lodash.default.get(pkgInfo, 'latest.maintainers', []);

  // for author.
  if (author && _lodash.default.isObject(author)) {
    const {
      email
    } = author;
    pkgInfoCopy.latest.author.avatar = (0, _utils.generateGravatarUrl)(email, online);
  }
  if (author && _lodash.default.isString(author)) {
    pkgInfoCopy.latest.author = {
      avatar: _utils.GENERIC_AVATAR,
      email: '',
      author
    };
  }

  // for contributors
  if (_lodash.default.isEmpty(contributors) === false) {
    pkgInfoCopy.latest.contributors = contributors.map(contributor => {
      if (isObject(contributor)) {
        contributor.avatar = (0, _utils.generateGravatarUrl)(contributor.email, online);
      } else if (_lodash.default.isString(contributor)) {
        contributor = {
          avatar: _utils.GENERIC_AVATAR,
          email: contributor,
          name: contributor
        };
      }
      return contributor;
    });
  }

  // for maintainers
  if (_lodash.default.isEmpty(maintainers) === false) {
    pkgInfoCopy.latest.maintainers = maintainers.map(maintainer => {
      maintainer.avatar = (0, _utils.generateGravatarUrl)(maintainer.email, online);
      return maintainer;
    });
  }
  return pkgInfoCopy;
}

/**
 * parse package readme - markdown/ascii
 * @param {String} packageName name of package
 * @param {String} readme package readme

 * @return {String} converted html template
 */
// TODO: rename, does not parse anymore
function parseReadme(packageName, readme) {
  if (_lodash.default.isEmpty(readme) === false) {
    return readme;
  }

  // logs readme not found error
  _logger.logger.info({
    packageName
  }, '@{packageName}: No readme found');
  return 'ERROR: No README data found!';
}
/**
 * Formats author field for webui.
 * @see https://docs.npmjs.com/files/package.json#author
 * @param {string|object|undefined} author
 */
function formatAuthor(author) {
  let authorDetails = {
    name: _constants.DEFAULT_USER,
    email: '',
    url: ''
  };
  if (_lodash.default.isNil(author)) {
    return authorDetails;
  }
  if (_lodash.default.isString(author)) {
    authorDetails = _objectSpread(_objectSpread({}, authorDetails), {}, {
      name: author
    });
  }
  if (_lodash.default.isObject(author)) {
    authorDetails = _objectSpread(_objectSpread({}, authorDetails), author);
  }
  return authorDetails;
}

/**
 * Apply whitespaces based on the length
 * @param {*} str the log message
 * @return {String}
 */
function pad(str, max) {
  if (str.length < max) {
    return str + ' '.repeat(max - str.length);
  }
  return str;
}

/**
 * return a masquerade string with its first and last {charNum} and three dots in between.
 * @param {String} str
 * @param {Number} charNum
 * @returns {String}
 */
function mask(str, charNum = 3) {
  return `${str.substr(0, charNum)}...${str.substr(-charNum)}`;
}
function encodeScopedUri(packageName) {
  return packageName.replace(/\//g, '%2f');
}
function hasDiffOneKey(versions) {
  return Object.keys(versions).length !== 1;
}
function isVersionValid(packageMeta, packageVersion) {
  const hasVersion = typeof packageVersion !== 'undefined';
  if (!hasVersion) {
    return false;
  }
  const hasMatchVersion = Object.keys(packageMeta.versions).includes(packageVersion);
  return hasMatchVersion;
}
function isRelatedToDeprecation(pkgInfo) {
  const {
    versions
  } = pkgInfo;
  for (const version in versions) {
    if (Object.prototype.hasOwnProperty.call(versions[version], 'deprecated')) {
      return true;
    }
  }
  return false;
}
function validateURL(publicUrl) {
  try {
    const parsed = new _url.URL(publicUrl);
    if (!validProtocols.includes(parsed.protocol.replace(':', ''))) {
      throw Error('invalid protocol');
    }
    return true;
  } catch (err) {
    // TODO: add error logger here
    return false;
  }
}
function isHost(url = '', options = {}) {
  return _validator.default.isURL(url, _objectSpread({
    require_host: true,
    allow_trailing_dot: false,
    require_valid_protocol: false,
    // @ts-ignore
    require_port: false,
    require_tld: false
  }, options));
}
function hasLogin(config) {
  var _config$web, _config$web2;
  // FIXME: types are not yet on the library verdaccio/monorepo
  // @ts-ignore
  return _lodash.default.isNil(config === null || config === void 0 ? void 0 : (_config$web = config.web) === null || _config$web === void 0 ? void 0 : _config$web.login) || (config === null || config === void 0 ? void 0 : (_config$web2 = config.web) === null || _config$web2 === void 0 ? void 0 : _config$web2.login) === true;
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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