"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;
var _JSONStream = _interopRequireDefault(require("JSONStream"));
var _debug = _interopRequireDefault(require("debug"));
var _lodash = _interopRequireDefault(require("lodash"));
var _request = _interopRequireDefault(require("request"));
var _stream = _interopRequireDefault(require("stream"));
var _url = _interopRequireDefault(require("url"));
var _zlib = _interopRequireDefault(require("zlib"));
var _streams = require("@verdaccio/streams");
var _utils = require("@verdaccio/utils");
var _constants = require("./constants");
var _logger = require("./logger");
var _utils2 = require("./utils");
function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }
function _defineProperty(obj, key, value) { key = _toPropertyKey(key); if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }
function _toPropertyKey(arg) { var key = _toPrimitive(arg, "string"); return typeof key === "symbol" ? key : String(key); }
function _toPrimitive(input, hint) { if (typeof input !== "object" || input === null) return input; var prim = input[Symbol.toPrimitive]; if (prim !== undefined) { var res = prim.call(input, hint || "default"); if (typeof res !== "object") return res; throw new TypeError("@@toPrimitive must return a primitive value."); } return (hint === "string" ? String : Number)(input); }
const debug = (0, _debug.default)('verdaccio:proxy');
const encode = function (thing) {
  return encodeURIComponent(thing).replace(/^%40/, '@');
};
const jsonContentType = _constants.HEADERS.JSON;
const contentTypeAccept = `${jsonContentType};`;

/**
 * Just a helper (`config[key] || default` doesn't work because of zeroes)
 */
const setConfig = (config, key, def) => {
  return _lodash.default.isNil(config[key]) === false ? config[key] : def;
};

/**
 * Implements Storage interface
 * (same for storage.js, local-storage.js, up-storage.js)
 */
class ProxyStorage {
  // FIXME: upname is assigned to each instance
  // @ts-ignore

  // FIXME: proxy can be boolean or object, something smells here
  // @ts-ignore

  // @ts-ignore

  /**
   * Constructor
   * @param {*} config
   * @param {*} mainConfig
   */
  constructor(config, mainConfig) {
    var _mainConfig$user_agen;
    _defineProperty(this, "config", void 0);
    _defineProperty(this, "failed_requests", void 0);
    _defineProperty(this, "userAgent", void 0);
    _defineProperty(this, "ca", void 0);
    _defineProperty(this, "logger", void 0);
    _defineProperty(this, "server_id", void 0);
    _defineProperty(this, "url", void 0);
    _defineProperty(this, "maxage", void 0);
    _defineProperty(this, "timeout", void 0);
    _defineProperty(this, "max_fails", void 0);
    _defineProperty(this, "fail_timeout", void 0);
    _defineProperty(this, "agent_options", void 0);
    _defineProperty(this, "upname", void 0);
    _defineProperty(this, "proxy", void 0);
    _defineProperty(this, "last_request_time", void 0);
    _defineProperty(this, "strict_ssl", void 0);
    this.config = config;
    this.failed_requests = 0;
    // @ts-ignore
    this.userAgent = (_mainConfig$user_agen = mainConfig.user_agent) !== null && _mainConfig$user_agen !== void 0 ? _mainConfig$user_agen : 'hidden';
    this.ca = config.ca;
    this.logger = _logger.logger;
    this.server_id = mainConfig.server_id;
    this.url = _url.default.parse(this.config.url);
    this._setupProxy(this.url.hostname, config, mainConfig, this.url.protocol === 'https:');
    this.config.url = this.config.url.replace(/\/$/, '');
    if (this.config.timeout && Number(this.config.timeout) >= 1000) {
      this.logger.warn(['Too big timeout value: ' + this.config.timeout, 'We changed time format to nginx-like one', '(see http://nginx.org/en/docs/syntax.html)', 'so please update your config accordingly'].join('\n'));
    }

    // a bunch of different configurable timers
    this.maxage = (0, _utils2.parseInterval)(setConfig(this.config, 'maxage', '2m'));
    this.timeout = (0, _utils2.parseInterval)(setConfig(this.config, 'timeout', '30s'));
    this.max_fails = Number(setConfig(this.config, 'max_fails', 2));
    this.fail_timeout = (0, _utils2.parseInterval)(setConfig(this.config, 'fail_timeout', '5m'));
    this.strict_ssl = Boolean(setConfig(this.config, 'strict_ssl', true));
    this.agent_options = setConfig(this.config, 'agent_options', {
      keepAlive: true,
      maxSockets: 40,
      maxFreeSockets: 10
    });
  }

  /**
   * Fetch an asset.
   * @param {*} options
   * @param {*} cb
   * @return {Request}
   */
  request(options, cb) {
    let json;
    if (this._statusCheck() === false) {
      const streamRead = new _stream.default.Readable();
      process.nextTick(function () {
        if (cb) {
          cb(_utils2.ErrorCode.getInternalError(_constants.API_ERROR.UPLINK_OFFLINE));
        }
        streamRead.emit('error', _utils2.ErrorCode.getInternalError(_constants.API_ERROR.UPLINK_OFFLINE));
      });
      streamRead._read = function () {};
      // preventing 'Uncaught, unspecified "error" event'
      streamRead.on('error', function () {});
      return streamRead;
    }
    const self = this;
    const headers = this._setHeaders(options);
    this._addProxyHeaders(options.req, headers);
    this._overrideWithUpLinkConfLocaligHeaders(headers);
    const method = options.method || 'GET';
    const uri = options.uri_full || this.config.url + options.uri;
    self.logger.info({
      method: method,
      uri: uri
    }, "making request: '@{method} @{uri}'");
    if ((0, _utils2.isObject)(options.json)) {
      json = JSON.stringify(options.json);
      headers['Content-Type'] = headers['Content-Type'] || _constants.HEADERS.JSON;
    }
    const requestCallback = cb ? function (err, res, body) {
      let error;
      const responseLength = err ? 0 : body.length;
      processBody();
      logActivity();
      cb(err, res, body);

      /**
       * Perform a decode.
       */
      function processBody() {
        if (err) {
          error = err.message;
          return;
        }
        if (options.json && res.statusCode < 300) {
          try {
            body = JSON.parse(body.toString(_constants.CHARACTER_ENCODING.UTF8));
          } catch (_err) {
            body = {};
            err = _err;
            error = err.message;
          }
        }
        if (!err && (0, _utils2.isObject)(body)) {
          if (_lodash.default.isString(body.error)) {
            error = body.error;
          }
        }
      }
      /**
       * Perform a log.
       */
      function logActivity() {
        let message = "@{!status}, req: '@{request.method} @{request.url}'";
        message += error ? ', error: @{!error}' : ', bytes: @{bytes.in}/@{bytes.out}';
        self.logger.http({
          err: err || undefined,
          // if error is null/false change this to undefined so it wont log
          request: {
            method: method,
            url: uri
          },
          status: res != null ? res.statusCode : 'ERR',
          error: error,
          bytes: {
            in: json ? json.length : 0,
            out: responseLength || 0
          }
        }, message);
      }
    } : undefined;
    let requestOptions = {
      url: uri,
      method: method,
      headers: headers,
      body: json,
      proxy: this.proxy,
      encoding: null,
      gzip: true,
      timeout: this.timeout,
      strictSSL: this.strict_ssl,
      agentOptions: this.agent_options
    };
    if (this.ca) {
      requestOptions = Object.assign({}, requestOptions, {
        ca: this.ca
      });
    }
    const req = (0, _request.default)(requestOptions, requestCallback);
    let statusCalled = false;
    req.on('response', function (res) {
      // FIXME: _verdaccio_aborted seems not used
      // @ts-ignore
      if (!req._verdaccio_aborted && !statusCalled) {
        statusCalled = true;
        self._statusCheck(true);
      }
      if (_lodash.default.isNil(requestCallback) === false) {
        (function do_log() {
          const message = "@{!status}, req: '@{request.method} @{request.url}' (streaming)";
          self.logger.http({
            request: {
              method: method,
              url: uri
            },
            status: _lodash.default.isNull(res) === false ? res.statusCode : 'ERR'
          }, message);
        })();
      }
    });
    req.on('error', function (_err) {
      // FIXME: _verdaccio_aborted seems not used
      // @ts-ignore
      if (!req._verdaccio_aborted && !statusCalled) {
        statusCalled = true;
        self._statusCheck(false);
      }
    });
    // @ts-ignore
    return req;
  }

  /**
   * Set default headers.
   * @param {Object} options
   * @return {Object}
   * @private
   */
  _setHeaders(options) {
    var _options$req;
    const headers = options.headers || {};
    const accept = _constants.HEADERS.ACCEPT;
    const acceptEncoding = _constants.HEADERS.ACCEPT_ENCODING;
    const userAgent = _constants.HEADERS.USER_AGENT;
    headers[accept] = headers[accept] || contentTypeAccept;
    headers[acceptEncoding] = headers[acceptEncoding] || 'gzip';
    // registry.npmjs.org will only return search result if user-agent include string 'npm'
    headers[userAgent] = this.userAgent ? `npm (${this.userAgent})` : (_options$req = options.req) === null || _options$req === void 0 ? void 0 : _options$req.get('user-agent');
    return this._setAuth(headers);
  }

  /**
   * Validate configuration auth and assign Header authorization
   * @param {Object} headers
   * @return {Object}
   * @private
   */
  _setAuth(headers) {
    const {
      auth
    } = this.config;
    if (_lodash.default.isNil(auth) || headers[_constants.HEADERS.AUTHORIZATION]) {
      return headers;
    }
    if (_lodash.default.isObject(auth) === false && _lodash.default.isObject(auth.token) === false) {
      this._throwErrorAuth('Auth invalid');
    }

    // get NPM_TOKEN http://blog.npmjs.org/post/118393368555/deploying-with-npm-private-modules
    // or get other variable export in env
    // https://github.com/verdaccio/verdaccio/releases/tag/v2.5.0
    let token;
    const tokenConf = auth;
    if (_lodash.default.isNil(tokenConf.token) === false && _lodash.default.isString(tokenConf.token)) {
      token = tokenConf.token;
    } else if (_lodash.default.isNil(tokenConf.token_env) === false) {
      if (_lodash.default.isString(tokenConf.token_env)) {
        token = process.env[tokenConf.token_env];
      } else if (_lodash.default.isBoolean(tokenConf.token_env) && tokenConf.token_env) {
        token = process.env.NPM_TOKEN;
      } else {
        this.logger.error(_constants.ERROR_CODE.token_required);
        this._throwErrorAuth(_constants.ERROR_CODE.token_required);
      }
    } else {
      token = process.env.NPM_TOKEN;
    }
    if (_lodash.default.isNil(token)) {
      this._throwErrorAuth(_constants.ERROR_CODE.token_required);
    }

    // define type Auth allow basic and bearer
    const type = tokenConf.type || _constants.TOKEN_BASIC;
    this._setHeaderAuthorization(headers, type, token);
    return headers;
  }

  /**
   * @param {string} message
   * @throws {Error}
   * @private
   */
  _throwErrorAuth(message) {
    this.logger.error(message);
    throw new Error(message);
  }

  /**
   * Assign Header authorization with type authentication
   * @param {Object} headers
   * @param {string} type
   * @param {string} token
   * @private
   */
  _setHeaderAuthorization(headers, type, token) {
    const _type = type.toLowerCase();
    if (_type !== _constants.TOKEN_BEARER.toLowerCase() && _type !== _constants.TOKEN_BASIC.toLowerCase()) {
      this._throwErrorAuth(`Auth type '${_type}' not allowed`);
    }
    type = _lodash.default.upperFirst(type);
    headers[_constants.HEADERS.AUTHORIZATION] = (0, _utils.buildToken)(type, token);
  }

  /**
   * It will add or override specified headers from config file.
   *
   * Eg:
   *
   * uplinks:
   npmjs:
   url: https://registry.npmjs.org/
   headers:
   Accept: "application/vnd.npm.install-v2+json; q=1.0"
   verdaccio-staging:
   url: https://mycompany.com/npm
   headers:
   Accept: "application/json"
   authorization: "Basic YourBase64EncodedCredentials=="
    * @param {Object} headers
   * @private
   */
  _overrideWithUpLinkConfLocaligHeaders(headers) {
    if (!this.config.headers) {
      return headers;
    }

    // add/override headers specified in the config
    /* eslint guard-for-in: 0 */
    for (const key in this.config.headers) {
      headers[key] = this.config.headers[key];
    }
  }

  /**
   * Determine whether can fetch from the provided URL
   * @param {*} url
   * @return {Boolean}
   */
  isUplinkValid(url) {
    const urlParsed = _url.default.parse(url);
    const isHTTPS = urlDomainParsed => urlDomainParsed.protocol === 'https:' && (urlParsed.port === null || urlParsed.port === '443');
    const getHost = urlDomainParsed => isHTTPS(urlDomainParsed) ? urlDomainParsed.hostname : urlDomainParsed.host;
    const isMatchProtocol = urlParsed.protocol === this.url.protocol;
    const isMatchHost = getHost(urlParsed) === getHost(this.url);
    // @ts-ignore
    const isMatchPath = urlParsed.path.indexOf(this.url.path) === 0;
    return isMatchProtocol && isMatchHost && isMatchPath;
  }

  /**
   * Get a remote package metadata
   * @param {*} name package name
   * @param {*} options request options, eg: eTag.
   * @param {*} callback
   */
  getRemoteMetadata(name, options, callback) {
    const headers = {};
    if (_lodash.default.isNil(options.etag) === false) {
      headers['If-None-Match'] = options.etag;
      headers[_constants.HEADERS.ACCEPT] = contentTypeAccept;
    }
    this.request({
      uri: `/${encode(name)}`,
      json: true,
      headers: headers,
      req: options.req
    }, (err, res, body) => {
      if (err) {
        return callback(err);
      }
      if (res.statusCode === _constants.HTTP_STATUS.NOT_FOUND) {
        return callback(_utils2.ErrorCode.getNotFound(_constants.API_ERROR.NOT_PACKAGE_UPLINK));
      }
      if (!(res.statusCode >= _constants.HTTP_STATUS.OK && res.statusCode < _constants.HTTP_STATUS.MULTIPLE_CHOICES)) {
        const error = _utils2.ErrorCode.getInternalError(`${_constants.API_ERROR.BAD_STATUS_CODE}: ${res.statusCode}`);
        error.remoteStatus = res.statusCode;
        return callback(error);
      }
      callback(null, body, res.headers.etag);
    });
  }

  /**
   * Fetch a tarball from the uplink.
   * @param {String} url
   * @return {Stream}
   */
  fetchTarball(url) {
    const stream = new _streams.ReadTarball({});
    let current_length = 0;
    let expected_length;
    stream.abort = () => {};
    const readStream = this.request({
      uri_full: url,
      encoding: null,
      headers: {
        Accept: contentTypeAccept
      }
    });
    readStream.on('response', function (res) {
      if (res.statusCode === _constants.HTTP_STATUS.NOT_FOUND) {
        return stream.emit('error', _utils2.ErrorCode.getNotFound(_constants.API_ERROR.NOT_FILE_UPLINK));
      }
      if (!(res.statusCode >= _constants.HTTP_STATUS.OK && res.statusCode < _constants.HTTP_STATUS.MULTIPLE_CHOICES)) {
        return stream.emit('error', _utils2.ErrorCode.getInternalError(`bad uplink status code: ${res.statusCode}`));
      }
      if (res.headers[_constants.HEADER_TYPE.CONTENT_LENGTH]) {
        expected_length = res.headers[_constants.HEADER_TYPE.CONTENT_LENGTH];
        stream.emit(_constants.HEADER_TYPE.CONTENT_LENGTH, res.headers[_constants.HEADER_TYPE.CONTENT_LENGTH]);
      }
      readStream.pipe(stream);
    });
    readStream.on('error', function (err) {
      stream.emit('error', err);
    });
    readStream.on('data', function (data) {
      current_length += data.length;
    });
    readStream.on('end', function (data) {
      if (data) {
        current_length += data.length;
      }
      if (expected_length && current_length != expected_length) {
        stream.emit('error', _utils2.ErrorCode.getInternalError(_constants.API_ERROR.CONTENT_MISMATCH));
      }
    });
    return stream;
  }

  /**
   * Perform a stream search.
   * @param {*} options request options
   * @return {Stream}
   */
  search(options) {
    const transformStream = new _stream.default.PassThrough({
      objectMode: true
    });
    const requestStream = this.request({
      uri: options.req.url,
      req: options.req,
      headers: {
        // query for search
        referer: options.req.get('referer')
      }
    });
    const parsePackage = pkg => {
      if ((0, _utils2.isObjectOrArray)(pkg)) {
        transformStream.emit('data', pkg);
      }
    };
    requestStream.on('response', res => {
      if (!String(res.statusCode).match(/^2\d\d$/)) {
        return transformStream.emit('error', _utils2.ErrorCode.getInternalError(`bad status code ${res.statusCode} from uplink`));
      }

      // See https://github.com/request/request#requestoptions-callback
      // Request library will not decode gzip stream.
      let jsonStream;
      if (res.headers[_constants.HEADER_TYPE.CONTENT_ENCODING] === _constants.HEADERS.GZIP) {
        jsonStream = res.pipe(_zlib.default.createUnzip());
      } else {
        jsonStream = res;
      }
      jsonStream.pipe(_JSONStream.default.parse('*')).on('data', parsePackage);
      jsonStream.on('end', () => {
        transformStream.emit('end');
      });
    });
    requestStream.on('error', err => {
      transformStream.emit('error', err);
    });
    transformStream.abort = () => {
      // FIXME: this is clearly a potential issue
      // there is no abort method on Stream.Readable
      // @ts-ignore
      requestStream.abort();
      transformStream.emit('end');
    };
    return transformStream;
  }

  /**
   * Add proxy headers.
   * FIXME: object mutations, it should return an new object
   * @param {*} req the http request
   * @param {*} headers the request headers
   */
  _addProxyHeaders(req, headers) {
    if (req) {
      // Only submit X-Forwarded-For field if we don't have a proxy selected
      // in the config file.
      //
      // Otherwise misconfigured proxy could return 407:
      // https://github.com/rlidwka/sinopia/issues/254
      //
      // FIXME: proxy logic is odd, something is wrong here.
      // @ts-ignore
      if (!this.proxy) {
        headers['x-forwarded-for'] = (req.get('x-forwarded-for') ? req.get('x-forwarded-for') + ', ' : '') + req.connection.remoteAddress;
      }
    }

    // always attach Via header to avoid loops, even if we're not proxying
    headers['via'] = req && req.get('via') ? req.get('via') + ', ' : '';
    headers['via'] += '1.1 ' + this.server_id + ' (Verdaccio)';
  }

  /**
   * Check whether the remote host is available.
   * @param {*} alive
   * @return {Boolean}
   */
  _statusCheck(alive) {
    if (arguments.length === 0) {
      return this._ifRequestFailure() === false;
    }
    if (alive) {
      if (this.failed_requests >= this.max_fails) {
        this.logger.warn({
          host: this.url.host
        }, 'host @{host} is back online');
      }
      this.failed_requests = 0;
    } else {
      this.failed_requests++;
      if (this.failed_requests === this.max_fails) {
        this.logger.warn({
          host: this.url.host
        }, 'host @{host} is now offline');
      }
    }
    this.last_request_time = Date.now();
  }

  /**
   * If the request failure.
   * @return {boolean}
   * @private
   */
  _ifRequestFailure() {
    return this.failed_requests >= this.max_fails && Math.abs(Date.now() - this.last_request_time) < this.fail_timeout;
  }

  /**
   * Set up a proxy.
   * @param {*} hostname
   * @param {*} config
   * @param {*} mainconfig
   * @param {*} isHTTPS
   */
  _setupProxy(hostname, config, mainconfig, isHTTPS) {
    let noProxyList;
    const proxy_key = isHTTPS ? 'https_proxy' : 'http_proxy';

    // get http_proxy and no_proxy configs
    if (proxy_key in config) {
      this.proxy = config[proxy_key];
    } else if (proxy_key in mainconfig) {
      this.proxy = mainconfig[proxy_key];
    }
    if ('no_proxy' in config) {
      noProxyList = config.no_proxy;
    } else if ('no_proxy' in mainconfig) {
      noProxyList = mainconfig.no_proxy;
    }

    // use wget-like algorithm to determine if proxy shouldn't be used
    if (hostname[0] !== '.') {
      hostname = '.' + hostname;
    }
    if (_lodash.default.isString(noProxyList) && noProxyList.length) {
      noProxyList = noProxyList.split(',');
    }
    if (_lodash.default.isArray(noProxyList)) {
      for (let i = 0; i < noProxyList.length; i++) {
        let noProxyItem = noProxyList[i];
        if (noProxyItem[0] !== '.') {
          noProxyItem = '.' + noProxyItem;
        }
        if (hostname.lastIndexOf(noProxyItem) === hostname.length - noProxyItem.length) {
          if (this.proxy) {
            debug('not using proxy for %o, excluded by %o rule', this.url.href, noProxyItem);
            // @ts-ignore
            this.proxy = false;
          }
          break;
        }
      }
    }

    // if it's non-string (i.e. "false"), don't use it
    if (_lodash.default.isString(this.proxy) === false) {
      // @ts-ignore
      delete this.proxy;
    } else {
      debug('using proxy %o for %o', this.url.href, this.proxy);
    }
  }
}
var _default = ProxyStorage;
exports.default = _default;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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