"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;
var _assert = _interopRequireDefault(require("assert"));
var _async = _interopRequireDefault(require("async"));
var _lodash = _interopRequireDefault(require("lodash"));
var _stream = _interopRequireDefault(require("stream"));
var _core = require("@verdaccio/core");
var _streams = require("@verdaccio/streams");
var _logger = require("../lib/logger");
var _configUtils = require("./config-utils");
var _constants = require("./constants");
var _localStorage = _interopRequireDefault(require("./local-storage"));
var _metadataUtils = require("./metadata-utils");
var _search = _interopRequireDefault(require("./search"));
var _storageUtils = require("./storage-utils");
var _upStorage = _interopRequireDefault(require("./up-storage"));
var _uplinkUtil = require("./uplink-util");
var _utils = require("./utils");
function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }
function _defineProperty(obj, key, value) { key = _toPropertyKey(key); if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }
function _toPropertyKey(arg) { var key = _toPrimitive(arg, "string"); return typeof key === "symbol" ? key : String(key); }
function _toPrimitive(input, hint) { if (typeof input !== "object" || input === null) return input; var prim = input[Symbol.toPrimitive]; if (prim !== undefined) { var res = prim.call(input, hint || "default"); if (typeof res !== "object") return res; throw new TypeError("@@toPrimitive must return a primitive value."); } return (hint === "string" ? String : Number)(input); }
class Storage {
  constructor(config) {
    _defineProperty(this, "localStorage", void 0);
    _defineProperty(this, "config", void 0);
    _defineProperty(this, "logger", void 0);
    _defineProperty(this, "uplinks", void 0);
    _defineProperty(this, "filters", void 0);
    this.config = config;
    this.uplinks = (0, _uplinkUtil.setupUpLinks)(config);
    this.logger = _logger.logger;
    this.filters = [];
    // @ts-ignore
    this.localStorage = null;
  }
  init(config, filters = []) {
    this.filters = filters;
    this.localStorage = new _localStorage.default(this.config, _logger.logger);
    return this.localStorage.getSecret(config);
  }

  /**
   *  Add a {name} package to a system
   Function checks if package with the same name is available from uplinks.
   If it isn't, we create package locally
   Used storages: local (write) && uplinks
   */
  async addPackage(name, metadata, callback) {
    try {
      await (0, _storageUtils.checkPackageLocal)(name, this.localStorage);
      await (0, _storageUtils.checkPackageRemote)(name, this._isAllowPublishOffline(), this._syncUplinksMetadata.bind(this));
      await (0, _storageUtils.publishPackage)(name, metadata, this.localStorage);
      callback();
    } catch (err) {
      callback(err);
    }
  }
  _isAllowPublishOffline() {
    return typeof this.config.publish !== 'undefined' && _lodash.default.isBoolean(this.config.publish.allow_offline) && this.config.publish.allow_offline;
  }
  readTokens(filter) {
    return this.localStorage.readTokens(filter);
  }
  saveToken(token) {
    return this.localStorage.saveToken(token);
  }
  deleteToken(user, tokenKey) {
    return this.localStorage.deleteToken(user, tokenKey);
  }

  /**
   * Add a new version of package {name} to a system
   Used storages: local (write)
   */
  addVersion(name, version, metadata, tag, callback) {
    this.localStorage.addVersion(name, version, metadata, tag, callback);
  }

  /**
   * Tags a package version with a provided tag
   Used storages: local (write)
   */
  mergeTags(name, tagHash, callback) {
    this.localStorage.mergeTags(name, tagHash, callback);
  }

  /**
   * Change an existing package (i.e. unpublish one version)
   Function changes a package info from local storage and all uplinks with write access./
   Used storages: local (write)
   */
  changePackage(name, metadata, revision, callback) {
    this.localStorage.changePackage(name, metadata, revision, callback);
  }

  /**
   * Remove a package from a system
   Function removes a package from local storage
   Used storages: local (write)
   */
  removePackage(name, callback) {
    this.localStorage.removePackage(name, callback);
    // update the indexer
    _search.default.remove(name);
  }

  /**
   Remove a tarball from a system
   Function removes a tarball from local storage.
   Tarball in question should not be linked to in any existing
   versions, i.e. package version should be unpublished first.
   Used storage: local (write)
   */
  removeTarball(name, filename, revision, callback) {
    this.localStorage.removeTarball(name, filename, revision, callback);
  }

  /**
   * Upload a tarball for {name} package
   Function is synchronous and returns a WritableStream
   Used storages: local (write)
   */
  addTarball(name, filename) {
    return this.localStorage.addTarball(name, filename);
  }
  hasLocalTarball(name, filename) {
    const self = this;
    return new Promise((resolve, reject) => {
      let localStream = self.localStorage.getTarball(name, filename);
      let isOpen = false;
      localStream.on('error', err => {
        if (isOpen || err.status !== _constants.HTTP_STATUS.NOT_FOUND) {
          reject(err);
        }
        // local reported 404 or request was aborted already
        if (localStream) {
          localStream.abort();
          localStream = null;
        }
        resolve(false);
      });
      localStream.on('open', function () {
        isOpen = true;
        localStream.abort();
        localStream = null;
        resolve(true);
      });
    });
  }

  /**
   Get a tarball from a storage for {name} package
   Function is synchronous and returns a ReadableStream
   Function tries to read tarball locally, if it fails then it reads package
   information in order to figure out where we can get this tarball from
   Used storages: local || uplink (just one)
   */
  getTarball(name, filename) {
    const readStream = new _streams.ReadTarball({});
    readStream.abort = function () {};
    const self = this;

    // if someone requesting tarball, it means that we should already have some
    // information about it, so fetching package info is unnecessary

    // trying local first
    // flow: should be IReadTarball
    let localStream = self.localStorage.getTarball(name, filename);
    let isOpen = false;
    localStream.on('error', err => {
      if (isOpen || err.status !== _constants.HTTP_STATUS.NOT_FOUND) {
        return readStream.emit('error', err);
      }

      // local reported 404
      const err404 = err;
      localStream.abort();
      localStream = null; // we force for garbage collector
      self.localStorage.getPackageMetadata(name, (err, info) => {
        if (_lodash.default.isNil(err) && info._distfiles && _lodash.default.isNil(info._distfiles[filename]) === false) {
          // information about this file exists locally
          serveFile(info._distfiles[filename]);
        } else {
          // we know nothing about this file, trying to get information elsewhere
          self._syncUplinksMetadata(name, info, {}, (err, info) => {
            if (_lodash.default.isNil(err) === false) {
              return readStream.emit('error', err);
            }
            if (_lodash.default.isNil(info._distfiles) || _lodash.default.isNil(info._distfiles[filename])) {
              return readStream.emit('error', err404);
            }
            serveFile(info._distfiles[filename]);
          });
        }
      });
    });
    localStream.on('content-length', function (v) {
      readStream.emit('content-length', v);
    });
    localStream.on('open', function () {
      isOpen = true;
      localStream.pipe(readStream);
    });
    return readStream;

    /**
     * Fetch and cache local/remote packages.
     * @param {Object} file define the package shape
     */
    function serveFile(file) {
      let uplink = null;
      for (const uplinkId in self.uplinks) {
        if ((0, _configUtils.hasProxyTo)(name, uplinkId, self.config.packages)) {
          uplink = self.uplinks[uplinkId];
        }
      }
      if (uplink == null) {
        uplink = new _upStorage.default({
          url: file.url,
          cache: true,
          _autogenerated: true
        }, self.config);
      }
      let savestream = null;
      if (uplink.config.cache) {
        savestream = self.localStorage.addTarball(name, filename);
      }
      let on_open = function () {
        // prevent it from being called twice
        on_open = function () {};
        const rstream2 = uplink.fetchTarball(file.url);
        rstream2.on('error', function (err) {
          if (savestream) {
            savestream.abort();
          }
          savestream = null;
          readStream.emit('error', err);
        });
        rstream2.on('end', function () {
          if (savestream) {
            savestream.done();
          }
        });
        rstream2.on('content-length', function (v) {
          readStream.emit('content-length', v);
          if (savestream) {
            savestream.emit('content-length', v);
          }
        });
        rstream2.pipe(readStream);
        if (savestream) {
          rstream2.pipe(savestream);
        }
      };
      if (savestream) {
        savestream.on('open', function () {
          on_open();
        });
        savestream.on('error', function (err) {
          self.logger.warn({
            err: err,
            fileName: file
          }, 'error saving file @{fileName}: @{err.message}\n@{err.stack}');
          if (savestream) {
            savestream.abort();
          }
          savestream = null;
          on_open();
        });
      } else {
        on_open();
      }
    }
  }

  /**
   Retrieve a package metadata for {name} package
   Function invokes localStorage.getPackage and uplink.get_package for every
   uplink with proxy_access rights against {name} and combines results
   into one json object
   Used storages: local && uplink (proxy_access)
    * @param {object} options
   * @property {string} options.name Package Name
   * @property {object}  options.req Express `req` object
   * @property {boolean} options.keepUpLinkData keep up link info in package meta, last update, etc.
   * @property {function} options.callback Callback for receive data
   */
  getPackage(options) {
    this.localStorage.getPackageMetadata(options.name, (err, data) => {
      if (err && (!err.status || err.status >= _constants.HTTP_STATUS.INTERNAL_ERROR)) {
        // report internal errors right away
        return options.callback(err);
      }
      this._syncUplinksMetadata(options.name, data, {
        req: options.req,
        uplinksLook: options.uplinksLook
      }, function getPackageSynUpLinksCallback(err, result, uplinkErrors) {
        if (err) {
          return options.callback(err);
        }
        (0, _utils.normalizeDistTags)((0, _storageUtils.cleanUpLinksRef)(options.keepUpLinkData, result));

        // npm can throw if this field doesn't exist
        result._attachments = {};
        if (options.abbreviated === true) {
          options.callback(null, (0, _storageUtils.convertAbbreviatedManifest)(result), uplinkErrors);
        } else {
          options.callback(null, result, uplinkErrors);
        }
      });
    });
  }

  /**
   Retrieve remote and local packages more recent than {startkey}
   Function streams all packages from all uplinks first, and then
   local packages.
   Note that local packages could override registry ones just because
   they appear in JSON last. That's a trade-off we make to avoid
   memory issues.
   Used storages: local && uplink (proxy_access)
   * @param {*} startkey
   * @param {*} options
   * @return {Stream}
   */
  search(startkey, options) {
    const self = this;
    const searchStream = new _stream.default.PassThrough({
      objectMode: true
    });
    _async.default.eachSeries(Object.keys(this.uplinks), function (up_name, cb) {
      var _options$req, _options$req$query;
      // shortcut: if `local=1` is supplied, don't call uplinks
      if (((_options$req = options.req) === null || _options$req === void 0 ? void 0 : (_options$req$query = _options$req.query) === null || _options$req$query === void 0 ? void 0 : _options$req$query.local) !== undefined) {
        return cb();
      }
      _logger.logger.info(`search for uplink ${up_name}`);
      // search by keyword for each uplink
      const uplinkStream = self.uplinks[up_name].search(options);
      // join uplink stream with streams PassThrough
      uplinkStream.pipe(searchStream, {
        end: false
      });
      uplinkStream.on('error', function (err) {
        self.logger.error({
          err: err
        }, 'uplink error: @{err.message}');
        cb();
        // to avoid call callback more than once
        cb = function () {};
      });
      uplinkStream.on('end', function () {
        cb();
        // to avoid call callback more than once
        cb = function () {};
      });
      searchStream.abort = function () {
        if (uplinkStream.abort) {
          uplinkStream.abort();
        }
        cb();
        // to avoid call callback more than once
        cb = function () {};
      };
    },
    // executed after all series
    function () {
      // attach a local search results
      const localSearchStream = self.localStorage.search(startkey, options);
      searchStream.abort = function () {
        localSearchStream.abort();
      };
      localSearchStream.pipe(searchStream, {
        end: true
      });
      localSearchStream.on('error', function (err) {
        self.logger.error({
          err: err
        }, 'search error: @{err.message}');
        searchStream.end();
      });
    });
    return searchStream;
  }

  /**
   * Retrieve only private local packages
   * @param {*} callback
   */
  getLocalDatabase(callback) {
    const self = this;
    this.localStorage.storagePlugin.get((err, locals) => {
      if (err) {
        callback(err);
      }
      const packages = [];
      const getPackage = function (itemPkg) {
        self.localStorage.getPackageMetadata(locals[itemPkg], function (err, pkgMetadata) {
          if (_lodash.default.isNil(err)) {
            const latest = pkgMetadata[_constants.DIST_TAGS].latest;
            if (latest && pkgMetadata.versions[latest]) {
              const version = pkgMetadata.versions[latest];
              const timeList = pkgMetadata.time;
              const time = timeList[latest];
              // @ts-ignore
              version.time = time;

              // Add for stars api
              // @ts-ignore
              version.users = pkgMetadata.users;
              packages.push(version);
            } else {
              self.logger.warn({
                package: locals[itemPkg]
              }, 'package @{package} does not have a "latest" tag?');
            }
          }
          if (itemPkg >= locals.length - 1) {
            callback(null, packages);
          } else {
            getPackage(itemPkg + 1);
          }
        });
      };
      if (locals.length) {
        getPackage(0);
      } else {
        callback(null, []);
      }
    });
  }

  /**
   * Function fetches package metadata from uplinks and synchronizes it with local data
   if package is available locally, it MUST be provided in pkginfo
   returns callback(err, result, uplink_errors)
   */
  _syncUplinksMetadata(name, packageInfo, options, callback) {
    let found = true;
    const self = this;
    const upLinks = [];
    const hasToLookIntoUplinks = _lodash.default.isNil(options.uplinksLook) || options.uplinksLook;
    if (!packageInfo) {
      found = false;
      packageInfo = (0, _storageUtils.generatePackageTemplate)(name);
    }
    for (const uplink in this.uplinks) {
      if ((0, _configUtils.hasProxyTo)(name, uplink, this.config.packages) && hasToLookIntoUplinks) {
        upLinks.push(this.uplinks[uplink]);
      }
    }
    _async.default.map(upLinks, (upLink, cb) => {
      const _options = Object.assign({}, options);
      const upLinkMeta = packageInfo._uplinks[upLink.upname];
      if ((0, _utils.isObject)(upLinkMeta)) {
        const fetched = upLinkMeta.fetched;
        if (fetched && Date.now() - fetched < upLink.maxage) {
          return cb();
        }
        _options.etag = upLinkMeta.etag;
      }
      upLink.getRemoteMetadata(name, _options, (err, upLinkResponse, eTag) => {
        if (err && err.remoteStatus === 304) {
          upLinkMeta.fetched = Date.now();
        }
        if (err || !upLinkResponse) {
          return cb(null, [err || _utils.ErrorCode.getInternalError('no data')]);
        }
        try {
          upLinkResponse = _core.validatioUtils.normalizeMetadata(upLinkResponse, name);
        } catch (err) {
          self.logger.error({
            sub: 'out',
            err: err
          }, 'package.json validating error @{!err.message}\n@{err.stack}');
          return cb(null, [err]);
        }
        packageInfo._uplinks[upLink.upname] = {
          etag: eTag,
          fetched: Date.now()
        };
        packageInfo = (0, _storageUtils.mergeUplinkTimeIntoLocal)(packageInfo, upLinkResponse);
        (0, _uplinkUtil.updateVersionsHiddenUpLink)(upLinkResponse.versions, upLink);
        try {
          (0, _metadataUtils.mergeVersions)(packageInfo, upLinkResponse);
        } catch (err) {
          self.logger.error({
            sub: 'out',
            err: err
          }, 'package.json parsing error @{!err.message}\n@{err.stack}');
          return cb(null, [err]);
        }

        // if we got to this point, assume that the correct package exists
        // on the uplink
        found = true;
        cb();
      });
    },
    // @ts-ignore
    (err, upLinksErrors) => {
      (0, _assert.default)(!err && Array.isArray(upLinksErrors));

      // Check for connection timeout or reset errors with uplink(s)
      // (these should be handled differently from the package not being found)
      if (!found) {
        let uplinkTimeoutError;
        for (let i = 0; i < upLinksErrors.length; i++) {
          if (upLinksErrors[i]) {
            for (let j = 0; j < upLinksErrors[i].length; j++) {
              if (upLinksErrors[i][j]) {
                const code = upLinksErrors[i][j].code;
                if (code === 'ETIMEDOUT' || code === 'ESOCKETTIMEDOUT' || code === 'ECONNRESET') {
                  uplinkTimeoutError = true;
                  break;
                }
              }
            }
          }
        }
        if (uplinkTimeoutError) {
          return callback(_utils.ErrorCode.getServiceUnavailable(), null, upLinksErrors);
        }
        return callback(_utils.ErrorCode.getNotFound(_constants.API_ERROR.NO_PACKAGE), null, upLinksErrors);
      }
      if (upLinks.length === 0) {
        return callback(null, packageInfo);
      }
      self.localStorage.updateVersions(name, packageInfo, async (err, packageJsonLocal) => {
        if (err) {
          return callback(err);
        }
        // Any error here will cause a 404, like an uplink error. This is likely the right thing to do
        // as a broken filter is a security risk.
        const filterErrors = [];
        // This MUST be done serially and not in parallel as they modify packageJsonLocal
        for (const filter of self.filters) {
          try {
            // These filters can assume it's save to modify packageJsonLocal and return it directly for
            // performance (i.e. need not be pure)
            packageJsonLocal = await filter.filter_metadata(packageJsonLocal);
          } catch (err) {
            filterErrors.push(err);
          }
        }
        callback(null, packageJsonLocal, _lodash.default.concat(upLinksErrors, filterErrors));
      });
    });
  }

  /**
   * Set a hidden value for each version.
   * @param {Array} versions list of version
   * @param {String} upLink uplink name
   * @private
   */
  _updateVersionsHiddenUpLink(versions, upLink) {
    for (const i in versions) {
      if (Object.prototype.hasOwnProperty.call(versions, i)) {
        const version = versions[i];

        // holds a "hidden" value to be used by the package storage.
        version[Symbol.for('__verdaccio_uplink')] = upLink.upname;
      }
    }
  }
}
var _default = Storage;
exports.default = _default;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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