"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.WHITELIST = void 0;
exports.checkPackageLocal = checkPackageLocal;
exports.checkPackageRemote = checkPackageRemote;
exports.cleanUpLinksRef = cleanUpLinksRef;
exports.cleanUpReadme = cleanUpReadme;
exports.convertAbbreviatedManifest = convertAbbreviatedManifest;
exports.generatePackageTemplate = generatePackageTemplate;
exports.generateRevision = generateRevision;
exports.getLatestReadme = getLatestReadme;
exports.hasInstallScript = hasInstallScript;
exports.isPublishablePackage = isPublishablePackage;
exports.mergeUplinkTimeIntoLocal = mergeUplinkTimeIntoLocal;
exports.normalizePackage = normalizePackage;
exports.prepareSearchPackage = prepareSearchPackage;
exports.publishPackage = publishPackage;
var _lodash = _interopRequireDefault(require("lodash"));
var _core = require("@verdaccio/core");
var _utils = require("@verdaccio/utils");
var _constants = require("./constants");
var _search = _interopRequireDefault(require("./search"));
var _utils2 = require("./utils");
function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }
function ownKeys(object, enumerableOnly) { var keys = Object.keys(object); if (Object.getOwnPropertySymbols) { var symbols = Object.getOwnPropertySymbols(object); enumerableOnly && (symbols = symbols.filter(function (sym) { return Object.getOwnPropertyDescriptor(object, sym).enumerable; })), keys.push.apply(keys, symbols); } return keys; }
function _objectSpread(target) { for (var i = 1; i < arguments.length; i++) { var source = null != arguments[i] ? arguments[i] : {}; i % 2 ? ownKeys(Object(source), !0).forEach(function (key) { _defineProperty(target, key, source[key]); }) : Object.getOwnPropertyDescriptors ? Object.defineProperties(target, Object.getOwnPropertyDescriptors(source)) : ownKeys(Object(source)).forEach(function (key) { Object.defineProperty(target, key, Object.getOwnPropertyDescriptor(source, key)); }); } return target; }
function _defineProperty(obj, key, value) { key = _toPropertyKey(key); if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }
function _toPropertyKey(arg) { var key = _toPrimitive(arg, "string"); return typeof key === "symbol" ? key : String(key); }
function _toPrimitive(input, hint) { if (typeof input !== "object" || input === null) return input; var prim = input[Symbol.toPrimitive]; if (prim !== undefined) { var res = prim.call(input, hint || "default"); if (typeof res !== "object") return res; throw new TypeError("@@toPrimitive must return a primitive value."); } return (hint === "string" ? String : Number)(input); }
function generatePackageTemplate(name) {
  return {
    // standard things
    name,
    versions: {},
    time: {},
    [_constants.USERS]: {},
    [_constants.DIST_TAGS]: {},
    _uplinks: {},
    _distfiles: {},
    _attachments: {},
    _rev: ''
  };
}

/**
 * Normalize package properties, tags, revision id.
 * @param {Object} pkg package reference.
 */
function normalizePackage(pkg) {
  const pkgProperties = ['versions', 'dist-tags', '_distfiles', '_attachments', '_uplinks', 'time'];
  pkgProperties.forEach(key => {
    const pkgProp = pkg[key];
    if (_lodash.default.isNil(pkgProp) || (0, _utils2.isObject)(pkgProp) === false) {
      pkg[key] = {};
    }
  });
  if (_lodash.default.isString(pkg._rev) === false) {
    pkg._rev = _constants.STORAGE.DEFAULT_REVISION;
  }
  if (_lodash.default.isString(pkg._id) === false) {
    pkg._id = pkg.name;
  }

  // normalize dist-tags
  (0, _utils2.normalizeDistTags)(pkg);
  return pkg;
}
function generateRevision(rev) {
  const _rev = rev.split('-');
  return (+_rev[0] || 0) + 1 + '-' + (0, _utils.generateRandomHexString)();
}
function getLatestReadme(pkg) {
  const versions = pkg['versions'] || {};
  const distTags = pkg[_constants.DIST_TAGS] || {};
  // FIXME: here is a bit tricky add the types
  const latestVersion = distTags['latest'] ? versions[distTags['latest']] || {} : {};
  let readme = _lodash.default.trim(pkg.readme || latestVersion.readme || '');
  if (readme) {
    return readme;
  }

  // In case of empty readme - trying to get ANY readme in the following order: 'next','beta','alpha','test','dev','canary'
  const readmeDistTagsPriority = ['next', 'beta', 'alpha', 'test', 'dev', 'canary'];
  readmeDistTagsPriority.map(function (tag) {
    if (readme) {
      return readme;
    }
    const version = distTags[tag] ? versions[distTags[tag]] || {} : {};
    readme = _lodash.default.trim(version.readme || readme);
  });
  return readme;
}
function cleanUpReadme(version) {
  if (_lodash.default.isNil(version) === false) {
    // @ts-ignore
    delete version.readme;
  }
  return version;
}
const WHITELIST = ['_rev', 'name', 'versions', 'dist-tags', 'readme', 'time', '_id', 'users'];
exports.WHITELIST = WHITELIST;
function cleanUpLinksRef(keepUpLinkData, result) {
  const propertyToKeep = [...WHITELIST];
  if (keepUpLinkData === true) {
    propertyToKeep.push('_uplinks');
  }
  for (const i in result) {
    if (propertyToKeep.indexOf(i) === -1) {
      // Remove sections like '_uplinks' from response
      delete result[i];
    }
  }
  return result;
}

/**
 * Check whether a package it is already a local package
 * @param {*} name
 * @param {*} localStorage
 */
function checkPackageLocal(name, localStorage) {
  return new Promise((resolve, reject) => {
    localStorage.getPackageMetadata(name, (err, results) => {
      if (!_lodash.default.isNil(err) && err.status !== _constants.HTTP_STATUS.NOT_FOUND) {
        return reject(err);
      }
      if (results) {
        return reject(_utils2.ErrorCode.getConflict(_constants.API_ERROR.PACKAGE_EXIST));
      }
      return resolve();
    });
  });
}
function publishPackage(name, metadata, localStorage) {
  return new Promise((resolve, reject) => {
    localStorage.addPackage(name, metadata, (err, latest) => {
      if (!_lodash.default.isNull(err)) {
        return reject(err);
      } else if (!_lodash.default.isUndefined(latest)) {
        _search.default.add(latest);
      }
      return resolve();
    });
  });
}
function checkPackageRemote(name, isAllowPublishOffline, syncMetadata) {
  return new Promise((resolve, reject) => {
    syncMetadata(name, null, {}, (err, packageJsonLocal, upLinksErrors) => {
      // something weird
      if (err && err.status !== _constants.HTTP_STATUS.NOT_FOUND) {
        return reject(err);
      }

      // checking package exist already
      if (_lodash.default.isNil(packageJsonLocal) === false) {
        return reject(_utils2.ErrorCode.getConflict(_constants.API_ERROR.PACKAGE_EXIST));
      }
      for (let errorItem = 0; errorItem < upLinksErrors.length; errorItem++) {
        // checking error
        // if uplink fails with a status other than 404, we report failure
        if (_lodash.default.isNil(upLinksErrors[errorItem][0]) === false) {
          if (upLinksErrors[errorItem][0].status !== _constants.HTTP_STATUS.NOT_FOUND) {
            if (isAllowPublishOffline) {
              return resolve();
            }
            return reject(_utils2.ErrorCode.getServiceUnavailable(_constants.API_ERROR.UPLINK_OFFLINE_PUBLISH));
          }
        }
      }
      return resolve();
    });
  });
}

// tested on v6
function mergeUplinkTimeIntoLocal(cacheManifest, remoteManifest) {
  if ('time' in remoteManifest) {
    // remote override cache time conflicts
    return _objectSpread(_objectSpread({}, cacheManifest), {}, {
      time: _objectSpread(_objectSpread({}, cacheManifest.time), remoteManifest.time)
    });
  }
  return cacheManifest;
}
function prepareSearchPackage(data, time) {
  const latest = _core.pkgUtils.getLatest(data);
  if (latest && data.versions[latest]) {
    const version = data.versions[latest];
    const versions = {
      [latest]: 'latest'
    };
    const pkg = {
      name: version.name,
      description: version.description,
      [_constants.DIST_TAGS]: {
        latest
      },
      maintainers: version.maintainers || [version.author].filter(Boolean),
      author: version.author,
      repository: version.repository,
      readmeFilename: version.readmeFilename || '',
      homepage: version.homepage,
      keywords: version.keywords,
      time: {
        modified: time
      },
      bugs: version.bugs,
      license: version.license,
      versions
    };
    return pkg;
  }
}

/**
 * Check whether the package metadta has enough data to be published
 * @param pkg metadata
 */
function isPublishablePackage(pkg) {
  const keys = Object.keys(pkg);
  return keys.includes('versions');
}
function hasInstallScript(version) {
  if (version !== null && version !== void 0 && version.scripts) {
    const scripts = Object.keys(version.scripts);
    return scripts.find(item => {
      return ['install', 'preinstall', 'postinstall'].includes(item);
    }) !== undefined;
  }
  return false;
}
function convertAbbreviatedManifest(manifest) {
  var _manifest$time;
  const abbreviatedVersions = Object.keys(manifest.versions).reduce((acc, version) => {
    const _version = manifest.versions[version];
    // This should be align with this document
    // https://github.com/npm/registry/blob/master/docs/responses/package-metadata.md#abbreviated-version-object
    const _version_abbreviated = {
      name: _version.name,
      version: _version.version,
      description: _version.description,
      deprecated: _version.deprecated,
      bin: _version.bin,
      dist: _version.dist,
      engines: _version.engines,
      cpu: _version.cpu,
      os: _version.os,
      funding: _version.funding,
      directories: _version.directories,
      dependencies: _version.dependencies,
      devDependencies: _version.devDependencies,
      peerDependencies: _version.peerDependencies,
      peerDependenciesMeta: _version.peerDependenciesMeta,
      optionalDependencies: _version.optionalDependencies,
      bundleDependencies: _version.bundleDependencies,
      // npm cli specifics
      _hasShrinkwrap: _version._hasShrinkwrap,
      hasInstallScript: hasInstallScript(_version)
    };
    acc[version] = _version_abbreviated;
    return acc;
  }, {});
  const convertedManifest = {
    name: manifest['name'],
    [_constants.DIST_TAGS]: manifest[_constants.DIST_TAGS],
    versions: abbreviatedVersions,
    // @ts-ignore
    modified: manifest === null || manifest === void 0 ? void 0 : (_manifest$time = manifest.time) === null || _manifest$time === void 0 ? void 0 : _manifest$time.modified,
    // NOTE: special case for pnpm https://github.com/pnpm/rfcs/pull/2
    time: manifest === null || manifest === void 0 ? void 0 : manifest.time
  };

  // @ts-ignore
  return convertedManifest;
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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