"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createServerFactory = createServerFactory;
exports.displayExperimentsInfoBox = displayExperimentsInfoBox;
exports.runServer = runServer;
var _constants = _interopRequireDefault(require("constants"));
var _debug = _interopRequireDefault(require("debug"));
var _fs = _interopRequireDefault(require("fs"));
var _http = _interopRequireDefault(require("http"));
var _https = _interopRequireDefault(require("https"));
var _lodash = _interopRequireWildcard(require("lodash"));
var _path = _interopRequireDefault(require("path"));
var _index = _interopRequireDefault(require("../api/index"));
var _utils = require("./cli/utils");
var _configPath = _interopRequireDefault(require("./config-path"));
var _constants2 = require("./constants");
var _utils2 = require("./utils");
function _getRequireWildcardCache(nodeInterop) { if (typeof WeakMap !== "function") return null; var cacheBabelInterop = new WeakMap(); var cacheNodeInterop = new WeakMap(); return (_getRequireWildcardCache = function (nodeInterop) { return nodeInterop ? cacheNodeInterop : cacheBabelInterop; })(nodeInterop); }
function _interopRequireWildcard(obj, nodeInterop) { if (!nodeInterop && obj && obj.__esModule) { return obj; } if (obj === null || typeof obj !== "object" && typeof obj !== "function") { return { default: obj }; } var cache = _getRequireWildcardCache(nodeInterop); if (cache && cache.has(obj)) { return cache.get(obj); } var newObj = {}; var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var key in obj) { if (key !== "default" && Object.prototype.hasOwnProperty.call(obj, key)) { var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null; if (desc && (desc.get || desc.set)) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } newObj.default = obj; if (cache) { cache.set(obj, newObj); } return newObj; }
function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }
function ownKeys(object, enumerableOnly) { var keys = Object.keys(object); if (Object.getOwnPropertySymbols) { var symbols = Object.getOwnPropertySymbols(object); enumerableOnly && (symbols = symbols.filter(function (sym) { return Object.getOwnPropertyDescriptor(object, sym).enumerable; })), keys.push.apply(keys, symbols); } return keys; }
function _objectSpread(target) { for (var i = 1; i < arguments.length; i++) { var source = null != arguments[i] ? arguments[i] : {}; i % 2 ? ownKeys(Object(source), !0).forEach(function (key) { _defineProperty(target, key, source[key]); }) : Object.getOwnPropertyDescriptors ? Object.defineProperties(target, Object.getOwnPropertyDescriptors(source)) : ownKeys(Object(source)).forEach(function (key) { Object.defineProperty(target, key, Object.getOwnPropertyDescriptor(source, key)); }); } return target; }
function _defineProperty(obj, key, value) { key = _toPropertyKey(key); if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }
function _toPropertyKey(arg) { var key = _toPrimitive(arg, "string"); return typeof key === "symbol" ? key : String(key); }
function _toPrimitive(input, hint) { if (typeof input !== "object" || input === null) return input; var prim = input[Symbol.toPrimitive]; if (prim !== undefined) { var res = prim.call(input, hint || "default"); if (typeof res !== "object") return res; throw new TypeError("@@toPrimitive must return a primitive value."); } return (hint === "string" ? String : Number)(input); }
const debug = (0, _debug.default)('verdaccio');
const logger = require('./logger');
function displayExperimentsInfoBox(flags) {
  if (!flags) {
    return;
  }
  const experimentList = Object.keys(flags);
  if (experimentList.length >= 1) {
    logger.warn(
    // eslint-disable-next-line max-len
    `experiments are enabled, it is recommended do not use experiments in production comment out this section to disable it`);
    experimentList.forEach(experiment => {
      // eslint-disable-next-line max-len
      logger.info(`support for experiment [${experiment}] ${flags[experiment] ? 'is enabled' : ' is disabled'}`);
    });
  }
}

/**
 * Exposes a server factory to be instantiated programmatically.
 *
    const app = await runServer(); // default configuration
    const app = await runServer('./config/config.yaml');
    const app = await runServer({ configuration });
    app.listen(4000, (event) => {
      // do something
    });
 * @param config
 */
async function runServer(config) {
  let configurationParsed;
  if (config === undefined || typeof config === 'string') {
    const configPathLocation = (0, _configPath.default)(config);
    configurationParsed = (0, _utils2.parseConfigFile)(configPathLocation);
    if (!configurationParsed.self_path) {
      configurationParsed.self_path = _path.default.resolve(configPathLocation);
    }
  } else if (_lodash.default.isObject(config)) {
    configurationParsed = config;
    if (!configurationParsed.self_path) {
      throw new Error('self_path is required, please provide a valid root path for storage');
    }
  } else {
    throw new Error(_constants2.API_ERROR.CONFIG_BAD_FORMAT);
  }
  const addresses = (0, _utils.getListListenAddresses)(undefined, configurationParsed.listen);
  if (addresses.length > 1) {
    process.emitWarning('You have specified multiple listen addresses, using this method only the first will be used');
  }
  const app = await (0, _index.default)(configurationParsed);
  return createServerFactory(configurationParsed, addresses[0], app);
}

/**
 * Return a native HTTP/HTTPS server instance
 * @param config
 * @param addr
 * @param app
 */
function createServerFactory(config, addr, app) {
  let serverFactory;
  if (addr.proto === 'https') {
    debug('https enabled');
    try {
      let httpsOptions = {
        // disable insecure SSLv2 and SSLv3
        secureOptions: _constants.default.SSL_OP_NO_SSLv2 | _constants.default.SSL_OP_NO_SSLv3
      };
      const keyCertConfig = config.https;
      const pfxConfig = config.https;

      // https must either have key and cert or a pfx and (optionally) a passphrase
      if (!(keyCertConfig.key && keyCertConfig.cert || pfxConfig.pfx)) {
        throw Error('bad format https configuration');
      }
      if (pfxConfig.pfx) {
        const {
          pfx,
          passphrase
        } = pfxConfig;
        httpsOptions = (0, _lodash.assign)(httpsOptions, {
          pfx: _fs.default.readFileSync(pfx),
          passphrase: passphrase || ''
        });
      } else {
        const {
          key,
          cert,
          ca
        } = keyCertConfig;
        httpsOptions = (0, _lodash.assign)(httpsOptions, _objectSpread({
          key: _fs.default.readFileSync(key),
          cert: _fs.default.readFileSync(cert)
        }, ca && {
          ca: _fs.default.readFileSync(ca)
        }));
      }
      // TODO: enable http2 as feature
      // if (config.server.http2) <-- check if force http2
      serverFactory = _https.default.createServer(httpsOptions, app);
    } catch (err) {
      throw new Error(`cannot create https server: ${err.message}`);
    }
  } else {
    // http
    debug('http enabled');
    serverFactory = _http.default.createServer(app);
  }
  if (config.server && typeof config.server.keepAliveTimeout !== 'undefined' &&
  // @ts-ignore
  config.server.keepAliveTimeout !== 'null') {
    // library definition for node is not up to date (doesn't contain recent 8.0 changes)
    serverFactory.keepAliveTimeout = config.server.keepAliveTimeout * 1000;
  }
  // FIXE: I could not find the reason of this code.
  unlinkAddressPath(addr);
  return serverFactory;
}
function unlinkAddressPath(addr) {
  if (addr.path && _fs.default.existsSync(addr.path)) {
    _fs.default.unlinkSync(addr.path);
  }
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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