"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = loadPlugin;
var _debug = _interopRequireDefault(require("debug"));
var _lodash = _interopRequireDefault(require("lodash"));
var _path = _interopRequireDefault(require("path"));
var _constants = require("./constants");
var _logger = require("./logger");
function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }
const debug = (0, _debug.default)('verdaccio:plugin:loader');

/**
 * Requires a module.
 * @param {*} path the module's path
 * @return {Object}
 */
function tryLoad(path) {
  try {
    debug('loading plugin %s', path);
    return require(path);
  } catch (err) {
    if (err.code === _constants.MODULE_NOT_FOUND) {
      debug('plugin %s not found', path);
      return null;
    }
    _logger.logger.error({
      err: err.msg
    }, 'error loading plugin @{err}');
    throw err;
  }
}
function mergeConfig(appConfig, pluginConfig) {
  return _lodash.default.merge(appConfig, pluginConfig);
}
function isValid(plugin) {
  return _lodash.default.isFunction(plugin) || _lodash.default.isFunction(plugin.default);
}
function isES6(plugin) {
  return Object.keys(plugin).includes('default');
}

// export type PluginGeneric<R, T extends IPlugin<R> = ;

/**
 * Load a plugin following the rules
 * - First try to load from the internal directory plugins (which will disappear soon or later).
 * - If the package is scoped eg: @scope/foo, try to load as a package
 * - A second attempt from the external plugin directory
 * - A third attempt from node_modules, in case to have multiple match as for instance verdaccio-ldap
 * and sinopia-ldap. All verdaccio prefix will have preferences.
 * @param {*} config a reference of the configuration settings
 * @param {*} pluginConfigs
 * @param {*} params a set of params to initialize the plugin
 * @param {*} sanityCheck callback that check the shape that should fulfill the plugin
 * @return {Array} list of plugins
 */
function loadPlugin(config, pluginConfigs = {}, params, sanityCheck, prefix = 'verdaccio') {
  return Object.keys(pluginConfigs).map(pluginId => {
    let plugin;
    const isScoped = pluginId.startsWith('@') && pluginId.includes('/');
    debug('isScoped %s', isScoped);
    if (isScoped) {
      plugin = tryLoad(pluginId);
    }
    const localPlugin = _path.default.resolve(__dirname + '/../plugins', pluginId);
    // try local plugins first
    plugin = tryLoad(localPlugin);

    // try the external plugin directory
    if (plugin === null && config.plugins) {
      const pluginDir = config.plugins;
      const externalFilePlugin = _path.default.resolve(pluginDir, pluginId);
      plugin = tryLoad(externalFilePlugin);

      // npm package
      if (plugin === null && pluginId.match(/^[^\.\/]/)) {
        plugin = tryLoad(_path.default.resolve(pluginDir, `${prefix}-${pluginId}`));
        // compatibility for old sinopia plugins
        if (!plugin) {
          plugin = tryLoad(_path.default.resolve(pluginDir, `sinopia-${pluginId}`));
          if (plugin) {
            _logger.logger.warn({
              name: pluginId
            }, `plugin names that start with sinopia-* will be removed in the future, please rename package to verdaccio-*`);
          }
        }
      }
    }

    // npm package
    if (plugin === null && pluginId.match(/^[^\.\/]/)) {
      plugin = tryLoad(`${prefix}-${pluginId}`);
      // compatibility for old sinopia plugins
      if (!plugin) {
        plugin = tryLoad(`sinopia-${pluginId}`);
      }
      if (plugin) {
        debug('plugin %s is an npm package', pluginId);
      }
    }
    if (plugin === null) {
      plugin = tryLoad(pluginId);
    }

    // relative to config path
    if (plugin === null && pluginId.match(/^\.\.?($|\/)/)) {
      plugin = tryLoad(_path.default.resolve(_path.default.dirname(config.self_path), pluginId));
    }
    if (plugin === null) {
      if (isScoped) {
        _logger.logger.error({
          content: pluginId
        }, 'plugin not found. try npm install @{content}');
      } else {
        _logger.logger.error({
          content: pluginId,
          prefix
        }, 'plugin not found. try npm install @{prefix}-@{content}');
      }
      const msg = isScoped ? `
      ${pluginId} plugin not found. try "npm install ${pluginId}"` : `
      ${prefix}-${pluginId} plugin not found. try "npm install ${prefix}-${pluginId}"`;
      throw Error(msg);
    }
    if (!isValid(plugin)) {
      _logger.logger.error({
        content: pluginId
      }, '@{prefix}-@{content} plugin does not have the right code structure');
      throw Error(`"${pluginId}" plugin does not have the right code structure`);
    }

    /* eslint new-cap:off */
    try {
      if (isES6(plugin)) {
        debug('plugin is ES6');
        plugin = new plugin.default(mergeConfig(config, pluginConfigs[pluginId]), params);
      } else {
        debug('plugin is commonJS');
        plugin = plugin(pluginConfigs[pluginId], params);
      }
    } catch (error) {
      plugin = null;
      _logger.logger.error({
        error,
        pluginId
      }, 'error loading a plugin @{pluginId}: @{error}');
    }
    /* eslint new-cap:off */

    if (plugin === null || !sanityCheck(plugin)) {
      if (isScoped) {
        _logger.logger.error({
          content: pluginId
        }, "@{content} doesn't look like a valid plugin");
      } else {
        _logger.logger.error({
          content: pluginId,
          prefix
        }, "@{prefix}-@{content} doesn't look like a valid plugin");
      }
      throw Error(`sanity check has failed, "${pluginId}" is not a valid plugin`);
    }
    if (isScoped) {
      _logger.logger.info({
        content: pluginId
      }, 'plugin successfully loaded: @{content}');
    } else {
      _logger.logger.info({
        content: pluginId,
        prefix
      }, 'plugin successfully loaded: @{prefix}-@{content}');
    }
    return plugin;
  });
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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