"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;
var _assert = _interopRequireDefault(require("assert"));
var _debug = _interopRequireDefault(require("debug"));
var _lodash = _interopRequireDefault(require("lodash"));
var _url = _interopRequireDefault(require("url"));
var _localStorage = _interopRequireDefault(require("@verdaccio/local-storage"));
var _streams = require("@verdaccio/streams");
var _utils = require("@verdaccio/utils");
var _pluginLoader = _interopRequireDefault(require("../lib/plugin-loader"));
var _constants = require("./constants");
var _storageUtils = require("./storage-utils");
var _utils2 = require("./utils");
function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }
function _defineProperty(obj, key, value) { key = _toPropertyKey(key); if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }
function _toPropertyKey(arg) { var key = _toPrimitive(arg, "string"); return typeof key === "symbol" ? key : String(key); }
function _toPrimitive(input, hint) { if (typeof input !== "object" || input === null) return input; var prim = input[Symbol.toPrimitive]; if (prim !== undefined) { var res = prim.call(input, hint || "default"); if (typeof res !== "object") return res; throw new TypeError("@@toPrimitive must return a primitive value."); } return (hint === "string" ? String : Number)(input); }
const debug = (0, _debug.default)('verdaccio:local-storage');
/**
 * Implements Storage interface (same for storage.js, local-storage.js, up-storage.js).
 */
class LocalStorage {
  constructor(config, logger) {
    _defineProperty(this, "config", void 0);
    _defineProperty(this, "storagePlugin", void 0);
    _defineProperty(this, "logger", void 0);
    this.logger = logger;
    this.config = config;
    this.storagePlugin = this._loadStorage(config, logger);
  }
  addPackage(name, pkg, callback) {
    const storage = this._getLocalStorage(name);
    if (_lodash.default.isNil(storage)) {
      return callback(_utils2.ErrorCode.getNotFound('this package cannot be added'));
    }
    storage.createPackage(name, (0, _storageUtils.generatePackageTemplate)(name), err => {
      // FIXME: it will be fixed here https://github.com/verdaccio/verdaccio/pull/1360
      // @ts-ignore
      if (_lodash.default.isNull(err) === false && (err.code === _constants.STORAGE.FILE_EXIST_ERROR || err.code === _constants.HTTP_STATUS.CONFLICT)) {
        return callback(_utils2.ErrorCode.getConflict());
      }
      const latest = (0, _utils.getLatestVersion)(pkg);
      if (_lodash.default.isNil(latest) === false && pkg.versions[latest]) {
        return callback(null, pkg.versions[latest]);
      }
      return callback();
    });
  }

  /**
   * Remove package.
   * @param {*} name
   * @param {*} callback
   * @return {Function}
   */
  removePackage(name, callback) {
    const storage = this._getLocalStorage(name);
    debug('[storage] removing package %o', name);
    if (_lodash.default.isNil(storage)) {
      return callback(_utils2.ErrorCode.getNotFound());
    }
    storage.readPackage(name, (err, data) => {
      if (_lodash.default.isNil(err) === false) {
        if (err.code === _constants.STORAGE.NO_SUCH_FILE_ERROR || err.code === _constants.HTTP_STATUS.NOT_FOUND) {
          return callback(_utils2.ErrorCode.getNotFound());
        }
        return callback(err);
      }
      data = (0, _storageUtils.normalizePackage)(data);
      this.storagePlugin.remove(name, removeFailed => {
        if (removeFailed) {
          // This will happen when database is locked
          this.logger.error({
            name
          }, `[storage/removePackage] the database is locked, removed has failed for @{name}`);
          return callback(_utils2.ErrorCode.getBadData(removeFailed.message));
        }
        storage.deletePackage(_constants.STORAGE.PACKAGE_FILE_NAME, err => {
          if (err) {
            return callback(err);
          }
          const attachments = Object.keys(data._attachments);
          this._deleteAttachments(storage, attachments, callback);
        });
      });
    });
  }

  /**
   * Synchronize remote package info with the local one
   * @param {*} name
   * @param {*} packageInfo
   * @param {*} callback
   */
  updateVersions(name, packageInfo, callback) {
    this._readCreatePackage(name, (err, packageLocalJson) => {
      if (err) {
        return callback(err);
      }
      let change = false;
      // updating readme
      packageLocalJson.readme = (0, _storageUtils.getLatestReadme)(packageInfo);
      if (packageInfo.readme !== packageLocalJson.readme) {
        change = true;
      }
      for (const versionId in packageInfo.versions) {
        if (_lodash.default.isNil(packageLocalJson.versions[versionId])) {
          let version = packageInfo.versions[versionId];

          // we don't keep readme for package versions,
          // only one readme per package
          version = (0, _storageUtils.cleanUpReadme)(version);
          version.contributors = (0, _utils.normalizeContributors)(version.contributors);
          change = true;
          packageLocalJson.versions[versionId] = version;
          if (version.dist && version.dist.tarball) {
            const urlObject = _url.default.parse(version.dist.tarball);
            const filename = urlObject.pathname.replace(/^.*\//, '');

            // we do NOT overwrite any existing records
            if (_lodash.default.isNil(packageLocalJson._distfiles[filename])) {
              const hash = packageLocalJson._distfiles[filename] = {
                url: version.dist.tarball,
                sha: version.dist.shasum
              };
              /* eslint spaced-comment: 0 */
              const upLink = version[Symbol.for('__verdaccio_uplink')];
              if (_lodash.default.isNil(upLink) === false) {
                this._updateUplinkToRemoteProtocol(hash, upLink);
              }
            }
          }
        }
      }
      for (const tag in packageInfo[_constants.DIST_TAGS]) {
        if (!packageLocalJson[_constants.DIST_TAGS][tag] || packageLocalJson[_constants.DIST_TAGS][tag] !== packageInfo[_constants.DIST_TAGS][tag]) {
          change = true;
          packageLocalJson[_constants.DIST_TAGS][tag] = packageInfo[_constants.DIST_TAGS][tag];
        }
      }
      for (const up in packageInfo._uplinks) {
        if (Object.prototype.hasOwnProperty.call(packageInfo._uplinks, up)) {
          const need_change = !(0, _utils2.isObject)(packageLocalJson._uplinks[up]) || packageInfo._uplinks[up].etag !== packageLocalJson._uplinks[up].etag || packageInfo._uplinks[up].fetched !== packageLocalJson._uplinks[up].fetched;
          if (need_change) {
            change = true;
            packageLocalJson._uplinks[up] = packageInfo._uplinks[up];
          }
        }
      }
      if ('time' in packageInfo && !_lodash.default.isEqual(packageLocalJson.time, packageInfo.time)) {
        packageLocalJson.time = packageInfo.time;
        change = true;
      }
      if (change) {
        debug('updating package %o info', name);
        this._writePackage(name, packageLocalJson, function (err) {
          callback(err, packageLocalJson);
        });
      } else {
        callback(null, packageLocalJson);
      }
    });
  }

  /**
   * Add a new version to a previous local package.
   * @param {*} name
   * @param {*} version
   * @param {*} metadata
   * @param {*} tag
   * @param {*} callback
   */
  addVersion(name, version, metadata, tag, callback) {
    this._updatePackage(name, (data, cb) => {
      // keep only one readme per package
      data.readme = metadata.readme;

      // TODO: lodash remove
      metadata = (0, _storageUtils.cleanUpReadme)(metadata);
      metadata.contributors = (0, _utils.normalizeContributors)(metadata.contributors);
      const hasVersion = data.versions[version] != null;
      if (hasVersion) {
        return cb(_utils2.ErrorCode.getConflict());
      }

      // if uploaded tarball has a different shasum, it's very likely that we have some kind of error
      if ((0, _utils2.isObject)(metadata.dist) && _lodash.default.isString(metadata.dist.tarball)) {
        const tarball = metadata.dist.tarball.replace(/.*\//, '');
        if ((0, _utils2.isObject)(data._attachments[tarball])) {
          if (_lodash.default.isNil(data._attachments[tarball].shasum) === false && _lodash.default.isNil(metadata.dist.shasum) === false) {
            if (data._attachments[tarball].shasum != metadata.dist.shasum) {
              const errorMessage = `shasum error, ${data._attachments[tarball].shasum} != ${metadata.dist.shasum}`;
              return cb(_utils2.ErrorCode.getBadRequest(errorMessage));
            }
          }
          const currentDate = new Date().toISOString();

          // some old storage do not have this field #740
          if (_lodash.default.isNil(data.time)) {
            data.time = {};
          }
          data.time['modified'] = currentDate;
          if ('created' in data.time === false) {
            data.time.created = currentDate;
          }
          data.time[version] = currentDate;
          data._attachments[tarball].version = version;
        }
      }
      data.versions[version] = metadata;
      (0, _utils2.tagVersion)(data, version, tag);
      this.storagePlugin.add(name, addFailed => {
        if (addFailed) {
          return cb(_utils2.ErrorCode.getBadData(addFailed.message));
        }
        cb();
      });
    }, callback);
  }

  /**
   * Merge a new list of tags for a local packages with the existing one.
   * @param {*} pkgName
   * @param {*} tags
   * @param {*} callback
   */
  mergeTags(pkgName, tags, callback) {
    this._updatePackage(pkgName, (data, cb) => {
      /* eslint guard-for-in: 0 */
      for (const tag in tags) {
        // this handle dist-tag rm command
        if (_lodash.default.isNull(tags[tag])) {
          delete data[_constants.DIST_TAGS][tag];
          continue;
        }
        if (_lodash.default.isNil(data.versions[tags[tag]])) {
          return cb(this._getVersionNotFound());
        }
        const version = tags[tag];
        (0, _utils2.tagVersion)(data, version, tag);
      }
      cb(null);
    }, callback);
  }

  /**
   * Return version not found
   * @return {String}
   * @private
   */
  _getVersionNotFound() {
    return _utils2.ErrorCode.getNotFound(_constants.API_ERROR.VERSION_NOT_EXIST);
  }

  /**
   * Return file no available
   * @return {String}
   * @private
   */
  _getFileNotAvailable() {
    return _utils2.ErrorCode.getNotFound('no such file available');
  }

  /**
   * Update the package metadata, tags and attachments (tarballs).
   * Note: Currently supports unpublishing and deprecation.
   * @param {*} name
   * @param {*} incomingPkg
   * @param {*} revision
   * @param {*} callback
   * @return {Function}
   */
  changePackage(name, incomingPkg, revision, callback) {
    if (!(0, _utils2.isObject)(incomingPkg.versions) || !(0, _utils2.isObject)(incomingPkg[_constants.DIST_TAGS])) {
      this.logger.error({
        name
      }, `changePackage bad data for @{name}`);
      return callback(_utils2.ErrorCode.getBadData());
    }
    debug('changePackage udapting package for %o', name);
    this._updatePackage(name, (localData, cb) => {
      for (const version in localData.versions) {
        const incomingVersion = incomingPkg.versions[version];
        if (_lodash.default.isNil(incomingVersion)) {
          this.logger.info({
            name: name,
            version: version
          }, 'unpublishing @{name}@@{version}');

          // FIXME: I prefer return a new object rather mutate the metadata
          delete localData.versions[version];
          delete localData.time[version];
          for (const file in localData._attachments) {
            if (localData._attachments[file].version === version) {
              delete localData._attachments[file].version;
            }
          }
        } else if (Object.prototype.hasOwnProperty.call(incomingVersion, 'deprecated')) {
          const incomingDeprecated = incomingVersion.deprecated;
          if (incomingDeprecated != localData.versions[version].deprecated) {
            if (!incomingDeprecated) {
              this.logger.info({
                name: name,
                version: version
              }, 'undeprecating @{name}@@{version}');
              delete localData.versions[version].deprecated;
            } else {
              this.logger.info({
                name: name,
                version: version
              }, 'deprecating @{name}@@{version}');
              localData.versions[version].deprecated = incomingDeprecated;
            }
            localData.time.modified = new Date().toISOString();
          }
        }
      }
      localData[_constants.USERS] = incomingPkg[_constants.USERS];
      localData[_constants.DIST_TAGS] = incomingPkg[_constants.DIST_TAGS];
      cb(null);
    }, function (err) {
      if (err) {
        return callback(err);
      }
      callback();
    });
  }
  /**
   * Remove a tarball.
   * @param {*} name
   * @param {*} filename
   * @param {*} revision
   * @param {*} callback
   */
  removeTarball(name, filename, revision, callback) {
    (0, _assert.default)((0, _utils.validateName)(filename));
    this._updatePackage(name, (data, cb) => {
      if (data._attachments[filename]) {
        delete data._attachments[filename];
        cb(null);
      } else {
        cb(this._getFileNotAvailable());
      }
    }, err => {
      if (err) {
        return callback(err);
      }
      const storage = this._getLocalStorage(name);
      if (storage) {
        storage.deletePackage(filename, callback);
      }
    });
  }

  /**
   * Add a tarball.
   * @param {String} name
   * @param {String} filename
   * @return {Stream}
   */
  addTarball(name, filename) {
    (0, _assert.default)((0, _utils.validateName)(filename));
    let length = 0;
    const shaOneHash = (0, _utils.createTarballHash)();
    const uploadStream = new _streams.UploadTarball({});
    const _transform = uploadStream._transform;
    const storage = this._getLocalStorage(name);
    uploadStream.abort = function () {};
    uploadStream.done = function () {};
    uploadStream._transform = function (data, ...args) {
      shaOneHash.update(data);
      // measure the length for validation reasons
      length += data.length;
      const appliedData = [data, ...args];
      // FIXME: not sure about this approach, tsc complains
      // @ts-ignore
      _transform.apply(uploadStream, appliedData);
    };
    if (name === '__proto__') {
      process.nextTick(() => {
        uploadStream.emit('error', _utils2.ErrorCode.getForbidden());
      });
      return uploadStream;
    }
    if (!storage) {
      process.nextTick(() => {
        uploadStream.emit('error', "can't upload this package");
      });
      return uploadStream;
    }
    const writeStream = storage.writeTarball(filename);
    writeStream.on('error', err => {
      // @ts-ignore
      if (err.code === _constants.STORAGE.FILE_EXIST_ERROR || err.code === _constants.HTTP_STATUS.CONFLICT) {
        uploadStream.emit('error', _utils2.ErrorCode.getConflict());
        uploadStream.abort();
        // @ts-ignore
      } else if (err.code === _constants.STORAGE.NO_SUCH_FILE_ERROR || err.code === _constants.HTTP_STATUS.NOT_FOUND) {
        // check if package exists to throw an appropriate message
        this.getPackageMetadata(name, function (_err, _res) {
          if (_err) {
            uploadStream.emit('error', _err);
          } else {
            uploadStream.emit('error', err);
          }
        });
      } else {
        uploadStream.emit('error', err);
      }
    });
    writeStream.on('open', function () {
      // re-emitting open because it's handled in storage.js
      uploadStream.emit('open');
    });
    writeStream.on('success', () => {
      this._updatePackage(name, function updater(data, cb) {
        data._attachments[filename] = {
          shasum: shaOneHash.digest('hex')
        };
        cb(null);
      }, function (err) {
        if (err) {
          uploadStream.emit('error', err);
        } else {
          uploadStream.emit('success');
        }
      });
    });
    uploadStream.abort = function () {
      writeStream.abort();
    };
    uploadStream.done = function () {
      if (!length) {
        uploadStream.emit('error', _utils2.ErrorCode.getBadData('refusing to accept zero-length file'));
        writeStream.abort();
      } else {
        writeStream.done();
      }
    };
    uploadStream.pipe(writeStream);
    return uploadStream;
  }

  /**
   * Get a tarball.
   * @param {*} name
   * @param {*} filename
   * @return {ReadTarball}
   */
  getTarball(name, filename) {
    (0, _assert.default)((0, _utils.validateName)(filename));
    const storage = this._getLocalStorage(name);
    if (_lodash.default.isNil(storage)) {
      return this._createFailureStreamResponse();
    }
    return this._streamSuccessReadTarBall(storage, filename);
  }

  /**
   * Return a stream that emits a read failure.
   * @private
   * @return {ReadTarball}
   */
  _createFailureStreamResponse() {
    const stream = new _streams.ReadTarball({});
    process.nextTick(() => {
      stream.emit('error', this._getFileNotAvailable());
    });
    return stream;
  }

  /**
   * Return a stream that emits the tarball data
   * @param {Object} storage
   * @param {String} filename
   * @private
   * @return {ReadTarball}
   */
  _streamSuccessReadTarBall(storage, filename) {
    const stream = new _streams.ReadTarball({});
    const readTarballStream = storage.readTarball(filename);
    const e404 = _utils2.ErrorCode.getNotFound;
    stream.abort = function () {
      if (_lodash.default.isNil(readTarballStream) === false) {
        readTarballStream.abort();
      }
    };
    readTarballStream.on('error', function (err) {
      // @ts-ignore
      if (err.code === _constants.STORAGE.NO_SUCH_FILE_ERROR || err.code === _constants.HTTP_STATUS.NOT_FOUND) {
        stream.emit('error', e404('no such file available'));
      } else {
        stream.emit('error', err);
      }
    });
    readTarballStream.on('content-length', function (content) {
      stream.emit('content-length', content);
    });
    readTarballStream.on('open', function () {
      // re-emitting open because it's handled in storage.js
      stream.emit('open');
      readTarballStream.pipe(stream);
    });
    return stream;
  }

  /**
   * Retrieve a package by name.
   * @param {*} name
   * @param {*} callback
   * @return {Function}
   */
  getPackageMetadata(name, callback = () => {}) {
    const storage = this._getLocalStorage(name);
    if (_lodash.default.isNil(storage)) {
      return callback(_utils2.ErrorCode.getNotFound());
    }
    this._readPackage(name, storage, callback);
  }

  /**
   * Search a local package.
   * @param {*} startKey
   * @param {*} options
   * @return {Function}
   */
  search(startKey, options) {
    const stream = new _streams.ReadTarball({
      objectMode: true
    });
    this._searchEachPackage((item, cb) => {
      // @ts-ignore
      if (item.time > parseInt(startKey, 10)) {
        this.getPackageMetadata(item.name, (err, data) => {
          if (err) {
            return cb(err);
          }

          // @ts-ignore
          const time = new Date(item.time).toISOString();
          const result = (0, _storageUtils.prepareSearchPackage)(data, time);
          if (_lodash.default.isNil(result) === false) {
            stream.push(result);
          }
          cb(null);
        });
      } else {
        cb(null);
      }
    }, function onEnd(err) {
      if (err) {
        stream.emit('error', err);
        return;
      }
      stream.end();
    });
    return stream;
  }

  /**
   * Retrieve a wrapper that provide access to the package location.
   * @param {Object} pkgName package name.
   * @return {Object}
   */
  _getLocalStorage(pkgName) {
    return this.storagePlugin.getPackageStorage(pkgName);
  }

  /**
   * Read a json file from storage.
   * @param {Object} storage
   * @param {Function} callback
   */
  _readPackage(name, storage, callback) {
    storage.readPackage(name, (err, result) => {
      if (err) {
        if (err.code === _constants.STORAGE.NO_SUCH_FILE_ERROR || err.code === _constants.HTTP_STATUS.NOT_FOUND) {
          return callback(_utils2.ErrorCode.getNotFound());
        }
        return callback(this._internalError(err, _constants.STORAGE.PACKAGE_FILE_NAME, 'error reading'));
      }
      callback(err, (0, _storageUtils.normalizePackage)(result));
    });
  }

  /**
   * Walks through each package and calls `on_package` on them.
   * @param {*} onPackage
   * @param {*} onEnd
   */
  _searchEachPackage(onPackage, onEnd) {
    // save wait whether plugin still do not support search functionality
    if (_lodash.default.isNil(this.storagePlugin.search)) {
      this.logger.warn('plugin search not implemented yet');
      onEnd();
    } else {
      this.storagePlugin.search(onPackage, onEnd, _utils.validateName);
    }
  }

  /**
   * Retrieve either a previous created local package or a boilerplate.
   * @param {*} pkgName
   * @param {*} callback
   * @return {Function}
   */
  _readCreatePackage(pkgName, callback) {
    const storage = this._getLocalStorage(pkgName);
    if (_lodash.default.isNil(storage)) {
      this._createNewPackage(pkgName, callback);
      return;
    }
    storage.readPackage(pkgName, (err, data) => {
      // TODO: race condition
      if (_lodash.default.isNil(err) === false) {
        if (err.code === _constants.STORAGE.NO_SUCH_FILE_ERROR || err.code === _constants.HTTP_STATUS.NOT_FOUND) {
          data = (0, _storageUtils.generatePackageTemplate)(pkgName);
        } else {
          return callback(this._internalError(err, _constants.STORAGE.PACKAGE_FILE_NAME, 'error reading'));
        }
      }
      callback(null, (0, _storageUtils.normalizePackage)(data));
    });
  }
  _createNewPackage(name, callback) {
    return callback(null, (0, _storageUtils.normalizePackage)((0, _storageUtils.generatePackageTemplate)(name)));
  }

  /**
   * Handle internal error
   * @param {*} err
   * @param {*} file
   * @param {*} message
   * @return {Object} Error instance
   */
  _internalError(err, file, message) {
    this.logger.error({
      err: err,
      file: file
    }, `${message}  @{file}: @{!err.message}`);
    return _utils2.ErrorCode.getInternalError();
  }

  /**
   * @param {*} name package name
   * @param {*} updateHandler function(package, cb) - update function
   * @param {*} callback callback that gets invoked after it's all updated
   * @return {Function}
   */
  _updatePackage(name, updateHandler, callback) {
    const storage = this._getLocalStorage(name);
    if (!storage) {
      return callback(_utils2.ErrorCode.getNotFound());
    }
    storage.updatePackage(name, updateHandler, this._writePackage.bind(this), _storageUtils.normalizePackage, callback);
  }

  /**
   * Update the revision (_rev) string for a package.
   * @param {*} name
   * @param {*} json
   * @param {*} callback
   * @return {Function}
   */
  _writePackage(name, json, callback) {
    const storage = this._getLocalStorage(name);
    if (_lodash.default.isNil(storage)) {
      return callback();
    }
    storage.savePackage(name, this._setDefaultRevision(json), callback);
  }
  _setDefaultRevision(json) {
    // calculate revision from couch db
    if (_lodash.default.isString(json._rev) === false) {
      json._rev = _constants.STORAGE.DEFAULT_REVISION;
    }

    // this is intended in debug mode we do not want modify the store revision
    if (_lodash.default.isNil(this.config._debug)) {
      json._rev = (0, _storageUtils.generateRevision)(json._rev);
    }
    return json;
  }
  _deleteAttachments(storage, attachments, callback) {
    debug('[storage/_deleteAttachments] delete attachments total: %o', attachments === null || attachments === void 0 ? void 0 : attachments.length);
    const unlinkNext = function (cb) {
      if (_lodash.default.isEmpty(attachments)) {
        return cb();
      }
      const attachment = attachments.shift();
      storage.deletePackage(attachment, function () {
        unlinkNext(cb);
      });
    };
    unlinkNext(function () {
      // try to unlink the directory, but ignore errors because it can fail
      storage.removePackage(function (err) {
        callback(err);
      });
    });
  }

  /**
   * Ensure the dist file remains as the same protocol
   * @param {Object} hash metadata
   * @param {String} upLinkKey registry key
   * @private
   */
  _updateUplinkToRemoteProtocol(hash, upLinkKey) {
    // if we got this information from a known registry,
    // use the same protocol for the tarball
    //
    // see https://github.com/rlidwka/sinopia/issues/166
    const tarballUrl = _url.default.parse(hash.url);
    const uplinkUrl = _url.default.parse(this.config.uplinks[upLinkKey].url);
    if (uplinkUrl.host === tarballUrl.host) {
      tarballUrl.protocol = uplinkUrl.protocol;
      hash.registry = upLinkKey;
      hash.url = _url.default.format(tarballUrl);
    }
  }
  async getSecret(config) {
    const secretKey = await this.storagePlugin.getSecret();
    return this.storagePlugin.setSecret(config.checkSecretKey(secretKey));
  }
  _loadStorage(config, logger) {
    const Storage = this._loadStorePlugin();
    if (_lodash.default.isNil(Storage)) {
      (0, _assert.default)(this.config.storage, 'CONFIG: storage path not defined');
      return new _localStorage.default(this.config, logger);
    }
    return Storage;
  }
  _loadStorePlugin() {
    const plugin_params = {
      config: this.config,
      logger: this.logger
    };

    // eslint-disable-next-line max-len
    const plugins = (0, _pluginLoader.default)(this.config, this.config.store, plugin_params, plugin => {
      return plugin.getPackageStorage;
    });
    return _lodash.default.head(plugins);
  }
  saveToken(token) {
    if (_lodash.default.isFunction(this.storagePlugin.saveToken) === false) {
      return Promise.reject(_utils2.ErrorCode.getCode(_constants.HTTP_STATUS.SERVICE_UNAVAILABLE, _constants.SUPPORT_ERRORS.PLUGIN_MISSING_INTERFACE));
    }
    return this.storagePlugin.saveToken(token);
  }
  deleteToken(user, tokenKey) {
    if (_lodash.default.isFunction(this.storagePlugin.deleteToken) === false) {
      return Promise.reject(_utils2.ErrorCode.getCode(_constants.HTTP_STATUS.SERVICE_UNAVAILABLE, _constants.SUPPORT_ERRORS.PLUGIN_MISSING_INTERFACE));
    }
    return this.storagePlugin.deleteToken(user, tokenKey);
  }
  readTokens(filter) {
    if (_lodash.default.isFunction(this.storagePlugin.readTokens) === false) {
      return Promise.reject(_utils2.ErrorCode.getCode(_constants.HTTP_STATUS.SERVICE_UNAVAILABLE, _constants.SUPPORT_ERRORS.PLUGIN_MISSING_INTERFACE));
    }
    return this.storagePlugin.readTokens(filter);
  }
}
var _default = LocalStorage;
exports.default = _default;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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