"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;
var _debug = _interopRequireDefault(require("debug"));
var _fs = _interopRequireDefault(require("fs"));
var _lodash = _interopRequireDefault(require("lodash"));
var _mkdirp = _interopRequireDefault(require("mkdirp"));
var _path = _interopRequireDefault(require("path"));
var _utils = require("./utils");
function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }
const debug = (0, _debug.default)('verdaccio:config');
const CONFIG_FILE = 'config.yaml';
const XDG = 'xdg';
const WIN = 'win';
const WIN32 = 'win32';
// eslint-disable-next-line
const pkgJSON = require('../../package.json');
/**
 * Find and get the first config file that match.
 * @return {String} the config file path
 */
function findConfigFile(configPath) {
  if (typeof configPath !== 'undefined') {
    return _path.default.resolve(configPath);
  }
  const configPaths = getConfigPaths();
  debug('%o posible locations found', configPaths.length);
  if (_lodash.default.isEmpty(configPaths)) {
    throw new Error('no configuration files can be processed');
  }
  const primaryConf = _lodash.default.find(configPaths, configLocation => (0, _utils.fileExists)(configLocation.path));
  if (typeof primaryConf !== 'undefined') {
    debug('previous location exist already %s', primaryConf === null || primaryConf === void 0 ? void 0 : primaryConf.path);
    return primaryConf.path;
  }
  return createConfigFile(_lodash.default.head(configPaths)).path;
}
function createConfigFile(configLocation) {
  createConfigFolder(configLocation);
  const defaultConfig = updateStorageLinks(configLocation, readDefaultConfig());
  _fs.default.writeFileSync(configLocation.path, defaultConfig);
  return configLocation;
}
function readDefaultConfig() {
  return _fs.default.readFileSync(require.resolve('../../conf/default.yaml'), 'utf-8');
}
function createConfigFolder(configLocation) {
  _mkdirp.default.sync(_path.default.dirname(configLocation.path));
}
function updateStorageLinks(configLocation, defaultConfig) {
  if (configLocation.type !== XDG) {
    return defaultConfig;
  }

  // $XDG_DATA_HOME defines the base directory relative to which user specific data files should be stored,
  // If $XDG_DATA_HOME is either not set or empty, a default equal to $HOME/.local/share should be used.
  let dataDir = process.env.XDG_DATA_HOME || _path.default.join(process.env.HOME, '.local', 'share');
  if ((0, _utils.folderExists)(dataDir)) {
    dataDir = _path.default.resolve(_path.default.join(dataDir, pkgJSON.name, 'storage'));
    return defaultConfig.replace(/^storage: .\/storage$/m, `storage: ${dataDir}`);
  }
  return defaultConfig;
}
function getConfigPaths() {
  const listPaths = [getXDGDirectory(), getWindowsDirectory(), getRelativeDefaultDirectory(), getOldDirectory()].reduce(function (acc, currentValue) {
    if (_lodash.default.isUndefined(currentValue) === false) {
      acc.push(currentValue);
    }
    return acc;
  }, []);
  return listPaths;
}
const getXDGDirectory = () => {
  const XDGConfig = getXDGHome() || process.env.HOME && _path.default.join(process.env.HOME, '.config');
  if (XDGConfig && (0, _utils.folderExists)(XDGConfig)) {
    return {
      path: _path.default.join(XDGConfig, pkgJSON.name, CONFIG_FILE),
      type: XDG
    };
  }
};
const getXDGHome = () => process.env.XDG_CONFIG_HOME;
const getWindowsDirectory = () => {
  if (process.platform === WIN32 && process.env.APPDATA && (0, _utils.folderExists)(process.env.APPDATA)) {
    return {
      path: _path.default.resolve(_path.default.join(process.env.APPDATA, pkgJSON.name, CONFIG_FILE)),
      type: WIN
    };
  }
};
const getRelativeDefaultDirectory = () => {
  return {
    path: _path.default.resolve(_path.default.join('.', pkgJSON.name, CONFIG_FILE)),
    type: 'def'
  };
};
const getOldDirectory = () => {
  return {
    path: _path.default.resolve(_path.default.join('.', CONFIG_FILE)),
    type: 'old'
  };
};
var _default = findConfigFile;
exports.default = _default;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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