"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.listenDefaultCallback = listenDefaultCallback;
exports.startVerdaccio = startVerdaccio;
var _constants = _interopRequireDefault(require("constants"));
var _fs = _interopRequireDefault(require("fs"));
var _http = _interopRequireDefault(require("http"));
var _https = _interopRequireDefault(require("https"));
var _lodash = require("lodash");
var _url = _interopRequireDefault(require("url"));
var _index = _interopRequireDefault(require("../api/index"));
var _utils = require("./cli/utils");
var _constants2 = require("./constants");
function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }
function ownKeys(object, enumerableOnly) { var keys = Object.keys(object); if (Object.getOwnPropertySymbols) { var symbols = Object.getOwnPropertySymbols(object); enumerableOnly && (symbols = symbols.filter(function (sym) { return Object.getOwnPropertyDescriptor(object, sym).enumerable; })), keys.push.apply(keys, symbols); } return keys; }
function _objectSpread(target) { for (var i = 1; i < arguments.length; i++) { var source = null != arguments[i] ? arguments[i] : {}; i % 2 ? ownKeys(Object(source), !0).forEach(function (key) { _defineProperty(target, key, source[key]); }) : Object.getOwnPropertyDescriptors ? Object.defineProperties(target, Object.getOwnPropertyDescriptors(source)) : ownKeys(Object(source)).forEach(function (key) { Object.defineProperty(target, key, Object.getOwnPropertyDescriptor(source, key)); }); } return target; }
function _defineProperty(obj, key, value) { key = _toPropertyKey(key); if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }
function _toPropertyKey(arg) { var key = _toPrimitive(arg, "string"); return typeof key === "symbol" ? key : String(key); }
function _toPrimitive(input, hint) { if (typeof input !== "object" || input === null) return input; var prim = input[Symbol.toPrimitive]; if (prim !== undefined) { var res = prim.call(input, hint || "default"); if (typeof res !== "object") return res; throw new TypeError("@@toPrimitive must return a primitive value."); } return (hint === "string" ? String : Number)(input); }
const logger = require('./logger');
function displayExperimentsInfoBox(experiments) {
  const experimentList = Object.keys(experiments);
  if (experimentList.length >= 1) {
    logger.logger.warn('⚠️  experiments are enabled, it is recommended do not use experiments in production, comment out the experiments section to disable this warning');
    experimentList.forEach(experiment => {
      logger.logger.warn(` - support for ${experiment} ${experiments[experiment] ? 'is enabled' : ' is disabled'}`);
    });
  }
}

/**
 * Trigger the server after configuration has been loaded.
 * @param {Object} config
 * @param {Object} cliArguments
 * @param {String} configPath
 * @param {String} pkgVersion
 * @param {String} pkgName
 * @deprecated use runServer instead
 */
function startVerdaccio(config, cliListen, configPath, pkgVersion, pkgName, callback) {
  if ((0, _lodash.isObject)(config) === false) {
    throw new Error(_constants2.API_ERROR.CONFIG_BAD_FORMAT);
  }
  if ('experiments' in config) {
    displayExperimentsInfoBox(config.experiments);
  }
  (0, _index.default)(config).then(app => {
    const addresses = (0, _utils.getListListenAddresses)(cliListen, config.listen);
    if (addresses.length > 1) {
      process.emitWarning('multiple listen addresses are deprecated, please use only one');
    }
    addresses.forEach(function (addr) {
      let webServer;
      if (addr.proto === 'https') {
        webServer = handleHTTPS(app, configPath, config);
      } else {
        // http
        webServer = _http.default.createServer(app);
      }
      if (config.server && typeof config.server.keepAliveTimeout !== 'undefined' && config.server.keepAliveTimeout !== 'null') {
        // library definition for node is not up to date (doesn't contain recent 8.0 changes)
        webServer.keepAliveTimeout = config.server.keepAliveTimeout * 1000;
      }
      unlinkAddressPath(addr);
      callback(webServer, addr, pkgName, pkgVersion);
    });
  });
}
function unlinkAddressPath(addr) {
  if (addr.path && _fs.default.existsSync(addr.path)) {
    _fs.default.unlinkSync(addr.path);
  }
}
function logHTTPSWarning(storageLocation) {
  logger.logger.fatal(['You have enabled HTTPS and need to specify either ', '    "https.key" and "https.cert" or ', '    "https.pfx" and optionally "https.passphrase" ', 'to run https server', '',
  // commands are borrowed from node.js docs
  'To quickly create self-signed certificate, use:', ' $ openssl genrsa -out ' + (0, _utils.resolveConfigPath)(storageLocation, _constants2.keyPem) + ' 2048', ' $ openssl req -new -sha256 -key ' + (0, _utils.resolveConfigPath)(storageLocation, _constants2.keyPem) + ' -out ' + (0, _utils.resolveConfigPath)(storageLocation, _constants2.csrPem), ' $ openssl x509 -req -in ' + (0, _utils.resolveConfigPath)(storageLocation, _constants2.csrPem) + ' -signkey ' + (0, _utils.resolveConfigPath)(storageLocation, _constants2.keyPem) + ' -out ' + (0, _utils.resolveConfigPath)(storageLocation, _constants2.certPem), '', 'And then add to config file (' + storageLocation + '):', '  https:', `    key: ${(0, _utils.resolveConfigPath)(storageLocation, _constants2.keyPem)}`, `    cert: ${(0, _utils.resolveConfigPath)(storageLocation, _constants2.certPem)}`].join('\n'));
  process.exit(2);
}
function handleHTTPS(app, configPath, config) {
  try {
    let httpsOptions = {
      secureOptions: _constants.default.SSL_OP_NO_SSLv2 | _constants.default.SSL_OP_NO_SSLv3 // disable insecure SSLv2 and SSLv3
    };

    const keyCertConfig = config.https;
    const pfxConfig = config.https;

    // https must either have key and cert or a pfx and (optionally) a passphrase
    if (!(keyCertConfig.key && keyCertConfig.cert || pfxConfig.pfx)) {
      logHTTPSWarning(configPath);
    }
    if (pfxConfig.pfx) {
      const {
        pfx,
        passphrase
      } = pfxConfig;
      httpsOptions = (0, _lodash.assign)(httpsOptions, {
        pfx: _fs.default.readFileSync(pfx),
        passphrase: passphrase || ''
      });
    } else {
      const {
        key,
        cert,
        ca
      } = keyCertConfig;
      httpsOptions = (0, _lodash.assign)(httpsOptions, _objectSpread({
        key: _fs.default.readFileSync(key),
        cert: _fs.default.readFileSync(cert)
      }, ca && {
        ca: _fs.default.readFileSync(ca)
      }));
    }
    return _https.default.createServer(httpsOptions, app);
  } catch (err) {
    // catch errors related to certificate loading
    logger.logger.fatal({
      err: err
    }, 'cannot create server: @{err.message}');
    process.exit(2);
  }
}
/**
 *
 * @param webServer
 * @param addr
 * @param pkgName
 * @param pkgVersion
 * @deprecated use initServer instead
 */
function listenDefaultCallback(webServer, addr, pkgName, pkgVersion) {
  const server = webServer.listen(addr.port || addr.path, addr.host, () => {
    // send a message for tests
    if ((0, _lodash.isFunction)(process.send)) {
      process.send({
        verdaccio_started: true
      });
    }
  }).on('error', function (err) {
    logger.logger.fatal({
      err: err
    }, 'cannot create server: @{err.message}');
    process.exit(2);
  });
  function handleShutdownGracefully() {
    logger.logger.fatal('received shutdown signal - closing server gracefully...');
    server.close(() => {
      logger.logger.info('server closed.');
      process.exit(0);
    });
  }

  // handle shutdown signals nicely when environment says so
  if (process.env.VERDACCIO_HANDLE_KILL_SIGNALS === 'true') {
    process.on('SIGINT', handleShutdownGracefully);
    process.on('SIGTERM', handleShutdownGracefully);
    process.on('SIGHUP', handleShutdownGracefully);
  }
  logger.logger.warn({
    addr: addr.path ? _url.default.format({
      protocol: 'unix',
      pathname: addr.path
    }) : _url.default.format({
      protocol: addr.proto,
      hostname: addr.host,
      port: addr.port,
      pathname: '/'
    }),
    version: pkgName + '/' + pkgVersion
  }, 'http address - @{addr} - @{version}');
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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