"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;
var _debug = _interopRequireDefault(require("debug"));
var _lodash = _interopRequireDefault(require("lodash"));
var _config = require("@verdaccio/config");
var _signature = require("@verdaccio/signature");
var _utils = require("@verdaccio/utils");
var _pluginLoader = _interopRequireDefault(require("../lib/plugin-loader"));
var _authUtils = require("./auth-utils");
var _constants = require("./constants");
var _logger = require("./logger");
var _utils2 = require("./utils");
function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }
function _defineProperty(obj, key, value) { key = _toPropertyKey(key); if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }
function _toPropertyKey(arg) { var key = _toPrimitive(arg, "string"); return typeof key === "symbol" ? key : String(key); }
function _toPrimitive(input, hint) { if (typeof input !== "object" || input === null) return input; var prim = input[Symbol.toPrimitive]; if (prim !== undefined) { var res = prim.call(input, hint || "default"); if (typeof res !== "object") return res; throw new TypeError("@@toPrimitive must return a primitive value."); } return (hint === "string" ? String : Number)(input); }
const debug = (0, _debug.default)('verdaccio:auth');
class Auth {
  // pragma: allowlist secret

  constructor(config) {
    _defineProperty(this, "config", void 0);
    _defineProperty(this, "logger", void 0);
    _defineProperty(this, "secret", void 0);
    _defineProperty(this, "plugins", void 0);
    this.config = config;
    this.logger = _logger.logger;
    this.secret = config.secret;
    this.plugins = this._loadPlugin(config);
    this._applyDefaultPlugins();
  }
  _loadPlugin(config) {
    const pluginOptions = {
      config,
      logger: this.logger
    };
    return (0, _pluginLoader.default)(config, config.auth, pluginOptions, plugin => {
      const {
        authenticate,
        allow_access,
        allow_publish
      } = plugin;
      // @ts-ignore
      return authenticate || allow_access || allow_publish;
    });
  }
  _applyDefaultPlugins() {
    this.plugins.push((0, _authUtils.getDefaultPlugins)(this.logger));
  }
  changePassword(username, password,
  // pragma: allowlist secret
  newPassword,
  // pragma: allowlist secret
  cb) {
    const validPlugins = _lodash.default.filter(this.plugins, plugin => _lodash.default.isFunction(plugin.changePassword));
    if (_lodash.default.isEmpty(validPlugins)) {
      return cb(_utils2.ErrorCode.getInternalError(_constants.SUPPORT_ERRORS.PLUGIN_MISSING_INTERFACE));
    }
    for (const plugin of validPlugins) {
      if (_lodash.default.isNil(plugin) || _lodash.default.isFunction(plugin.changePassword) === false) {
        debug('auth plugin does not implement changePassword, trying next one');
        continue;
      } else {
        debug('updating password for %o', username);
        plugin.changePassword(username, password, newPassword, (err, profile) => {
          if (err) {
            this.logger.error({
              username,
              err
            }, `An error has been produced
            updating the password for @{username}. Error: @{err.message}`);
            return cb(err);
          }
          this.logger.info({
            username
          }, 'updated password for @{username} was successful');
          return cb(null, profile);
        });
      }
    }
  }
  authenticate(username, password, cb) {
    const plugins = this.plugins.slice(0);
    const self = this;
    (function next() {
      const plugin = plugins.shift();
      if (_lodash.default.isFunction(plugin.authenticate) === false) {
        return next();
      }
      debug('authenticating %o', username);
      plugin.authenticate(username, password, function (err, groups) {
        if (err) {
          self.logger.error({
            username,
            err
          }, 'authenticating for user @{username} failed. Error: @{err.message}');
          return cb(err);
        }

        // Expect: SKIP if groups is falsey and not an array
        //         with at least one item (truthy length)
        // Expect: CONTINUE otherwise (will error if groups is not
        //         an array, but this is current behavior)
        // Caveat: STRING (if valid) will pass successfully
        //         bug give unexpected results
        // Info: Cannot use `== false to check falsey values`
        if (!!groups && groups.length !== 0) {
          // TODO: create a better understanding of expectations
          if (_lodash.default.isString(groups)) {
            throw new TypeError('plugin group error: invalid type for function');
          }
          const isGroupValid = _lodash.default.isArray(groups);
          if (!isGroupValid) {
            throw new TypeError(_constants.API_ERROR.BAD_FORMAT_USER_GROUP);
          }
          debug('authentication for user %o was successfully. Groups: %o', username, groups);
          return cb(err, (0, _config.createRemoteUser)(username, groups));
        }
        next();
      });
    })();
  }
  add_user(user, password, cb) {
    const self = this;
    const plugins = this.plugins.slice(0);
    debug('add user %o', user);
    (function next() {
      const plugin = plugins.shift();
      let method = 'adduser';
      if (_lodash.default.isFunction(plugin[method]) === false) {
        method = 'add_user';
        self.logger.warn('the plugin method add_user in the auth plugin is deprecated and will be removed in next major release, notify to the plugin author');
      }
      if (_lodash.default.isFunction(plugin[method]) === false) {
        next();
      } else {
        // p.add_user() execution
        plugin[method](user, password, function (err, ok) {
          if (err) {
            self.logger.error({
              user,
              err: err.message
            }, 'the user @{user} could not being added. Error: @{err}');
            return cb(err);
          }
          if (ok) {
            self.logger.info({
              user
            }, 'the user @{user} has been added');
            return self.authenticate(user, password, cb);
          }
          next();
        });
      }
    })();
  }

  /**
   * Allow user to access a package.
   */
  allow_access({
    packageName,
    packageVersion
  }, user, callback) {
    const plugins = this.plugins.slice(0);
    const self = this;
    const pkgAllowAcces = {
      name: packageName,
      version: packageVersion
    };
    const pkg = Object.assign({}, pkgAllowAcces, (0, _utils.getMatchedPackagesSpec)(packageName, this.config.packages));
    debug('allow access for %o', packageName);
    (function next() {
      const plugin = plugins.shift();
      if (_lodash.default.isNil(plugin) || _lodash.default.isFunction(plugin.allow_access) === false) {
        return next();
      }
      plugin.allow_access(user, pkg, function (err, ok) {
        if (err) {
          self.logger.error({
            packageName,
            err
          }, 'forbidden access for @{packageName}. Error: @{err.message}');
          return callback(err);
        }
        if (ok) {
          self.logger.info({
            packageName
          }, 'allowed access for @{packageName}');
          return callback(null, ok);
        }
        next(); // cb(null, false) causes next plugin to roll
      });
    })();
  }

  allow_unpublish({
    packageName,
    packageVersion
  }, user, callback) {
    const pkg = Object.assign({
      name: packageName,
      version: packageVersion
    }, (0, _utils.getMatchedPackagesSpec)(packageName, this.config.packages));
    debug('allow unpublish for %o', packageName);
    for (const plugin of this.plugins) {
      if (_lodash.default.isNil(plugin) || _lodash.default.isFunction(plugin.allow_unpublish) === false) {
        debug('allow unpublish for %o plugin does not implement allow_unpublish', packageName);
        continue;
      } else {
        plugin.allow_unpublish(user, pkg, (err, ok) => {
          if (err) {
            this.logger.error({
              packageName,
              user: user === null || user === void 0 ? void 0 : user.name
            }, '@{user} forbidden publish for @{packageName}, it will fallback on unpublish permissions');
            return callback(err);
          }
          if (_lodash.default.isNil(ok) === true) {
            debug('we bypass unpublish for %o, publish will handle the access', packageName);
            // @ts-ignore
            // eslint-disable-next-line
            return this.allow_publish(...arguments);
          }
          if (ok) {
            this.logger.info({
              packageName,
              user: user === null || user === void 0 ? void 0 : user.name
            }, '@{user} allowed unpublish for @{packageName}');
            return callback(null, ok);
          }
        });
      }
    }
  }

  /**
   * Allow user to publish a package.
   */
  allow_publish({
    packageName,
    packageVersion
  }, user, callback) {
    const plugins = this.plugins.slice(0);
    const self = this;
    const pkg = Object.assign({
      name: packageName,
      version: packageVersion
    }, (0, _utils.getMatchedPackagesSpec)(packageName, this.config.packages));
    debug('allow publish for %o init | plugins: %o', packageName, plugins);
    (function next() {
      const plugin = plugins.shift();
      if (_lodash.default.isNil(plugin) || _lodash.default.isFunction(plugin.allow_publish) === false) {
        debug('allow publish for %o plugin does not implement allow_publish', packageName);
        return next();
      }

      // @ts-ignore
      plugin.allow_publish(user, pkg, (err, ok) => {
        if (_lodash.default.isNil(err) === false && _lodash.default.isError(err)) {
          self.logger.error({
            packageName,
            user: user === null || user === void 0 ? void 0 : user.name
          }, '@{user} is forbidden publish for @{packageName}');
          return callback(err);
        }
        if (ok) {
          self.logger.info({
            packageName,
            user: user === null || user === void 0 ? void 0 : user.name
          }, '@{user} is allowed publish for @{packageName}');
          return callback(null, ok);
        }
        debug('allow publish skip validation for %o', packageName);
        next(); // cb(null, false) causes next plugin to roll
      });
    })();
  }

  apiJWTmiddleware() {
    const plugins = this.plugins.slice(0);
    const helpers = {
      createAnonymousRemoteUser: _config.createAnonymousRemoteUser,
      createRemoteUser: _config.createRemoteUser
    };
    for (const plugin of plugins) {
      if (plugin.apiJWTmiddleware) {
        return plugin.apiJWTmiddleware(helpers);
      }
    }
    return (req, res, _next) => {
      req.pause();
      const next = function (err) {
        req.resume();
        // uncomment this to reject users with bad auth headers
        // return _next.apply(null, arguments)
        // swallow error, user remains unauthorized
        // set remoteUserError to indicate that user was attempting authentication
        if (err) {
          req.remote_user.error = err.message;
        }
        return _next();
      };
      if (this._isRemoteUserValid(req.remote_user)) {
        // @ts-ignore
        return next();
      }

      // in case auth header does not exist we return anonymous function
      req.remote_user = (0, _config.createAnonymousRemoteUser)();
      const {
        authorization
      } = req.headers;
      if (_lodash.default.isNil(authorization)) {
        // @ts-ignore
        return next();
      }
      if (!(0, _authUtils.isAuthHeaderValid)(authorization)) {
        debug('api middleware auth heather is not valid');
        return next(_utils2.ErrorCode.getBadRequest(_constants.API_ERROR.BAD_AUTH_HEADER));
      }
      const security = (0, _authUtils.getSecurity)(this.config);
      const {
        secret
      } = this.config;
      if ((0, _authUtils.isAESLegacy)(security)) {
        debug('api middleware using legacy auth token');
        this._handleAESMiddleware(req, security, secret, authorization, next);
      } else {
        debug('api middleware using JWT auth token');
        this._handleJWTAPIMiddleware(req, security, secret, authorization, next);
      }
    };
  }
  _handleJWTAPIMiddleware(req, security, secret, authorization, next) {
    const {
      scheme,
      token
    } = (0, _authUtils.parseAuthTokenHeader)(authorization);
    if (scheme.toUpperCase() === _constants.TOKEN_BASIC.toUpperCase()) {
      // this should happen when client tries to login with an existing user
      const credentials = (0, _utils2.convertPayloadToBase64)(token).toString();
      const {
        user,
        password
      } = (0, _authUtils.parseBasicPayload)(credentials);
      this.authenticate(user, password, (err, user) => {
        if (!err) {
          req.remote_user = user;
          next();
        } else {
          req.remote_user = (0, _config.createAnonymousRemoteUser)();
          next(err);
        }
      });
    } else {
      // jwt handler
      const credentials = (0, _authUtils.getMiddlewareCredentials)(security, secret, authorization);
      if (credentials) {
        // if the signature is valid we rely on it
        req.remote_user = credentials;
        next();
      } else {
        // with JWT throw 401
        next(_utils2.ErrorCode.getForbidden(_constants.API_ERROR.BAD_USERNAME_PASSWORD));
      }
    }
  }
  _handleAESMiddleware(req, security, secret, authorization, next) {
    const credentials = (0, _authUtils.getMiddlewareCredentials)(security, secret, authorization);
    if (credentials) {
      const {
        user,
        password
      } = credentials;
      this.authenticate(user, password, (err, user) => {
        if (!err) {
          req.remote_user = user;
          next();
        } else {
          req.remote_user = (0, _config.createAnonymousRemoteUser)();
          next(err);
        }
      });
    } else {
      // we force npm client to ask again with basic authentication
      return next(_utils2.ErrorCode.getBadRequest(_constants.API_ERROR.BAD_AUTH_HEADER));
    }
  }
  _isRemoteUserValid(remote_user) {
    return _lodash.default.isUndefined(remote_user) === false && _lodash.default.isUndefined(remote_user.name) === false;
  }

  /**
   * JWT middleware for WebUI
   */
  webUIJWTmiddleware() {
    return (req, res, _next) => {
      if (this._isRemoteUserValid(req.remote_user)) {
        return _next();
      }
      req.pause();
      const next = err => {
        req.resume();
        if (err) {
          // req.remote_user.error = err.message;
          res.status(err.statusCode).send(err.message);
        }
        return _next();
      };
      const {
        authorization
      } = req.headers;
      if (_lodash.default.isNil(authorization)) {
        // @ts-ignore
        return next();
      }
      if (!(0, _authUtils.isAuthHeaderValid)(authorization)) {
        return next(_utils2.ErrorCode.getBadRequest(_constants.API_ERROR.BAD_AUTH_HEADER));
      }
      const token = (authorization || '').replace(`${_constants.TOKEN_BEARER} `, '');
      if (!token) {
        // @ts-ignore
        return next();
      }
      let credentials;
      try {
        credentials = (0, _authUtils.verifyJWTPayload)(token, this.config.secret);
      } catch (err) {
        // FIXME: intended behaviour, do we want it?
      }
      if (this._isRemoteUserValid(credentials)) {
        const {
          name,
          groups
        } = credentials;
        req.remote_user = (0, _config.createRemoteUser)(name, groups);
      } else {
        req.remote_user = (0, _config.createAnonymousRemoteUser)();
      }
      // @ts-ignore
      next();
    };
  }
  async jwtEncrypt(user, signOptions) {
    const {
      real_groups,
      name,
      groups
    } = user;
    const realGroupsValidated = _lodash.default.isNil(real_groups) ? [] : real_groups;
    const groupedGroups = _lodash.default.isNil(groups) ? real_groups : Array.from(new Set([...groups.concat(realGroupsValidated)]));
    const payload = {
      real_groups: realGroupsValidated,
      name,
      groups: groupedGroups
    };
    const token = await (0, _signature.signPayload)(payload, this.secret, signOptions);
    return token;
  }

  /**
   * Encrypt a string.
   */
  aesEncrypt(buf) {
    return (0, _signature.aesEncryptDeprecated)(buf, this.secret);
  }
}
var _default = Auth;
exports.default = _default;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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