"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.allow_action = allow_action;
exports.expireReasons = exports.defaultSecurity = void 0;
exports.getApiToken = getApiToken;
exports.getDefaultPlugins = getDefaultPlugins;
exports.getMiddlewareCredentials = getMiddlewareCredentials;
exports.getSecurity = getSecurity;
exports.handleSpecialUnpublish = handleSpecialUnpublish;
exports.isAESLegacy = isAESLegacy;
exports.isAuthHeaderValid = isAuthHeaderValid;
exports.parseAESCredentials = parseAESCredentials;
exports.parseAuthTokenHeader = parseAuthTokenHeader;
exports.parseBasicPayload = parseBasicPayload;
exports.validatePassword = validatePassword;
exports.verifyJWTPayload = verifyJWTPayload;
var _debug = _interopRequireDefault(require("debug"));
var _lodash = _interopRequireDefault(require("lodash"));
var _config = require("@verdaccio/config");
var _signature = require("@verdaccio/signature");
var _utils = require("@verdaccio/utils");
var _constants = require("./constants");
var _logger = require("./logger");
var _utils2 = require("./utils");
function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }
const debug = (0, _debug.default)('verdaccio');
function validatePassword(password,
// pragma: allowlist secret
minLength = _constants.DEFAULT_MIN_LIMIT_PASSWORD) {
  return typeof password === 'string' && password.length >= minLength;
}
function allow_action(action) {
  return function (user, pkg, callback) {
    debug('[auth/allow_action]: user: %o', user === null || user === void 0 ? void 0 : user.name);
    const {
      name,
      groups
    } = user;
    const groupAccess = pkg[action];
    const hasPermission = groupAccess.some(group => name === group || groups.includes(group));
    debug('[auth/allow_action]: hasPermission? %o} for user: %o', hasPermission, user === null || user === void 0 ? void 0 : user.name);
    if (hasPermission) {
      _logger.logger.info({
        remote: user.name
      }, `auth/allow_action: access granted to: @{user}`);
      return callback(null, true);
    }
    if (name) {
      callback(_utils2.ErrorCode.getForbidden(`user ${name} is not allowed to ${action} package ${pkg.name}`));
    } else {
      callback(_utils2.ErrorCode.getUnauthorized(`authorization required to ${action} package ${pkg.name}`));
    }
  };
}

/**
 *
 */
function handleSpecialUnpublish() {
  return function (user, pkg, callback) {
    const action = 'unpublish';
    // verify whether the unpublish prop has been defined
    const isUnpublishMissing = _lodash.default.isNil(pkg[action]);
    const hasGroups = isUnpublishMissing ? false : pkg[action].length > 0;
    debug('fallback unpublish for @{name} has groups: %o for %o', hasGroups, user === null || user === void 0 ? void 0 : user.name);
    if (isUnpublishMissing || hasGroups === false) {
      return callback(null, undefined);
    }
    debug('allow_action for %o for %o has groups: %o for %o', action, user === null || user === void 0 ? void 0 : user.name, hasGroups, user);
    return allow_action(action)(user, pkg, callback);
  };
}
function getDefaultPlugins(logger) {
  return {
    authenticate(_user, _password, cb) {
      // pragma: allowlist secret
      cb(_utils2.ErrorCode.getForbidden(_constants.API_ERROR.BAD_USERNAME_PASSWORD));
    },
    add_user(_user, _password, cb) {
      // pragma: allowlist secret
      return cb(_utils2.ErrorCode.getConflict(_constants.API_ERROR.BAD_USERNAME_PASSWORD));
    },
    // FIXME: allow_action and allow_publish should be in the @verdaccio/types
    // @ts-ignore
    allow_access: allow_action('access', logger),
    // @ts-ignore
    allow_publish: allow_action('publish', logger),
    allow_unpublish: handleSpecialUnpublish()
  };
}
const defaultWebTokenOptions = {
  sign: {
    // The expiration token for the website is 1 hour
    expiresIn: _constants.TIME_EXPIRATION_1H
  },
  verify: {}
};
const defaultApiTokenConf = {
  legacy: true
};
const defaultSecurity = {
  web: defaultWebTokenOptions,
  api: defaultApiTokenConf
};
exports.defaultSecurity = defaultSecurity;
function getSecurity(config) {
  if (_lodash.default.isNil(config.security) === false) {
    return _lodash.default.merge(defaultSecurity, config.security);
  }
  return defaultSecurity;
}
function isAESLegacy(security) {
  const {
    legacy,
    jwt
  } = security.api;
  return _lodash.default.isNil(legacy) === false && _lodash.default.isNil(jwt) && legacy === true;
}
async function getApiToken(auth, config, remoteUser, aesPassword) {
  const security = getSecurity(config);
  if (isAESLegacy(security)) {
    // fallback all goes to AES encryption
    return await new Promise(resolve => {
      resolve(auth.aesEncrypt((0, _utils.buildUserBuffer)(remoteUser.name, aesPassword)).toString('base64'));
    });
  }
  // i am wiling to use here _.isNil but flow does not like it yet.
  const {
    jwt
  } = security.api;
  if (jwt && jwt.sign) {
    return await auth.jwtEncrypt(remoteUser, jwt.sign);
  }
  return await new Promise(resolve => {
    resolve(auth.aesEncrypt((0, _utils.buildUserBuffer)(remoteUser.name, aesPassword)).toString('base64'));
  });
}
function parseAuthTokenHeader(authorizationHeader) {
  const parts = authorizationHeader.split(' ');
  const [scheme, token] = parts;
  return {
    scheme,
    token
  };
}
function parseBasicPayload(credentials) {
  const index = credentials.indexOf(':');
  if (index < 0) {
    return;
  }
  const user = credentials.slice(0, index);
  const password = credentials.slice(index + 1);
  return {
    user,
    password
  };
}
function parseAESCredentials(authorizationHeader, secret) {
  const {
    scheme,
    token
  } = parseAuthTokenHeader(authorizationHeader);

  // basic is deprecated and should not be enforced
  if (scheme.toUpperCase() === _constants.TOKEN_BASIC.toUpperCase()) {
    const credentials = (0, _utils2.convertPayloadToBase64)(token).toString();
    return credentials;
  } else if (scheme.toUpperCase() === _constants.TOKEN_BEARER.toUpperCase()) {
    const tokenAsBuffer = (0, _utils2.convertPayloadToBase64)(token);
    const credentials = (0, _signature.aesDecryptDeprecated)(tokenAsBuffer, secret).toString('utf8');
    return credentials;
  }
}
const expireReasons = ['JsonWebTokenError', 'TokenExpiredError'];
exports.expireReasons = expireReasons;
function verifyJWTPayload(token, secret) {
  try {
    const payload = (0, _signature.verifyPayload)(token, secret);
    return payload;
  } catch (error) {
    // #168 this check should be removed as soon AES encrypt is removed.
    if (expireReasons.includes(error.name)) {
      // it might be possible the jwt configuration is enabled and
      // old tokens fails still remains in usage, thus
      // we return an anonymous user to force log in.
      return (0, _config.createAnonymousRemoteUser)();
    }
    throw _utils2.ErrorCode.getCode(_constants.HTTP_STATUS.UNAUTHORIZED, error.message);
  }
}
function isAuthHeaderValid(authorization) {
  return authorization.split(' ').length === 2;
}
function getMiddlewareCredentials(security, secret, authorizationHeader) {
  if (isAESLegacy(security)) {
    const credentials = parseAESCredentials(authorizationHeader, secret);
    if (!credentials) {
      return;
    }
    const parsedCredentials = parseBasicPayload(credentials);
    if (!parsedCredentials) {
      return;
    }
    return parsedCredentials;
  }
  const {
    scheme,
    token
  } = parseAuthTokenHeader(authorizationHeader);
  if (_lodash.default.isString(token) && scheme.toUpperCase() === _constants.TOKEN_BEARER.toUpperCase()) {
    return verifyJWTPayload(token, secret);
  }
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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