"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.errorReportingMiddleware = errorReportingMiddleware;
exports.handleError = handleError;
exports.serveFavicon = serveFavicon;
var _debug = _interopRequireDefault(require("debug"));
var _fs = _interopRequireDefault(require("fs"));
var _lodash = _interopRequireDefault(require("lodash"));
var _path = _interopRequireDefault(require("path"));
var _validator = _interopRequireDefault(require("validator"));
var _constants = require("../lib/constants");
var _logger = require("../lib/logger");
function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }
const debug = (0, _debug.default)('verdaccio');
function serveFavicon(config) {
  return function (req, res, next) {
    try {
      var _config$web;
      // @ts-ignore
      const logoConf = config === null || config === void 0 ? void 0 : (_config$web = config.web) === null || _config$web === void 0 ? void 0 : _config$web.favicon;
      if (logoConf === '') {
        debug('favicon disabled');
        res.status(404);
      } else if (!_lodash.default.isEmpty(logoConf)) {
        debug('custom favicon');
        if (_validator.default.isURL(logoConf, {
          require_host: true,
          require_valid_protocol: true
        })) {
          debug('redirect to %o', logoConf);
          res.redirect(logoConf);
          return;
        } else {
          const faviconPath = _path.default.normalize(logoConf);
          debug('serving favicon from %o', faviconPath);
          _fs.default.access(faviconPath, _fs.default.constants.R_OK, err => {
            if (err) {
              debug('no read permissions to read: %o, reason:', logoConf, err === null || err === void 0 ? void 0 : err.message);
              return res.status(_constants.HTTP_STATUS.NOT_FOUND).end();
            } else {
              res.setHeader('content-type', 'image/x-icon');
              _fs.default.createReadStream(faviconPath).pipe(res);
              debug('rendered custom ico');
            }
          });
        }
      } else {
        res.setHeader('content-type', 'image/x-icon');
        _fs.default.createReadStream(_path.default.posix.join(__dirname, './web/html/favicon.ico')).pipe(res);
        debug('rendered ico');
      }
    } catch (err) {
      debug('error triggered, favicon not found');
      res.status(_constants.HTTP_STATUS.NOT_FOUND).end();
    }
  };
}
function handleError(err, req, res, next) {
  debug('error handler init');
  if (_lodash.default.isError(err)) {
    debug('is native error');
    if (err.code === 'ECONNABORT' && res.statusCode === _constants.HTTP_STATUS.NOT_MODIFIED) {
      return next();
    }
    if (_lodash.default.isFunction(res.locals.report_error) === false) {
      debug('is locals error report ref');
      // in case of very early error this middleware may not be loaded before error is generated
      // fixing that
      errorReportingMiddleware(req, res, _lodash.default.noop);
    }
    debug('set locals error report ref');
    res.locals.report_error(err);
  } else {
    // Fall to Middleware.final
    debug('no error to report, jump next layer');
    return next(err);
  }
}
// Middleware
function errorReportingMiddleware(req, res, next) {
  res.locals.report_error = res.locals.report_error || function (err) {
    if (err.status && err.status >= _constants.HTTP_STATUS.BAD_REQUEST && err.status < 600) {
      if (!res.headersSent) {
        res.status(err.status);
        next({
          error: err.message || _constants.API_ERROR.UNKNOWN_ERROR
        });
      }
    } else {
      _logger.logger.error({
        err: err
      }, 'unexpected error: @{!err.message}\n@{err.stack}');
      if (!res.status || !res.send) {
        _logger.logger.error('this is an error in express.js, please report this');
        res.destroy();
      } else if (!res.headersSent) {
        res.status(_constants.HTTP_STATUS.INTERNAL_ERROR);
        next({
          error: _constants.API_ERROR.INTERNAL_SERVER_ERROR
        });
      } else {
        // socket should be already closed
      }
    }
  };
  next();
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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