"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;
exports.loadTheme = loadTheme;
var _compression = _interopRequireDefault(require("compression"));
var _cors = _interopRequireDefault(require("cors"));
var _express = _interopRequireDefault(require("express"));
var _lodash = _interopRequireDefault(require("lodash"));
var _config = require("@verdaccio/config");
var _middleware = require("@verdaccio/middleware");
var _auth = _interopRequireDefault(require("../lib/auth"));
var _config2 = _interopRequireDefault(require("../lib/config"));
var _constants = require("../lib/constants");
var _logger = require("../lib/logger");
var _pluginLoader = _interopRequireDefault(require("../lib/plugin-loader"));
var _storage = _interopRequireDefault(require("../lib/storage"));
var _utils = require("../lib/utils");
var _debug = _interopRequireDefault(require("./debug"));
var _endpoint = _interopRequireDefault(require("./endpoint"));
var _middleware2 = require("./middleware");
var _web = _interopRequireDefault(require("./web"));
function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }
function loadTheme(config) {
  if (_lodash.default.isNil(config.theme) === false) {
    return _lodash.default.head((0, _pluginLoader.default)(config, config.theme, {}, function (plugin) {
      return plugin.staticPath && plugin.manifest && plugin.manifestFiles;
    }, 'verdaccio-theme'));
  }
}
const defineAPI = function (config, storage) {
  var _config$server;
  const auth = new _auth.default(config);
  const app = (0, _express.default)();

  // run in production mode by default, just in case
  // it shouldn't make any difference anyway
  app.set('env', process.env.NODE_ENV || 'production');

  // Allow `req.ip` to resolve properly when Verdaccio is behind a proxy or load-balancer
  // See: https://expressjs.com/en/guide/behind-proxies.html
  if ((_config$server = config.server) !== null && _config$server !== void 0 && _config$server.trustProxy) {
    app.set('trust proxy', config.server.trustProxy);
  }
  app.use((0, _cors.default)());

  // Router setup
  app.use((0, _middleware.log)(_logger.logger));
  app.use(_middleware2.errorReportingMiddleware);
  if (config.user_agent) {
    app.use(function (req, res, next) {
      res.setHeader('X-Powered-By', (0, _config.getUserAgent)(config.user_agent));
      next();
    });
  } else {
    app.disable('x-powered-by');
  }
  app.use((0, _compression.default)());
  app.get('/-/static/favicon.ico', (0, _middleware2.serveFavicon)(config));

  // Hook for tests only
  if (config._debug) {
    (0, _debug.default)(app, config.self_path);
  }

  // register middleware plugins
  const plugin_params = {
    config: config,
    logger: _logger.logger
  };
  const plugins = (0, _pluginLoader.default)(config, config.middlewares, plugin_params, function (plugin) {
    return plugin.register_middlewares;
  });
  plugins.forEach(plugin => {
    plugin.register_middlewares(app, auth, storage);
  });

  // For  npm request
  app.use((0, _endpoint.default)(config, auth, storage));

  // For WebUI & WebUI API
  if (_lodash.default.get(config, 'web.enable', true)) {
    app.use('/', (0, _web.default)(config, auth, storage));
  } else {
    app.get('/', function (req, res, next) {
      next(_utils.ErrorCode.getNotFound(_constants.API_ERROR.WEB_DISABLED));
    });
  }

  // Catch 404
  app.get('/*', function (req, res, next) {
    next(_utils.ErrorCode.getNotFound(_constants.API_ERROR.FILE_NOT_FOUND));
  });
  app.use(_middleware2.handleError);
  app.use(_middleware.final);
  return app;
};
var _default = async function _default(configHash) {
  (0, _logger.setup)(configHash.logs);
  const config = new _config2.default(_lodash.default.cloneDeep(configHash));
  // register middleware plugins
  const plugin_params = {
    config: config,
    logger: _logger.logger
  };
  const filters = (0, _pluginLoader.default)(config, config.filters || {}, plugin_params, plugin => plugin.filter_metadata);
  const storage = new _storage.default(config);
  // waits until init calls have been initialized
  await storage.init(config, filters);
  return defineAPI(config, storage);
};
exports.default = _default;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJuYW1lcyI6WyJfY29tcHJlc3Npb24iLCJfaW50ZXJvcFJlcXVpcmVEZWZhdWx0IiwicmVxdWlyZSIsIl9jb3JzIiwiX2V4cHJlc3MiLCJfbG9kYXNoIiwiX2NvbmZpZyIsIl9taWRkbGV3YXJlIiwiX2F1dGgiLCJfY29uZmlnMiIsIl9jb25zdGFudHMiLCJfbG9nZ2VyIiwiX3BsdWdpbkxvYWRlciIsIl9zdG9yYWdlIiwiX3V0aWxzIiwiX2RlYnVnIiwiX2VuZHBvaW50IiwiX21pZGRsZXdhcmUyIiwiX3dlYiIsIm9iaiIsIl9fZXNNb2R1bGUiLCJkZWZhdWx0IiwibG9hZFRoZW1lIiwiY29uZmlnIiwiXyIsImlzTmlsIiwidGhlbWUiLCJoZWFkIiwibG9hZFBsdWdpbiIsInBsdWdpbiIsInN0YXRpY1BhdGgiLCJtYW5pZmVzdCIsIm1hbmlmZXN0RmlsZXMiLCJkZWZpbmVBUEkiLCJzdG9yYWdlIiwiX2NvbmZpZyRzZXJ2ZXIiLCJhdXRoIiwiQXV0aCIsImFwcCIsImV4cHJlc3MiLCJzZXQiLCJwcm9jZXNzIiwiZW52IiwiTk9ERV9FTlYiLCJzZXJ2ZXIiLCJ0cnVzdFByb3h5IiwidXNlIiwiY29ycyIsImxvZyIsImxvZ2dlciIsImVycm9yUmVwb3J0aW5nTWlkZGxld2FyZSIsInVzZXJfYWdlbnQiLCJyZXEiLCJyZXMiLCJuZXh0Iiwic2V0SGVhZGVyIiwiZ2V0VXNlckFnZW50IiwiZGlzYWJsZSIsImNvbXByZXNzaW9uIiwiZ2V0Iiwic2VydmVGYXZpY29uIiwiaG9va0RlYnVnIiwic2VsZl9wYXRoIiwicGx1Z2luX3BhcmFtcyIsInBsdWdpbnMiLCJtaWRkbGV3YXJlcyIsInJlZ2lzdGVyX21pZGRsZXdhcmVzIiwiZm9yRWFjaCIsImFwaUVuZHBvaW50Iiwid2ViTWlkZGxld2FyZSIsIkVycm9yQ29kZSIsImdldE5vdEZvdW5kIiwiQVBJX0VSUk9SIiwiV0VCX0RJU0FCTEVEIiwiRklMRV9OT1RfRk9VTkQiLCJoYW5kbGVFcnJvciIsImZpbmFsIiwiX2RlZmF1bHQiLCJjb25maWdIYXNoIiwic2V0dXAiLCJsb2dzIiwiQXBwQ29uZmlnIiwiY2xvbmVEZWVwIiwiZmlsdGVycyIsImZpbHRlcl9tZXRhZGF0YSIsIlN0b3JhZ2UiLCJpbml0IiwiZXhwb3J0cyJdLCJzb3VyY2VzIjpbIi4uLy4uL3NyYy9hcGkvaW5kZXgudHMiXSwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IGNvbXByZXNzaW9uIGZyb20gJ2NvbXByZXNzaW9uJztcbmltcG9ydCBjb3JzIGZyb20gJ2NvcnMnO1xuaW1wb3J0IGV4cHJlc3MsIHsgQXBwbGljYXRpb24gfSBmcm9tICdleHByZXNzJztcbmltcG9ydCBfIGZyb20gJ2xvZGFzaCc7XG5cbmltcG9ydCB7IGdldFVzZXJBZ2VudCB9IGZyb20gJ0B2ZXJkYWNjaW8vY29uZmlnJztcbmltcG9ydCB7IGZpbmFsIH0gZnJvbSAnQHZlcmRhY2Npby9taWRkbGV3YXJlJztcbmltcG9ydCB7IGxvZyB9IGZyb20gJ0B2ZXJkYWNjaW8vbWlkZGxld2FyZSc7XG5pbXBvcnQgeyBDb25maWcgYXMgSUNvbmZpZywgSVBsdWdpbk1pZGRsZXdhcmUsIElQbHVnaW5TdG9yYWdlRmlsdGVyIH0gZnJvbSAnQHZlcmRhY2Npby90eXBlcyc7XG5cbmltcG9ydCBBdXRoIGZyb20gJy4uL2xpYi9hdXRoJztcbmltcG9ydCBBcHBDb25maWcgZnJvbSAnLi4vbGliL2NvbmZpZyc7XG5pbXBvcnQgeyBBUElfRVJST1IgfSBmcm9tICcuLi9saWIvY29uc3RhbnRzJztcbmltcG9ydCB7IGxvZ2dlciwgc2V0dXAgfSBmcm9tICcuLi9saWIvbG9nZ2VyJztcbmltcG9ydCBsb2FkUGx1Z2luIGZyb20gJy4uL2xpYi9wbHVnaW4tbG9hZGVyJztcbmltcG9ydCBTdG9yYWdlIGZyb20gJy4uL2xpYi9zdG9yYWdlJztcbmltcG9ydCB7IEVycm9yQ29kZSB9IGZyb20gJy4uL2xpYi91dGlscyc7XG5pbXBvcnQge1xuICAkTmV4dEZ1bmN0aW9uVmVyLFxuICAkUmVxdWVzdEV4dGVuZCxcbiAgJFJlc3BvbnNlRXh0ZW5kLFxuICBJQXV0aCxcbiAgSVN0b3JhZ2VIYW5kbGVyLFxufSBmcm9tICcuLi90eXBlcyc7XG5pbXBvcnQgaG9va0RlYnVnIGZyb20gJy4vZGVidWcnO1xuaW1wb3J0IGFwaUVuZHBvaW50IGZyb20gJy4vZW5kcG9pbnQnO1xuaW1wb3J0IHsgZXJyb3JSZXBvcnRpbmdNaWRkbGV3YXJlLCBoYW5kbGVFcnJvciwgc2VydmVGYXZpY29uIH0gZnJvbSAnLi9taWRkbGV3YXJlJztcbmltcG9ydCB3ZWJNaWRkbGV3YXJlIGZyb20gJy4vd2ViJztcblxuZXhwb3J0IGZ1bmN0aW9uIGxvYWRUaGVtZShjb25maWcpIHtcbiAgaWYgKF8uaXNOaWwoY29uZmlnLnRoZW1lKSA9PT0gZmFsc2UpIHtcbiAgICByZXR1cm4gXy5oZWFkKFxuICAgICAgbG9hZFBsdWdpbihcbiAgICAgICAgY29uZmlnLFxuICAgICAgICBjb25maWcudGhlbWUsXG4gICAgICAgIHt9LFxuICAgICAgICBmdW5jdGlvbiAocGx1Z2luKSB7XG4gICAgICAgICAgcmV0dXJuIHBsdWdpbi5zdGF0aWNQYXRoICYmIHBsdWdpbi5tYW5pZmVzdCAmJiBwbHVnaW4ubWFuaWZlc3RGaWxlcztcbiAgICAgICAgfSxcbiAgICAgICAgJ3ZlcmRhY2Npby10aGVtZSdcbiAgICAgIClcbiAgICApO1xuICB9XG59XG5cbmNvbnN0IGRlZmluZUFQSSA9IGZ1bmN0aW9uIChjb25maWc6IElDb25maWcsIHN0b3JhZ2U6IElTdG9yYWdlSGFuZGxlcik6IGFueSB7XG4gIGNvbnN0IGF1dGg6IElBdXRoID0gbmV3IEF1dGgoY29uZmlnKTtcbiAgY29uc3QgYXBwOiBBcHBsaWNhdGlvbiA9IGV4cHJlc3MoKTtcblxuICAvLyBydW4gaW4gcHJvZHVjdGlvbiBtb2RlIGJ5IGRlZmF1bHQsIGp1c3QgaW4gY2FzZVxuICAvLyBpdCBzaG91bGRuJ3QgbWFrZSBhbnkgZGlmZmVyZW5jZSBhbnl3YXlcbiAgYXBwLnNldCgnZW52JywgcHJvY2Vzcy5lbnYuTk9ERV9FTlYgfHwgJ3Byb2R1Y3Rpb24nKTtcblxuICAvLyBBbGxvdyBgcmVxLmlwYCB0byByZXNvbHZlIHByb3Blcmx5IHdoZW4gVmVyZGFjY2lvIGlzIGJlaGluZCBhIHByb3h5IG9yIGxvYWQtYmFsYW5jZXJcbiAgLy8gU2VlOiBodHRwczovL2V4cHJlc3Nqcy5jb20vZW4vZ3VpZGUvYmVoaW5kLXByb3hpZXMuaHRtbFxuICBpZiAoY29uZmlnLnNlcnZlcj8udHJ1c3RQcm94eSkge1xuICAgIGFwcC5zZXQoJ3RydXN0IHByb3h5JywgY29uZmlnLnNlcnZlci50cnVzdFByb3h5KTtcbiAgfVxuXG4gIGFwcC51c2UoY29ycygpKTtcblxuICAvLyBSb3V0ZXIgc2V0dXBcbiAgYXBwLnVzZShsb2cobG9nZ2VyKSk7XG4gIGFwcC51c2UoZXJyb3JSZXBvcnRpbmdNaWRkbGV3YXJlKTtcbiAgaWYgKGNvbmZpZy51c2VyX2FnZW50KSB7XG4gICAgYXBwLnVzZShmdW5jdGlvbiAocmVxOiAkUmVxdWVzdEV4dGVuZCwgcmVzOiAkUmVzcG9uc2VFeHRlbmQsIG5leHQ6ICROZXh0RnVuY3Rpb25WZXIpOiB2b2lkIHtcbiAgICAgIHJlcy5zZXRIZWFkZXIoJ1gtUG93ZXJlZC1CeScsIGdldFVzZXJBZ2VudChjb25maWcudXNlcl9hZ2VudCkpO1xuICAgICAgbmV4dCgpO1xuICAgIH0pO1xuICB9IGVsc2Uge1xuICAgIGFwcC5kaXNhYmxlKCd4LXBvd2VyZWQtYnknKTtcbiAgfVxuXG4gIGFwcC51c2UoY29tcHJlc3Npb24oKSk7XG5cbiAgYXBwLmdldCgnLy0vc3RhdGljL2Zhdmljb24uaWNvJywgc2VydmVGYXZpY29uKGNvbmZpZykpO1xuXG4gIC8vIEhvb2sgZm9yIHRlc3RzIG9ubHlcbiAgaWYgKGNvbmZpZy5fZGVidWcpIHtcbiAgICBob29rRGVidWcoYXBwLCBjb25maWcuc2VsZl9wYXRoKTtcbiAgfVxuXG4gIC8vIHJlZ2lzdGVyIG1pZGRsZXdhcmUgcGx1Z2luc1xuICBjb25zdCBwbHVnaW5fcGFyYW1zID0ge1xuICAgIGNvbmZpZzogY29uZmlnLFxuICAgIGxvZ2dlcjogbG9nZ2VyLFxuICB9O1xuXG4gIGNvbnN0IHBsdWdpbnM6IElQbHVnaW5NaWRkbGV3YXJlPElDb25maWc+W10gPSBsb2FkUGx1Z2luKFxuICAgIGNvbmZpZyxcbiAgICBjb25maWcubWlkZGxld2FyZXMsXG4gICAgcGx1Z2luX3BhcmFtcyxcbiAgICBmdW5jdGlvbiAocGx1Z2luOiBJUGx1Z2luTWlkZGxld2FyZTxJQ29uZmlnPikge1xuICAgICAgcmV0dXJuIHBsdWdpbi5yZWdpc3Rlcl9taWRkbGV3YXJlcztcbiAgICB9XG4gICk7XG4gIHBsdWdpbnMuZm9yRWFjaCgocGx1Z2luOiBJUGx1Z2luTWlkZGxld2FyZTxJQ29uZmlnPikgPT4ge1xuICAgIHBsdWdpbi5yZWdpc3Rlcl9taWRkbGV3YXJlcyhhcHAsIGF1dGgsIHN0b3JhZ2UpO1xuICB9KTtcblxuICAvLyBGb3IgIG5wbSByZXF1ZXN0XG4gIGFwcC51c2UoYXBpRW5kcG9pbnQoY29uZmlnLCBhdXRoLCBzdG9yYWdlKSk7XG5cbiAgLy8gRm9yIFdlYlVJICYgV2ViVUkgQVBJXG4gIGlmIChfLmdldChjb25maWcsICd3ZWIuZW5hYmxlJywgdHJ1ZSkpIHtcbiAgICBhcHAudXNlKCcvJywgd2ViTWlkZGxld2FyZShjb25maWcsIGF1dGgsIHN0b3JhZ2UpKTtcbiAgfSBlbHNlIHtcbiAgICBhcHAuZ2V0KCcvJywgZnVuY3Rpb24gKHJlcTogJFJlcXVlc3RFeHRlbmQsIHJlczogJFJlc3BvbnNlRXh0ZW5kLCBuZXh0OiAkTmV4dEZ1bmN0aW9uVmVyKSB7XG4gICAgICBuZXh0KEVycm9yQ29kZS5nZXROb3RGb3VuZChBUElfRVJST1IuV0VCX0RJU0FCTEVEKSk7XG4gICAgfSk7XG4gIH1cblxuICAvLyBDYXRjaCA0MDRcbiAgYXBwLmdldCgnLyonLCBmdW5jdGlvbiAocmVxOiAkUmVxdWVzdEV4dGVuZCwgcmVzOiAkUmVzcG9uc2VFeHRlbmQsIG5leHQ6ICROZXh0RnVuY3Rpb25WZXIpIHtcbiAgICBuZXh0KEVycm9yQ29kZS5nZXROb3RGb3VuZChBUElfRVJST1IuRklMRV9OT1RfRk9VTkQpKTtcbiAgfSk7XG4gIGFwcC51c2UoaGFuZGxlRXJyb3IpO1xuICBhcHAudXNlKGZpbmFsKTtcblxuICByZXR1cm4gYXBwO1xufTtcblxuZXhwb3J0IGRlZmF1bHQgKGFzeW5jIGZ1bmN0aW9uIChjb25maWdIYXNoOiBhbnkpOiBQcm9taXNlPGFueT4ge1xuICBzZXR1cChjb25maWdIYXNoLmxvZ3MpO1xuICBjb25zdCBjb25maWc6IElDb25maWcgPSBuZXcgQXBwQ29uZmlnKF8uY2xvbmVEZWVwKGNvbmZpZ0hhc2gpKTtcbiAgLy8gcmVnaXN0ZXIgbWlkZGxld2FyZSBwbHVnaW5zXG4gIGNvbnN0IHBsdWdpbl9wYXJhbXMgPSB7XG4gICAgY29uZmlnOiBjb25maWcsXG4gICAgbG9nZ2VyOiBsb2dnZXIsXG4gIH07XG4gIGNvbnN0IGZpbHRlcnMgPSBsb2FkUGx1Z2luKFxuICAgIGNvbmZpZyxcbiAgICBjb25maWcuZmlsdGVycyB8fCB7fSxcbiAgICBwbHVnaW5fcGFyYW1zLFxuICAgIChwbHVnaW46IElQbHVnaW5TdG9yYWdlRmlsdGVyPElDb25maWc+KSA9PiBwbHVnaW4uZmlsdGVyX21ldGFkYXRhXG4gICk7XG4gIGNvbnN0IHN0b3JhZ2U6IElTdG9yYWdlSGFuZGxlciA9IG5ldyBTdG9yYWdlKGNvbmZpZyk7XG4gIC8vIHdhaXRzIHVudGlsIGluaXQgY2FsbHMgaGF2ZSBiZWVuIGluaXRpYWxpemVkXG4gIGF3YWl0IHN0b3JhZ2UuaW5pdChjb25maWcsIGZpbHRlcnMpO1xuICByZXR1cm4gZGVmaW5lQVBJKGNvbmZpZywgc3RvcmFnZSk7XG59KTtcbiJdLCJtYXBwaW5ncyI6Ijs7Ozs7OztBQUFBLElBQUFBLFlBQUEsR0FBQUMsc0JBQUEsQ0FBQUMsT0FBQTtBQUNBLElBQUFDLEtBQUEsR0FBQUYsc0JBQUEsQ0FBQUMsT0FBQTtBQUNBLElBQUFFLFFBQUEsR0FBQUgsc0JBQUEsQ0FBQUMsT0FBQTtBQUNBLElBQUFHLE9BQUEsR0FBQUosc0JBQUEsQ0FBQUMsT0FBQTtBQUVBLElBQUFJLE9BQUEsR0FBQUosT0FBQTtBQUNBLElBQUFLLFdBQUEsR0FBQUwsT0FBQTtBQUlBLElBQUFNLEtBQUEsR0FBQVAsc0JBQUEsQ0FBQUMsT0FBQTtBQUNBLElBQUFPLFFBQUEsR0FBQVIsc0JBQUEsQ0FBQUMsT0FBQTtBQUNBLElBQUFRLFVBQUEsR0FBQVIsT0FBQTtBQUNBLElBQUFTLE9BQUEsR0FBQVQsT0FBQTtBQUNBLElBQUFVLGFBQUEsR0FBQVgsc0JBQUEsQ0FBQUMsT0FBQTtBQUNBLElBQUFXLFFBQUEsR0FBQVosc0JBQUEsQ0FBQUMsT0FBQTtBQUNBLElBQUFZLE1BQUEsR0FBQVosT0FBQTtBQVFBLElBQUFhLE1BQUEsR0FBQWQsc0JBQUEsQ0FBQUMsT0FBQTtBQUNBLElBQUFjLFNBQUEsR0FBQWYsc0JBQUEsQ0FBQUMsT0FBQTtBQUNBLElBQUFlLFlBQUEsR0FBQWYsT0FBQTtBQUNBLElBQUFnQixJQUFBLEdBQUFqQixzQkFBQSxDQUFBQyxPQUFBO0FBQWtDLFNBQUFELHVCQUFBa0IsR0FBQSxXQUFBQSxHQUFBLElBQUFBLEdBQUEsQ0FBQUMsVUFBQSxHQUFBRCxHQUFBLEtBQUFFLE9BQUEsRUFBQUYsR0FBQTtBQUUzQixTQUFTRyxTQUFTQSxDQUFDQyxNQUFNLEVBQUU7RUFDaEMsSUFBSUMsZUFBQyxDQUFDQyxLQUFLLENBQUNGLE1BQU0sQ0FBQ0csS0FBSyxDQUFDLEtBQUssS0FBSyxFQUFFO0lBQ25DLE9BQU9GLGVBQUMsQ0FBQ0csSUFBSSxDQUNYLElBQUFDLHFCQUFVLEVBQ1JMLE1BQU0sRUFDTkEsTUFBTSxDQUFDRyxLQUFLLEVBQ1osQ0FBQyxDQUFDLEVBQ0YsVUFBVUcsTUFBTSxFQUFFO01BQ2hCLE9BQU9BLE1BQU0sQ0FBQ0MsVUFBVSxJQUFJRCxNQUFNLENBQUNFLFFBQVEsSUFBSUYsTUFBTSxDQUFDRyxhQUFhO0lBQ3JFLENBQUMsRUFDRCxpQkFBaUIsQ0FDbEIsQ0FDRjtFQUNIO0FBQ0Y7QUFFQSxNQUFNQyxTQUFTLEdBQUcsU0FBQUEsQ0FBVVYsTUFBZSxFQUFFVyxPQUF3QixFQUFPO0VBQUEsSUFBQUMsY0FBQTtFQUMxRSxNQUFNQyxJQUFXLEdBQUcsSUFBSUMsYUFBSSxDQUFDZCxNQUFNLENBQUM7RUFDcEMsTUFBTWUsR0FBZ0IsR0FBRyxJQUFBQyxnQkFBTyxHQUFFOztFQUVsQztFQUNBO0VBQ0FELEdBQUcsQ0FBQ0UsR0FBRyxDQUFDLEtBQUssRUFBRUMsT0FBTyxDQUFDQyxHQUFHLENBQUNDLFFBQVEsSUFBSSxZQUFZLENBQUM7O0VBRXBEO0VBQ0E7RUFDQSxLQUFBUixjQUFBLEdBQUlaLE1BQU0sQ0FBQ3FCLE1BQU0sY0FBQVQsY0FBQSxlQUFiQSxjQUFBLENBQWVVLFVBQVUsRUFBRTtJQUM3QlAsR0FBRyxDQUFDRSxHQUFHLENBQUMsYUFBYSxFQUFFakIsTUFBTSxDQUFDcUIsTUFBTSxDQUFDQyxVQUFVLENBQUM7RUFDbEQ7RUFFQVAsR0FBRyxDQUFDUSxHQUFHLENBQUMsSUFBQUMsYUFBSSxHQUFFLENBQUM7O0VBRWY7RUFDQVQsR0FBRyxDQUFDUSxHQUFHLENBQUMsSUFBQUUsZUFBRyxFQUFDQyxjQUFNLENBQUMsQ0FBQztFQUNwQlgsR0FBRyxDQUFDUSxHQUFHLENBQUNJLHFDQUF3QixDQUFDO0VBQ2pDLElBQUkzQixNQUFNLENBQUM0QixVQUFVLEVBQUU7SUFDckJiLEdBQUcsQ0FBQ1EsR0FBRyxDQUFDLFVBQVVNLEdBQW1CLEVBQUVDLEdBQW9CLEVBQUVDLElBQXNCLEVBQVE7TUFDekZELEdBQUcsQ0FBQ0UsU0FBUyxDQUFDLGNBQWMsRUFBRSxJQUFBQyxvQkFBWSxFQUFDakMsTUFBTSxDQUFDNEIsVUFBVSxDQUFDLENBQUM7TUFDOURHLElBQUksRUFBRTtJQUNSLENBQUMsQ0FBQztFQUNKLENBQUMsTUFBTTtJQUNMaEIsR0FBRyxDQUFDbUIsT0FBTyxDQUFDLGNBQWMsQ0FBQztFQUM3QjtFQUVBbkIsR0FBRyxDQUFDUSxHQUFHLENBQUMsSUFBQVksb0JBQVcsR0FBRSxDQUFDO0VBRXRCcEIsR0FBRyxDQUFDcUIsR0FBRyxDQUFDLHVCQUF1QixFQUFFLElBQUFDLHlCQUFZLEVBQUNyQyxNQUFNLENBQUMsQ0FBQzs7RUFFdEQ7RUFDQSxJQUFJQSxNQUFNLENBQUNSLE1BQU0sRUFBRTtJQUNqQixJQUFBOEMsY0FBUyxFQUFDdkIsR0FBRyxFQUFFZixNQUFNLENBQUN1QyxTQUFTLENBQUM7RUFDbEM7O0VBRUE7RUFDQSxNQUFNQyxhQUFhLEdBQUc7SUFDcEJ4QyxNQUFNLEVBQUVBLE1BQU07SUFDZDBCLE1BQU0sRUFBRUE7RUFDVixDQUFDO0VBRUQsTUFBTWUsT0FBcUMsR0FBRyxJQUFBcEMscUJBQVUsRUFDdERMLE1BQU0sRUFDTkEsTUFBTSxDQUFDMEMsV0FBVyxFQUNsQkYsYUFBYSxFQUNiLFVBQVVsQyxNQUFrQyxFQUFFO0lBQzVDLE9BQU9BLE1BQU0sQ0FBQ3FDLG9CQUFvQjtFQUNwQyxDQUFDLENBQ0Y7RUFDREYsT0FBTyxDQUFDRyxPQUFPLENBQUV0QyxNQUFrQyxJQUFLO0lBQ3REQSxNQUFNLENBQUNxQyxvQkFBb0IsQ0FBQzVCLEdBQUcsRUFBRUYsSUFBSSxFQUFFRixPQUFPLENBQUM7RUFDakQsQ0FBQyxDQUFDOztFQUVGO0VBQ0FJLEdBQUcsQ0FBQ1EsR0FBRyxDQUFDLElBQUFzQixpQkFBVyxFQUFDN0MsTUFBTSxFQUFFYSxJQUFJLEVBQUVGLE9BQU8sQ0FBQyxDQUFDOztFQUUzQztFQUNBLElBQUlWLGVBQUMsQ0FBQ21DLEdBQUcsQ0FBQ3BDLE1BQU0sRUFBRSxZQUFZLEVBQUUsSUFBSSxDQUFDLEVBQUU7SUFDckNlLEdBQUcsQ0FBQ1EsR0FBRyxDQUFDLEdBQUcsRUFBRSxJQUFBdUIsWUFBYSxFQUFDOUMsTUFBTSxFQUFFYSxJQUFJLEVBQUVGLE9BQU8sQ0FBQyxDQUFDO0VBQ3BELENBQUMsTUFBTTtJQUNMSSxHQUFHLENBQUNxQixHQUFHLENBQUMsR0FBRyxFQUFFLFVBQVVQLEdBQW1CLEVBQUVDLEdBQW9CLEVBQUVDLElBQXNCLEVBQUU7TUFDeEZBLElBQUksQ0FBQ2dCLGdCQUFTLENBQUNDLFdBQVcsQ0FBQ0Msb0JBQVMsQ0FBQ0MsWUFBWSxDQUFDLENBQUM7SUFDckQsQ0FBQyxDQUFDO0VBQ0o7O0VBRUE7RUFDQW5DLEdBQUcsQ0FBQ3FCLEdBQUcsQ0FBQyxJQUFJLEVBQUUsVUFBVVAsR0FBbUIsRUFBRUMsR0FBb0IsRUFBRUMsSUFBc0IsRUFBRTtJQUN6RkEsSUFBSSxDQUFDZ0IsZ0JBQVMsQ0FBQ0MsV0FBVyxDQUFDQyxvQkFBUyxDQUFDRSxjQUFjLENBQUMsQ0FBQztFQUN2RCxDQUFDLENBQUM7RUFDRnBDLEdBQUcsQ0FBQ1EsR0FBRyxDQUFDNkIsd0JBQVcsQ0FBQztFQUNwQnJDLEdBQUcsQ0FBQ1EsR0FBRyxDQUFDOEIsaUJBQUssQ0FBQztFQUVkLE9BQU90QyxHQUFHO0FBQ1osQ0FBQztBQUFDLElBQUF1QyxRQUFBLEdBRWMsZUFBQUEsU0FBZ0JDLFVBQWUsRUFBZ0I7RUFDN0QsSUFBQUMsYUFBSyxFQUFDRCxVQUFVLENBQUNFLElBQUksQ0FBQztFQUN0QixNQUFNekQsTUFBZSxHQUFHLElBQUkwRCxnQkFBUyxDQUFDekQsZUFBQyxDQUFDMEQsU0FBUyxDQUFDSixVQUFVLENBQUMsQ0FBQztFQUM5RDtFQUNBLE1BQU1mLGFBQWEsR0FBRztJQUNwQnhDLE1BQU0sRUFBRUEsTUFBTTtJQUNkMEIsTUFBTSxFQUFFQTtFQUNWLENBQUM7RUFDRCxNQUFNa0MsT0FBTyxHQUFHLElBQUF2RCxxQkFBVSxFQUN4QkwsTUFBTSxFQUNOQSxNQUFNLENBQUM0RCxPQUFPLElBQUksQ0FBQyxDQUFDLEVBQ3BCcEIsYUFBYSxFQUNabEMsTUFBcUMsSUFBS0EsTUFBTSxDQUFDdUQsZUFBZSxDQUNsRTtFQUNELE1BQU1sRCxPQUF3QixHQUFHLElBQUltRCxnQkFBTyxDQUFDOUQsTUFBTSxDQUFDO0VBQ3BEO0VBQ0EsTUFBTVcsT0FBTyxDQUFDb0QsSUFBSSxDQUFDL0QsTUFBTSxFQUFFNEQsT0FBTyxDQUFDO0VBQ25DLE9BQU9sRCxTQUFTLENBQUNWLE1BQU0sRUFBRVcsT0FBTyxDQUFDO0FBQ25DLENBQUM7QUFBQXFELE9BQUEsQ0FBQWxFLE9BQUEsR0FBQXdELFFBQUEifQ==