"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = _default;
var _debug = _interopRequireDefault(require("debug"));
var _express = require("express");
var _lodash = _interopRequireDefault(require("lodash"));
var _middleware = require("@verdaccio/middleware");
var _utils = require("@verdaccio/utils");
var _authUtils = require("../../../../lib/auth-utils");
var _constants = require("../../../../lib/constants");
var _logger = require("../../../../lib/logger");
var _utils2 = require("../../../../lib/utils");
function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }
function ownKeys(object, enumerableOnly) { var keys = Object.keys(object); if (Object.getOwnPropertySymbols) { var symbols = Object.getOwnPropertySymbols(object); enumerableOnly && (symbols = symbols.filter(function (sym) { return Object.getOwnPropertyDescriptor(object, sym).enumerable; })), keys.push.apply(keys, symbols); } return keys; }
function _objectSpread(target) { for (var i = 1; i < arguments.length; i++) { var source = null != arguments[i] ? arguments[i] : {}; i % 2 ? ownKeys(Object(source), !0).forEach(function (key) { _defineProperty(target, key, source[key]); }) : Object.getOwnPropertyDescriptors ? Object.defineProperties(target, Object.getOwnPropertyDescriptors(source)) : ownKeys(Object(source)).forEach(function (key) { Object.defineProperty(target, key, Object.getOwnPropertyDescriptor(source, key)); }); } return target; }
function _defineProperty(obj, key, value) { key = _toPropertyKey(key); if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }
function _toPropertyKey(arg) { var key = _toPrimitive(arg, "string"); return typeof key === "symbol" ? key : String(key); }
function _toPrimitive(input, hint) { if (typeof input !== "object" || input === null) return input; var prim = input[Symbol.toPrimitive]; if (prim !== undefined) { var res = prim.call(input, hint || "default"); if (typeof res !== "object") return res; throw new TypeError("@@toPrimitive must return a primitive value."); } return (hint === "string" ? String : Number)(input); }
const debug = (0, _debug.default)('verdaccio:token');
function normalizeToken(token) {
  return _objectSpread(_objectSpread({}, token), {}, {
    created: new Date(token.created).toISOString()
  });
}

// https://github.com/npm/npm-profile/blob/latest/lib/index.js
function _default(auth, storage, config) {
  const tokenRoute = (0, _express.Router)(); /* eslint new-cap: 0 */
  tokenRoute.get('/tokens', (0, _middleware.rateLimit)(config === null || config === void 0 ? void 0 : config.userRateLimit), async function (req, res, next) {
    const {
      name
    } = req.remote_user;
    if (_lodash.default.isNil(name) === false) {
      try {
        const tokens = await storage.readTokens({
          user: name
        });
        const totalTokens = tokens.length;
        debug('token list retrieved: %o', totalTokens);
        res.status(_constants.HTTP_STATUS.OK);
        return next({
          objects: tokens.map(normalizeToken),
          urls: {
            next: '' // TODO: pagination?
          }
        });
      } catch (error) {
        _logger.logger.error({
          error: error.msg
        }, 'token list has failed: @{error}');
        return next(_utils2.ErrorCode.getCode(_constants.HTTP_STATUS.INTERNAL_ERROR, error.message));
      }
    }
    return next(_utils2.ErrorCode.getUnauthorized());
  });
  tokenRoute.post('/tokens', (0, _middleware.rateLimit)(config === null || config === void 0 ? void 0 : config.userRateLimit), function (req, res, next) {
    const {
      password,
      readonly,
      cidr_whitelist
    } = req.body;
    const {
      name
    } = req.remote_user;
    if (!_lodash.default.isBoolean(readonly) || !_lodash.default.isArray(cidr_whitelist)) {
      return next(_utils2.ErrorCode.getCode(_constants.HTTP_STATUS.BAD_DATA, _constants.SUPPORT_ERRORS.PARAMETERS_NOT_VALID));
    }
    auth.authenticate(name, password, async (err, user) => {
      if (err) {
        const errorCode = err.message ? _constants.HTTP_STATUS.UNAUTHORIZED : _constants.HTTP_STATUS.INTERNAL_ERROR;
        return next(_utils2.ErrorCode.getCode(errorCode, err.message));
      }
      req.remote_user = user;
      if (!_lodash.default.isFunction(storage.saveToken)) {
        return next(_utils2.ErrorCode.getCode(_constants.HTTP_STATUS.NOT_IMPLEMENTED, _constants.SUPPORT_ERRORS.STORAGE_NOT_IMPLEMENT));
      }
      try {
        const token = await (0, _authUtils.getApiToken)(auth, config, user, password);
        const key = (0, _utils.stringToMD5)(token);
        // TODO: use a utility here
        const maskedToken = (0, _utils2.mask)(token, 5);
        const created = new Date().getTime();

        /**
         * cidr_whitelist: is not being used, we pass it through
         * token: we do not store the real token (it is generated once and retrieved to the user), just a mask of it.
         */
        const saveToken = {
          user: name,
          token: maskedToken,
          key,
          cidr: cidr_whitelist,
          readonly,
          created
        };
        await storage.saveToken(saveToken);
        debug('token %o was created for user %o', key, name);
        res.set(_constants.HEADERS.CACHE_CONTROL, 'no-cache, no-store');
        return next(normalizeToken({
          token,
          user: name,
          key: saveToken.key,
          cidr: cidr_whitelist,
          readonly,
          created: saveToken.created
        }));
      } catch (error) {
        _logger.logger.error({
          error: error.msg
        }, 'token creation has failed: @{error}');
        return next(_utils2.ErrorCode.getCode(_constants.HTTP_STATUS.INTERNAL_ERROR, error.message));
      }
    });
  });
  tokenRoute.delete('/tokens/token/:tokenKey', (0, _middleware.rateLimit)(config === null || config === void 0 ? void 0 : config.userRateLimit), async (req, res, next) => {
    const {
      params: {
        tokenKey
      }
    } = req;
    const {
      name
    } = req.remote_user;
    if (_lodash.default.isNil(name) === false) {
      debug('%o has requested remove a token', name);
      try {
        await storage.deleteToken(name, tokenKey);
        _logger.logger.info({
          tokenKey,
          name
        }, 'token id @{tokenKey} was revoked for user @{name}');
        return next({});
      } catch (error) {
        _logger.logger.error({
          error: error.msg
        }, 'token creation has failed: @{error}');
        return next(_utils2.ErrorCode.getCode(_constants.HTTP_STATUS.INTERNAL_ERROR, error.message));
      }
    }
    return next(_utils2.ErrorCode.getUnauthorized());
  });
  return tokenRoute;
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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