"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = _default;
var _lodash = _interopRequireDefault(require("lodash"));
var _semver = _interopRequireDefault(require("semver"));
var _constants = require("../../../../lib/constants");
var _logger = require("../../../../lib/logger");
function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }
const personMatch = (person, search) => {
  if (typeof person === 'string') {
    return person.includes(search);
  }
  if (typeof person === 'object') {
    for (const field of Object.values(person)) {
      if (typeof field === 'string' && field.includes(search)) {
        return true;
      }
    }
  }
  return false;
};
const matcher = function (query) {
  const match = query.match(/author:(.*)/);
  if (match !== null) {
    return function (pkg) {
      return personMatch(pkg.author, match[1]);
    };
  }

  // TODO: maintainer, keywords, boost-exact
  // TODO implement some scoring system for freetext
  return pkg => {
    return ['name', 'displayName', 'description'].map(k => {
      return pkg[k];
    }).filter(x => {
      return x !== undefined;
    }).some(txt => {
      return txt.includes(query);
    });
  };
};
function compileTextSearch(textSearch) {
  const textMatchers = (textSearch || '').split(' ').map(matcher);
  return pkg => textMatchers.every(m => m(pkg));
}
function removeDuplicates(results) {
  const pkgNames = [];
  return results.filter(pkg => {
    var _pkg$package, _pkg$package2;
    if (pkgNames.includes(pkg === null || pkg === void 0 ? void 0 : (_pkg$package = pkg.package) === null || _pkg$package === void 0 ? void 0 : _pkg$package.name)) {
      return false;
    }
    pkgNames.push(pkg === null || pkg === void 0 ? void 0 : (_pkg$package2 = pkg.package) === null || _pkg$package2 === void 0 ? void 0 : _pkg$package2.name);
    return true;
  });
}
function checkAccess(pkg, auth, remoteUser) {
  return new Promise((resolve, reject) => {
    var _pkg$package3;
    auth.allow_access({
      packageName: pkg === null || pkg === void 0 ? void 0 : (_pkg$package3 = pkg.package) === null || _pkg$package3 === void 0 ? void 0 : _pkg$package3.name
    }, remoteUser, function (err, allowed) {
      if (err) {
        if (err.status && String(err.status).match(/^4\d\d$/)) {
          // auth plugin returns 4xx user error,
          // that's equivalent of !allowed basically
          allowed = false;
          return resolve(null);
        } else {
          reject(err);
        }
      } else {
        return resolve(allowed ? pkg : null);
      }
    });
  });
}
async function sendResponse(resultBuf, resultStream, auth, req, from, size) {
  resultStream.destroy();
  const resultsCollection = resultBuf.map(pkg => {
    if (pkg !== null && pkg !== void 0 && pkg.name) {
      return {
        package: pkg,
        // not sure if flags is need it
        flags: {
          unstable: Object.keys(pkg.versions).some(v => _semver.default.satisfies(v, '^1.0.0')) ? undefined : true
        },
        local: true,
        score: {
          final: 1,
          detail: {
            quality: 1,
            popularity: 1,
            maintenance: 0
          }
        },
        searchScore: 100000
      };
    } else {
      return pkg;
    }
  });
  const checkAccessPromises = await Promise.all(removeDuplicates(resultsCollection).map(pkgItem => {
    return checkAccess(pkgItem, auth, req.remote_user);
  }));
  const final = checkAccessPromises.filter(i => !_lodash.default.isNull(i)).slice(from, size);
  _logger.logger.debug(`search results ${final === null || final === void 0 ? void 0 : final.length}`);
  const response = {
    objects: final,
    total: final.length,
    time: new Date().toUTCString()
  };
  _logger.logger.debug(`total response ${final.length}`);
  return response;
}

/**
 * Endpoint for npm search v1
 * req: 'GET /-/v1/search?text=react&size=20&from=0&quality=0.65&popularity=0.98&maintenance=0.5'
 */
function _default(route, auth, storage) {
  route.get('/-/v1/search', async (req, res, next) => {
    // TODO: implement proper result scoring weighted by quality, popularity and maintenance query parameters
    let [text, size, from /* , quality, popularity, maintenance */] = ['text', 'size', 'from' /* , 'quality', 'popularity', 'maintenance' */].map(k => req.query[k]);
    size = parseInt(size) || 20;
    from = parseInt(from) || 0;
    const isInteresting = compileTextSearch(text);
    const resultStream = storage.search(0, {
      req
    });
    let resultBuf = [];
    let completed = false;
    resultStream.on('data', pkg => {
      // packages from the upstreams
      if (_lodash.default.isArray(pkg)) {
        resultBuf = resultBuf.concat(pkg.filter(pkgItem => {
          var _pkgItem$package;
          if (!isInteresting(pkgItem === null || pkgItem === void 0 ? void 0 : pkgItem.package)) {
            return;
          }
          _logger.logger.debug(`[remote] pkg name ${pkgItem === null || pkgItem === void 0 ? void 0 : (_pkgItem$package = pkgItem.package) === null || _pkgItem$package === void 0 ? void 0 : _pkgItem$package.name}`);
          return true;
        }));
      } else {
        // packages from local
        // due compability with `/-/all` we cannot refactor storage.search();
        if (!isInteresting(pkg)) {
          return;
        }
        _logger.logger.debug(`[local] pkg name ${pkg === null || pkg === void 0 ? void 0 : pkg.name}`);
        resultBuf.push(pkg);
      }
    });
    resultStream.on('error', function () {
      _logger.logger.error('search endpoint has failed');
      res.socket.destroy();
    });
    resultStream.on('end', async () => {
      if (!completed) {
        completed = true;
        try {
          const response = await sendResponse(resultBuf, resultStream, auth, req, from, size);
          _logger.logger.info('search endpoint ok results @{total}', {
            total: response.total
          });
          res.status(_constants.HTTP_STATUS.OK).json(response);
        } catch (err) {
          _logger.logger.error('search endpoint has failed @{err}', {
            err
          });
          next(err);
        }
      }
    });
  });
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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