"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.addVersion = addVersion;
exports.default = publish;
exports.publishPackage = publishPackage;
exports.removeTarball = removeTarball;
exports.unPublishPackage = unPublishPackage;
exports.uploadPackageTarball = uploadPackageTarball;
var _debug = _interopRequireDefault(require("debug"));
var _lodash = _interopRequireDefault(require("lodash"));
var _mime = _interopRequireDefault(require("mime"));
var _path = _interopRequireDefault(require("path"));
var _core = require("@verdaccio/core");
var _middleware = require("@verdaccio/middleware");
var _constants = require("../../../lib/constants");
var _logger = require("../../../lib/logger");
var _notify = require("../../../lib/notify");
var _storageUtils = require("../../../lib/storage-utils");
var _utils = require("../../../lib/utils");
var _star = _interopRequireDefault(require("./star"));
function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }
function ownKeys(object, enumerableOnly) { var keys = Object.keys(object); if (Object.getOwnPropertySymbols) { var symbols = Object.getOwnPropertySymbols(object); enumerableOnly && (symbols = symbols.filter(function (sym) { return Object.getOwnPropertyDescriptor(object, sym).enumerable; })), keys.push.apply(keys, symbols); } return keys; }
function _objectSpread(target) { for (var i = 1; i < arguments.length; i++) { var source = null != arguments[i] ? arguments[i] : {}; i % 2 ? ownKeys(Object(source), !0).forEach(function (key) { _defineProperty(target, key, source[key]); }) : Object.getOwnPropertyDescriptors ? Object.defineProperties(target, Object.getOwnPropertyDescriptors(source)) : ownKeys(Object(source)).forEach(function (key) { Object.defineProperty(target, key, Object.getOwnPropertyDescriptor(source, key)); }); } return target; }
function _defineProperty(obj, key, value) { key = _toPropertyKey(key); if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }
function _toPropertyKey(arg) { var key = _toPrimitive(arg, "string"); return typeof key === "symbol" ? key : String(key); }
function _toPrimitive(input, hint) { if (typeof input !== "object" || input === null) return input; var prim = input[Symbol.toPrimitive]; if (prim !== undefined) { var res = prim.call(input, hint || "default"); if (typeof res !== "object") return res; throw new TypeError("@@toPrimitive must return a primitive value."); } return (hint === "string" ? String : Number)(input); }
const debug = (0, _debug.default)('verdaccio:publish');
function publish(router, auth, storage, config) {
  const can = (0, _middleware.allow)(auth, {
    beforeAll: (params, message) => _logger.logger.trace(params, message),
    afterAll: (params, message) => _logger.logger.trace(params, message)
  });

  /**
   * Publish a package / update package / un/start a package
   *
   * There are multiples scenarios here to be considered:
   *
   * 1. Publish scenario
   *
   * Publish a package consist of at least 1 step (PUT) with a metadata payload.
   * When a package is published, an _attachment property is present that contains the data
   * of the tarball.
   *
   * Example flow of publish.
   *
   *  npm http fetch PUT 201 http://localhost:4873/@scope%2ftest1 9627ms
      npm info lifecycle @scope/test1@1.0.1~publish: @scope/test1@1.0.1
      npm info lifecycle @scope/test1@1.0.1~postpublish: @scope/test1@1.0.1
      + @scope/test1@1.0.1
      npm verb exit [ 0, true ]
   *
   *
   * 2. Unpublish scenario
   *
   * Unpublish consist in 3 steps.
   *  1. Try to fetch  metadata -> if it fails, return 404
   *  2. Compute metadata locally (client side) and send a mutate payload excluding the version to be unpublished
   *    eg: if metadata reflects 1.0.1, 1.0.2 and 1.0.3, the computed metadata won't include 1.0.3.
   *  3. Once the second step has been successfully finished, delete the tarball.
   *
   *  All these steps are consecutive and required, there is no transacions here, if step 3 fails, metadata might
   *  get corrupted.
   *
   *  Note the unpublish call will suffix in the url a /-rev/14-5d500cfce92f90fd revision number, this not
   *  used internally.
   *
   *
   * Example flow of unpublish.
   *
   * npm http fetch GET 200 http://localhost:4873/@scope%2ftest1?write=true 1680ms
     npm http fetch PUT 201 http://localhost:4873/@scope%2ftest1/-rev/14-5d500cfce92f90fd 956606ms attempt #2
     npm http fetch GET 200 http://localhost:4873/@scope%2ftest1?write=true 1601ms
     npm http fetch DELETE 201 http://localhost:4873/@scope%2ftest1/-/test1-1.0.3.tgz/-rev/16-e11c8db282b2d992 19ms
   *
   * 3. Star a package
   *
   * Permissions: start a package depends of the publish and unpublish permissions, there is no specific flag for star or un start.
   * The URL for star is similar to the unpublish (change package format)
   *
   * npm has no enpoint for star a package, rather mutate the metadata and acts as, the difference is the
   * users property which is part of the payload and the body only includes
   *
   * {
    "_id": pkgName,
   	"_rev": "3-b0cdaefc9bdb77c8",
    "users": {
      [username]: boolean value (true, false)
    }
  }
   *
   */
  router.put('/:package/:_rev?/:revision?', can('publish'), (0, _middleware.media)(_mime.default.getType('json')), _middleware.expectJson, publishPackage(storage, config, auth));

  /**
   * Un-publishing an entire package.
   *
   * This scenario happens when the first call detect there is only one version remaining
   * in the metadata, then the client decides to DELETE the resource
   * npm http fetch GET 304 http://localhost:4873/@scope%2ftest1?write=true 1076ms (from cache)
     npm http fetch DELETE 201 http://localhost:4873/@scope%2ftest1/-rev/18-d8ebe3020bd4ac9c 22ms
   */
  router.delete('/:package/-rev/*', can('unpublish'), unPublishPackage(storage));

  // removing a tarball
  router.delete('/:package/-/:filename/-rev/:revision', can('unpublish'), can('publish'), removeTarball(storage));

  // uploading package tarball
  router.put('/:package/-/:filename/*', can('publish'), (0, _middleware.media)(_constants.HEADERS.OCTET_STREAM), uploadPackageTarball(storage));

  // adding a version
  router.put('/:package/:version/-tag/:tag', can('publish'), (0, _middleware.media)(_mime.default.getType('json')), _middleware.expectJson, addVersion(storage));
}

/**
 * Publish a package
 */
function publishPackage(storage, config, auth) {
  const starApi = (0, _star.default)(storage);
  return function (req, res, next) {
    const packageName = req.params.package;
    debug('publishing or updating a new version for %o', packageName);
    /**
     * Write tarball of stream data from package clients.
     */
    const createTarball = function (filename, data, cb) {
      const stream = storage.addTarball(packageName, filename);
      stream.on('error', function (err) {
        cb(err);
      });
      stream.on('success', function () {
        cb();
      });
      // this is dumb and memory-consuming, but what choices do we have?
      // flow: we need first refactor this file before decides which type use here
      stream.end(Buffer.from(data.data, 'base64'));
      stream.done();
    };

    /**
     * Add new package version in storage
     */
    const createVersion = function (version, metadata, cb) {
      storage.addVersion(packageName, version, metadata, null, cb);
    };

    /**
     * Add new tags in storage
     */
    const addTags = function (tags, cb) {
      storage.mergeTags(packageName, tags, cb);
    };
    const afterChange = function (error, okMessage, metadata) {
      const metadataCopy = _objectSpread({}, metadata);
      const {
        _attachments,
        versions
      } = metadataCopy;

      // `npm star` wouldn't have attachments
      // and `npm deprecate` would have attachments as a empty object, i.e {}
      if (_lodash.default.isNil(_attachments) || JSON.stringify(_attachments) === '{}') {
        if (error) {
          return next(error);
        }
        res.status(_constants.HTTP_STATUS.CREATED);
        return next({
          ok: okMessage,
          success: true
        });
      }

      // npm-registry-client 0.3+ embeds tarball into the json upload
      // https://github.com/isaacs/npm-registry-client/commit/e9fbeb8b67f249394f735c74ef11fe4720d46ca0
      // issue https://github.com/rlidwka/sinopia/issues/31, dealing with it here:
      const isInvalidBodyFormat = (0, _utils.isObject)(_attachments) === false || (0, _utils.hasDiffOneKey)(_attachments) || (0, _utils.isObject)(versions) === false || (0, _utils.hasDiffOneKey)(versions);
      if (isInvalidBodyFormat) {
        // npm is doing something strange again
        // if this happens in normal circumstances, report it as a bug
        _logger.logger.info({
          packageName
        }, `wrong package format on publish a package @{packageName}`);
        return next(_utils.ErrorCode.getBadRequest(_constants.API_ERROR.UNSUPORTED_REGISTRY_CALL));
      }
      if (error && error.status !== _constants.HTTP_STATUS.CONFLICT) {
        return next(error);
      }

      // at this point document is either created or existed before
      const [firstAttachmentKey] = Object.keys(_attachments);
      createTarball(_path.default.basename(firstAttachmentKey), _attachments[firstAttachmentKey], function (error) {
        if (error) {
          return next(error);
        }
        const versionToPublish = Object.keys(versions)[0];
        const versionMetadataToPublish = versions[versionToPublish];
        versionMetadataToPublish.readme = _lodash.default.isNil(versionMetadataToPublish.readme) === false ? String(versionMetadataToPublish.readme) : '';
        createVersion(versionToPublish, versionMetadataToPublish, function (error) {
          if (error) {
            return next(error);
          }
          addTags(metadataCopy[_constants.DIST_TAGS], async function (error) {
            if (error) {
              return next(error);
            }
            try {
              await (0, _notify.notify)(metadataCopy, config, req.remote_user, `${metadataCopy.name}@${versionToPublish}`);
            } catch (error) {
              _logger.logger.error({
                error
              }, 'notify batch service has failed: @{error}');
            }
            res.status(_constants.HTTP_STATUS.CREATED);
            return next({
              ok: okMessage,
              success: true
            });
          });
        });
      });
    };
    if ((0, _storageUtils.isPublishablePackage)(req.body) === false && (0, _utils.isObject)(req.body.users)) {
      return starApi(req, res, next);
    }
    try {
      const metadata = _core.validatioUtils.normalizeMetadata(req.body, packageName);
      // check _attachments to distinguish publish and deprecate
      if (req.params._rev || (0, _utils.isRelatedToDeprecation)(req.body) && _lodash.default.isEmpty(req.body._attachments)) {
        debug('updating a new version for %o', packageName);
        // we check unpublish permissions, an update is basically remove versions
        const remote = req.remote_user;
        auth.allow_unpublish({
          packageName
        }, remote, error => {
          if (error) {
            _logger.logger.error({
              packageName
            }, `not allowed to unpublish a version for @{packageName}`);
            return next(error);
          }
          storage.changePackage(packageName, metadata, req.params.revision, function (error) {
            afterChange(error, _constants.API_MESSAGE.PKG_CHANGED, metadata);
          });
        });
      } else {
        debug('adding a new version for %o', packageName);
        storage.addPackage(packageName, metadata, function (error) {
          afterChange(error, _constants.API_MESSAGE.PKG_CREATED, metadata);
        });
      }
    } catch (error) {
      _logger.logger.error({
        packageName
      }, 'error on publish, bad package data for @{packageName}');
      return next(_utils.ErrorCode.getBadData(_constants.API_ERROR.BAD_PACKAGE_DATA));
    }
  };
}

/**
 * un-publish a package
 */
function unPublishPackage(storage) {
  return function (req, res, next) {
    const packageName = req.params.package;
    debug('unpublishing %o', packageName);
    storage.removePackage(packageName, function (err) {
      if (err) {
        return next(err);
      }
      res.status(_constants.HTTP_STATUS.CREATED);
      return next({
        ok: _constants.API_MESSAGE.PKG_REMOVED
      });
    });
  };
}

/**
 * Delete tarball
 */
function removeTarball(storage) {
  return function (req, res, next) {
    const packageName = req.params.package;
    const {
      filename,
      revision
    } = req.params;
    debug('removing a tarball for %o-%o-%o', packageName, filename, revision);
    storage.removeTarball(packageName, filename, revision, function (err) {
      if (err) {
        return next(err);
      }
      res.status(_constants.HTTP_STATUS.CREATED);
      debug('success remove tarball for %o-%o-%o', packageName, filename, revision);
      return next({
        ok: _constants.API_MESSAGE.TARBALL_REMOVED
      });
    });
  };
}
/**
 * Adds a new version
 */
function addVersion(storage) {
  return function (req, res, next) {
    const {
      version,
      tag
    } = req.params;
    const packageName = req.params.package;
    storage.addVersion(packageName, version, req.body, tag, function (error) {
      if (error) {
        return next(error);
      }
      res.status(_constants.HTTP_STATUS.CREATED);
      return next({
        ok: _constants.API_MESSAGE.PKG_PUBLISHED
      });
    });
  };
}

/**
 * uploadPackageTarball
 */
function uploadPackageTarball(storage) {
  return function (req, res, next) {
    const packageName = req.params.package;
    const stream = storage.addTarball(packageName, req.params.filename);
    req.pipe(stream);

    // checking if end event came before closing
    let complete = false;
    req.on('end', function () {
      complete = true;
      stream.done();
    });
    req.on('close', function () {
      if (!complete) {
        stream.abort();
      }
    });
    stream.on('error', function (err) {
      return res.locals.report_error(err);
    });
    stream.on('success', function () {
      res.status(_constants.HTTP_STATUS.CREATED);
      return next({
        ok: _constants.API_MESSAGE.TARBALL_UPLOADED
      });
    });
  };
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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