"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = _default;
var _lodash = _interopRequireDefault(require("lodash"));
var _middleware = require("@verdaccio/middleware");
var _tarball = require("@verdaccio/tarball");
var _constants = require("../../../lib/constants");
var _logger = require("../../../lib/logger");
var _utils = require("../../../lib/utils");
var _string = require("../../../utils/string");
function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }
const downloadStream = (packageName, filename, storage, req, res) => {
  const stream = storage.getTarball(packageName, filename);
  stream.on('content-length', function (content) {
    res.header('Content-Length', content);
  });
  stream.on('error', function (err) {
    return res.locals.report_error(err);
  });
  res.header(_constants.HEADERS.CONTENT_TYPE, _constants.HEADERS.OCTET_STREAM);
  stream.pipe(res);
};
const redirectOrDownloadStream = (packageName, filename, storage, req, res, config) => {
  const tarballUrlRedirect = _lodash.default.get(config, 'experiments.tarball_url_redirect');
  storage.hasLocalTarball(packageName, filename).then(hasLocalTarball => {
    if (hasLocalTarball) {
      const context = {
        packageName,
        filename
      };
      const tarballUrl = typeof tarballUrlRedirect === 'function' ? tarballUrlRedirect(context) : _lodash.default.template(tarballUrlRedirect)(context);
      res.redirect(tarballUrl);
    } else {
      downloadStream(packageName, filename, storage, req, res);
    }
  }).catch(err => {
    res.locals.report_error(err);
  });
};
function _default(route, auth, storage, config) {
  const can = (0, _middleware.allow)(auth, {
    beforeAll: (params, message) => _logger.logger.trace(params, message),
    afterAll: (params, message) => _logger.logger.trace(params, message)
  });
  // TODO: anonymous user?
  route.get('/:package/:version?', can('access'), function (req, res, next) {
    const abbreviated = (0, _string.getByQualityPriorityValue)(req.get('Accept')) === 'application/vnd.npm.install-v1+json';
    const getPackageMetaCallback = function (err, metadata) {
      if (err) {
        return next(err);
      }
      metadata = (0, _tarball.convertDistRemoteToLocalTarballUrls)(metadata, {
        protocol: req.protocol,
        headers: req.headers,
        host: req.host,
        remoteAddress: req.socket.remoteAddress
      }, config.url_prefix);
      let queryVersion = req.params.version;
      if (_lodash.default.isNil(queryVersion)) {
        if (abbreviated) {
          res.setHeader(_constants.HEADERS.CONTENT_TYPE, 'application/vnd.npm.install-v1+json');
        } else {
          res.setHeader(_constants.HEADERS.CONTENT_TYPE, _constants.HEADERS.JSON);
        }
        return next(metadata);
      }
      let version = (0, _utils.getVersion)(metadata, queryVersion);
      if (_lodash.default.isNil(version) === false) {
        res.setHeader(_constants.HEADERS.CONTENT_TYPE, _constants.HEADERS.JSON);
        return next(version);
      }
      if (_lodash.default.isNil(metadata[_constants.DIST_TAGS]) === false) {
        if (_lodash.default.isNil(metadata[_constants.DIST_TAGS][queryVersion]) === false) {
          queryVersion = metadata[_constants.DIST_TAGS][queryVersion];
          version = (0, _utils.getVersion)(metadata, queryVersion);
          if (_lodash.default.isNil(version) === false) {
            res.setHeader(_constants.HEADERS.CONTENT_TYPE, _constants.HEADERS.JSON);
            return next(version);
          }
        }
      }
      return next(_utils.ErrorCode.getNotFound(`${_constants.API_ERROR.VERSION_NOT_EXIST}: ${req.params.version}`));
    };
    storage.getPackage({
      name: req.params.package,
      uplinksLook: true,
      req,
      abbreviated,
      callback: getPackageMetaCallback
    });
  });
  route.get('/:scopedPackage/-/:scope/:filename', can('access'), function (req, res) {
    const {
      scopedPackage,
      filename
    } = req.params;
    if (_lodash.default.get(config, 'experiments.tarball_url_redirect') === undefined) {
      downloadStream(scopedPackage, filename, storage, req, res);
    } else {
      redirectOrDownloadStream(scopedPackage, filename, storage, req, res, config);
    }
  });
  route.get('/:package/-/:filename', can('access'), function (req, res) {
    if (_lodash.default.get(config, 'experiments.tarball_url_redirect') === undefined) {
      downloadStream(req.params.package, req.params.filename, storage, req, res);
    } else {
      redirectOrDownloadStream(req.params.package, req.params.filename, storage, req, res, config);
    }
  });
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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