// Copyright (c) Jupyter Development Team.
// Distributed under the terms of the Modified BSD License.
import { Signal } from '@lumino/signaling';
import mergeWith from 'lodash.mergewith';
/**
 * Class to manager the registered features of the language servers.
 */
export class FeatureManager {
    constructor() {
        /**
         * List of registered features
         */
        this.features = [];
        this._featuresRegistered = new Signal(this);
    }
    /**
     * Signal emitted when a new feature is registered.
     */
    get featuresRegistered() {
        return this._featuresRegistered;
    }
    /**
     * Register a new feature, skip if it is already registered.
     */
    register(feature) {
        if (this.features.some(ft => ft.id === feature.id)) {
            console.warn(`Feature with id ${feature.id} is already registered, skipping.`);
        }
        else {
            this.features.push(feature);
            this._featuresRegistered.emit(feature);
        }
    }
    /**
     * Get the capabilities of all clients.
     */
    clientCapabilities() {
        let capabilities = {};
        for (const feature of this.features) {
            if (!feature.capabilities) {
                continue;
            }
            capabilities = mergeWith(capabilities, feature.capabilities);
        }
        return capabilities;
    }
}
//# sourceMappingURL=feature.js.map