// Copyright (c) Jupyter Development Team.
// Distributed under the terms of the Modified BSD License.
/**
 * Manager for the code extractors
 */
export class CodeExtractorsManager {
    constructor() {
        this._extractorMap = new Map();
        this._extractorMapAnyLanguage = new Map();
    }
    /**
     * Get the extractors for the input cell type and the main language of
     * the document
     *
     * @param  cellType - type of cell
     * @param  hostLanguage - main language of the document
     */
    getExtractors(cellType, hostLanguage) {
        var _a, _b;
        if (hostLanguage) {
            const currentMap = this._extractorMap.get(cellType);
            if (!currentMap) {
                return [];
            }
            return (_a = currentMap.get(hostLanguage)) !== null && _a !== void 0 ? _a : [];
        }
        else {
            return (_b = this._extractorMapAnyLanguage.get(cellType)) !== null && _b !== void 0 ? _b : [];
        }
    }
    /**
     * Register an extractor to extract foreign code from host documents of specified language.
     */
    register(extractor, hostLanguage) {
        const cellType = extractor.cellType;
        if (hostLanguage) {
            cellType.forEach(type => {
                if (!this._extractorMap.has(type)) {
                    this._extractorMap.set(type, new Map());
                }
                const currentMap = this._extractorMap.get(type);
                const extractorList = currentMap.get(hostLanguage);
                if (!extractorList) {
                    currentMap.set(hostLanguage, [extractor]);
                }
                else {
                    extractorList.push(extractor);
                }
            });
        }
        else {
            cellType.forEach(type => {
                if (!this._extractorMapAnyLanguage.has(type)) {
                    this._extractorMapAnyLanguage.set(type, []);
                }
                this._extractorMapAnyLanguage.get(type).push(extractor);
            });
        }
    }
}
//# sourceMappingURL=manager.js.map