import { IDocumentWidget } from '@jupyterlab/docregistry';
import { ISignal } from '@lumino/signaling';
import { WidgetLSPAdapter } from './adapters/adapter';
import { LSPConnection } from './connection';
import { AskServersToSendTraceNotifications } from './plugin';
import { Document, IDocumentConnectionData, ILanguageServerManager, ILSPConnection, ILSPDocumentConnectionManager, ISocketConnectionOptions, TLanguageServerConfigurations, TLanguageServerId } from './tokens';
import { VirtualDocument } from './virtual/document';
/**
 * Each Widget with a document (whether file or a notebook) has the same DocumentConnectionManager
 * (see JupyterLabWidgetAdapter). Using id_path instead of uri led to documents being overwritten
 * as two identical id_paths could be created for two different notebooks.
 */
export declare class DocumentConnectionManager implements ILSPDocumentConnectionManager {
    constructor(options: DocumentConnectionManager.IOptions);
    /**
     * Map between the URI of the virtual document and its connection
     * to the language server
     */
    readonly connections: Map<VirtualDocument.uri, LSPConnection>;
    /**
     * Map between the path of the document and its adapter
     */
    readonly adapters: Map<string, WidgetLSPAdapter<IDocumentWidget>>;
    /**
     * Map between the URI of the virtual document and the document itself.
     */
    readonly documents: Map<VirtualDocument.uri, VirtualDocument>;
    /**
     * The language server manager plugin.
     */
    readonly languageServerManager: ILanguageServerManager;
    /**
     * Initial configuration for the language servers.
     */
    initialConfigurations: TLanguageServerConfigurations;
    /**
     * Signal emitted when the manager is initialized.
     */
    get initialized(): ISignal<ILSPDocumentConnectionManager, IDocumentConnectionData>;
    /**
     * Signal emitted when the manager is connected to the server
     */
    get connected(): ISignal<ILSPDocumentConnectionManager, IDocumentConnectionData>;
    /**
     * Connection temporarily lost or could not be fully established; a re-connection will be attempted;
     */
    get disconnected(): ISignal<ILSPDocumentConnectionManager, IDocumentConnectionData>;
    /**
     * Connection was closed permanently and no-reconnection will be attempted, e.g.:
     *  - there was a serious server error
     *  - user closed the connection,
     *  - re-connection attempts exceeded,
     */
    get closed(): ISignal<ILSPDocumentConnectionManager, IDocumentConnectionData>;
    /**
     * Signal emitted when the document is changed.
     */
    get documentsChanged(): ISignal<ILSPDocumentConnectionManager, Map<VirtualDocument.uri, VirtualDocument>>;
    /**
     * Promise resolved when the language server manager is ready.
     */
    get ready(): Promise<void>;
    /**
     * Helper to connect various virtual document signal with callbacks of
     * this class.
     *
     * @param  virtualDocument - virtual document to be connected.
     */
    connectDocumentSignals(virtualDocument: VirtualDocument): void;
    /**
     * Helper to disconnect various virtual document signal with callbacks of
     * this class.
     *
     * @param  virtualDocument - virtual document to be disconnected.
     */
    disconnectDocumentSignals(virtualDocument: VirtualDocument, emit?: boolean): void;
    /**
     * Handle foreign document opened event.
     */
    onForeignDocumentOpened(_host: VirtualDocument, context: Document.IForeignContext): void;
    /**
     * Handle foreign document closed event.
     */
    onForeignDocumentClosed(_host: VirtualDocument, context: Document.IForeignContext): void;
    /**
     * Register a widget adapter with this manager
     *
     * @param  path - path to the inner document of the adapter
     * @param  adapter - the adapter to be registered
     */
    registerAdapter(path: string, adapter: WidgetLSPAdapter<IDocumentWidget>): void;
    /**
     * Handles the settings that do not require an existing connection
     * with a language server (or can influence to which server the
     * connection will be created, e.g. `rank`).
     *
     * This function should be called **before** initialization of servers.
     */
    updateConfiguration(allServerSettings: TLanguageServerConfigurations): void;
    /**
     * Handles the settings that the language servers accept using
     * `onDidChangeConfiguration` messages, which should be passed under
     * the "serverSettings" keyword in the setting registry.
     * Other configuration options are handled by `updateConfiguration` instead.
     *
     * This function should be called **after** initialization of servers.
     */
    updateServerConfigurations(allServerSettings: TLanguageServerConfigurations): void;
    /**
     * Fired the first time a connection is opened. These _should_ be the only
     * invocation of `.on` (once remaining LSPFeature.connection_handlers are made
     * singletons).
     */
    onNewConnection: (connection: LSPConnection) => void;
    /**
     * Retry to connect to the server each `reconnectDelay` seconds
     * and for `retrialsLeft` times.
     * TODO: presently no longer referenced. A failing connection would close
     * the socket, triggering the language server on the other end to exit.
     */
    retryToConnect(options: ISocketConnectionOptions, reconnectDelay: number, retrialsLeft?: number): Promise<void>;
    /**
     * Disconnect the connection to the language server of the requested
     * language.
     */
    disconnect(languageId: TLanguageServerId): void;
    /**
     * Create a new connection to the language server
     * @return A promise of the LSP connection
     */
    connect(options: ISocketConnectionOptions, firstTimeoutSeconds?: number, secondTimeoutMinutes?: number): Promise<ILSPConnection | undefined>;
    /**
     * Disconnect the signals of requested virtual document uri.
     */
    unregisterDocument(uri: string, emit?: boolean): void;
    /**
     * Enable or disable the logging feature of the language servers
     */
    updateLogging(logAllCommunication: boolean, setTrace: AskServersToSendTraceNotifications): void;
    /**
     * Create the LSP connection for requested virtual document.
     *
     * @return  Return the promise of the LSP connection.
     */
    private _connectSocket;
    /**
     * Helper to apply callback on all documents of a connection.
     */
    private _forEachDocumentOfConnection;
    private _initialized;
    private _connected;
    private _disconnected;
    private _closed;
    private _documentsChanged;
    /**
     * Set of ignored languages
     */
    private _ignoredLanguages;
}
export declare namespace DocumentConnectionManager {
    interface IOptions {
        /**
         * The language server manager instance.
         */
        languageServerManager: ILanguageServerManager;
    }
    /**
     * Generate the URI of a virtual document from input
     *
     * @param  virtualDocument - the virtual document
     * @param  language - language of the document
     */
    function solveUris(virtualDocument: VirtualDocument, language: string): IURIs | undefined;
    interface IURIs {
        /**
         * The root URI set by server.
         *
         */
        base: string;
        /**
         * The URI to the virtual document.
         *
         */
        document: string;
        /**
         * Address of websocket endpoint for LSP services.
         *
         */
        server: string;
        /**
         * Address of websocket endpoint for the language server.
         *
         */
        socket: string;
    }
}
