import { DocumentRegistry, IDocumentWidget } from '@jupyterlab/docregistry';
import { ITranslator, TranslationBundle } from '@jupyterlab/translation';
import { IDisposable } from '@lumino/disposable';
import { ISignal, Signal } from '@lumino/signaling';
import { LanguageIdentifier } from '../lsp';
import { IVirtualPosition } from '../positioning';
import { Document, IDocumentConnectionData, ILSPCodeExtractorsManager, ILSPDocumentConnectionManager, ILSPFeatureManager } from '../tokens';
import { VirtualDocument } from '../virtual/document';
export interface IEditorChangedData {
    /**
     * The CM editor invoking the change event.
     */
    editor: Document.IEditor;
}
export interface IAdapterOptions {
    /**
     * The LSP document and connection manager instance.
     */
    connectionManager: ILSPDocumentConnectionManager;
    /**
     * The LSP feature manager instance.
     */
    featureManager: ILSPFeatureManager;
    /**
     * The LSP foreign code extractor manager.
     */
    foreignCodeExtractorsManager: ILSPCodeExtractorsManager;
    /**
     * The translator provider.
     */
    translator?: ITranslator;
}
/**
 * Foreign code: low level adapter is not aware of the presence of foreign languages;
 * it operates on the virtual document and must not attempt to infer the language dependencies
 * as this would make the logic of inspections caching impossible to maintain, thus the WidgetAdapter
 * has to handle that, keeping multiple connections and multiple virtual documents.
 */
export declare abstract class WidgetLSPAdapter<T extends IDocumentWidget> implements IDisposable {
    widget: T;
    protected options: IAdapterOptions;
    constructor(widget: T, options: IAdapterOptions);
    /**
     * Check if the adapter is disposed
     */
    get isDisposed(): boolean;
    /**
     * Check if the document contains multiple editors
     */
    get hasMultipleEditors(): boolean;
    /**
     * Get the ID of the internal widget.
     */
    get widgetId(): string;
    /**
     * Get the language identifier of the document
     */
    get language(): LanguageIdentifier;
    /**
     * Signal emitted when the adapter is connected.
     */
    get adapterConnected(): ISignal<WidgetLSPAdapter<T>, IDocumentConnectionData>;
    /**
     * Signal emitted when the active editor have changed.
     */
    get activeEditorChanged(): ISignal<WidgetLSPAdapter<T>, IEditorChangedData>;
    /**
     * Signal emitted when the adapter is disposed.
     */
    get disposed(): ISignal<WidgetLSPAdapter<T>, void>;
    /**
     * Signal emitted when the an editor is changed.
     */
    get editorAdded(): ISignal<WidgetLSPAdapter<T>, IEditorChangedData>;
    /**
     * Signal emitted when the an editor is removed.
     */
    get editorRemoved(): ISignal<WidgetLSPAdapter<T>, IEditorChangedData>;
    /**
     * Get the inner HTMLElement of the document widget.
     */
    abstract get wrapperElement(): HTMLElement;
    /**
     * Get current path of the document.
     */
    abstract get documentPath(): string;
    /**
     * Get the mime type of the document.
     */
    abstract get mimeType(): string;
    /**
     * Get the file extension of the document.
     */
    abstract get languageFileExtension(): string | undefined;
    /**
     * Get the activated CM editor.
     */
    abstract get activeEditor(): Document.IEditor | undefined;
    /**
     * Get the list of CM editors in the document, there is only one editor
     * in the case of file editor.
     */
    abstract get editors(): Document.ICodeBlockOptions[];
    /**
     * Promise that resolves once the adapter is initialized
     */
    abstract get ready(): Promise<void>;
    /**
     * The virtual document is connected or not
     */
    get isConnected(): boolean;
    /**
     * The LSP document and connection manager instance.
     */
    get connectionManager(): ILSPDocumentConnectionManager;
    /**
     * The translator provider.
     */
    get trans(): TranslationBundle;
    /**
     * Promise that resolves once the document is updated
     */
    get updateFinished(): Promise<void>;
    /**
     * Internal virtual document of the adapter.
     */
    get virtualDocument(): VirtualDocument | null;
    /**
     * Callback on connection closed event.
     */
    onConnectionClosed(_: ILSPDocumentConnectionManager, { virtualDocument }: IDocumentConnectionData): void;
    /**
     * Dispose the adapter.
     */
    dispose(): void;
    /**
     * Disconnect virtual document from the language server.
     */
    disconnect(): void;
    /**
     * Update the virtual document.
     */
    updateDocuments(): Promise<void>;
    /**
     * Callback called on the document changed event.
     */
    documentChanged(virtualDocument: VirtualDocument, document: VirtualDocument, isInit?: boolean): void;
    /**
     * (re)create virtual document using current path and language
     */
    protected abstract createVirtualDocument(): VirtualDocument;
    /**
     * Get the index of editor from the cursor position in the virtual
     * document. Since there is only one editor, this method always return
     * 0
     *
     * @param position - the position of cursor in the virtual document.
     * @return - index of the virtual editor
     */
    abstract getEditorIndexAt(position: IVirtualPosition): number;
    /**
     * Get the index of input editor
     *
     * @param ceEditor - instance of the code editor
     */
    abstract getEditorIndex(ceEditor: Document.IEditor): number;
    /**
     * Get the wrapper of input editor.
     *
     * @param ceEditor
     */
    abstract getEditorWrapper(ceEditor: Document.IEditor): HTMLElement;
    protected reloadConnection(): void;
    /**
     * Callback on document saved event.
     */
    protected onSaveState(context: DocumentRegistry.IContext<DocumentRegistry.IModel>, state: DocumentRegistry.SaveState): void;
    /**
     * Connect the virtual document with the language server.
     */
    protected onConnected(data: IDocumentConnectionData): Promise<void>;
    /**
     * Opens a connection for the document. The connection may or may
     * not be initialized, yet, and depending on when this is called, the client
     * may not be fully connected.
     *
     * @param virtualDocument a VirtualDocument
     * @param sendOpen whether to open the document immediately
     */
    protected connectDocument(virtualDocument: VirtualDocument, sendOpen?: boolean): Promise<void>;
    /**
     * Create the virtual document using current path and language.
     */
    protected initVirtual(): void;
    /**
     * Handler for opening a document contained in a parent document. The assumption
     * is that the editor already exists for this, and as such the document
     * should be queued for immediate opening.
     *
     * @param host the VirtualDocument that contains the VirtualDocument in another language
     * @param context information about the foreign VirtualDocument
     */
    protected onForeignDocumentOpened(_: VirtualDocument, context: Document.IForeignContext): Promise<void>;
    /**
     * Signal emitted when the adapter is connected.
     */
    protected _adapterConnected: Signal<WidgetLSPAdapter<T>, IDocumentConnectionData>;
    /**
     * Signal emitted when the active editor have changed.
     */
    protected _activeEditorChanged: Signal<WidgetLSPAdapter<T>, IEditorChangedData>;
    /**
     * Signal emitted when an editor is changed.
     */
    protected _editorAdded: Signal<WidgetLSPAdapter<T>, IEditorChangedData>;
    /**
     * Signal emitted when an editor is removed.
     */
    protected _editorRemoved: Signal<WidgetLSPAdapter<T>, IEditorChangedData>;
    /**
     * Signal emitted when the adapter is disposed.
     */
    protected _disposed: Signal<WidgetLSPAdapter<T>, void>;
    private _isDisposed;
    private readonly _connectionManager;
    private readonly _trans;
    private _isConnected;
    private _updateFinished;
    private _virtualDocument;
    /**
     * Callback called when a foreign document is closed,
     * the associated signals with this virtual document
     * are disconnected.
     */
    private _onForeignDocumentClosed;
    /**
     * Detect the capabilities for the document type then
     * open the websocket connection with the language server.
     */
    private _connect;
    /**
     * Handle content changes and update all virtual documents after a change.
     *
     * #### Notes
     * Update to the state of a notebook may be done without a notice on the
     * CodeMirror level, e.g. when a cell is deleted. Therefore a
     * JupyterLab-specific signal is watched instead.
     *
     * While by not using the change event of CodeMirror editors we lose an easy
     * way to send selective (range) updates this can be still implemented by
     * comparison of before/after states of the virtual documents, which is
     * more resilient and editor-independent.
     */
    private _onContentChanged;
}
