"use strict";
var __assign = (this && this.__assign) || function () {
    __assign = Object.assign || function(t) {
        for (var s, i = 1, n = arguments.length; i < n; i++) {
            s = arguments[i];
            for (var p in s) if (Object.prototype.hasOwnProperty.call(s, p))
                t[p] = s[p];
        }
        return t;
    };
    return __assign.apply(this, arguments);
};
exports.__esModule = true;
exports.fileTests = exports.testTree = void 0;
var common_1 = require("@lezer/common");
var none = [];
var TestSpec = /** @class */ (function () {
    function TestSpec(name, props, children, wildcard) {
        if (children === void 0) { children = none; }
        if (wildcard === void 0) { wildcard = false; }
        this.name = name;
        this.props = props;
        this.children = children;
        this.wildcard = wildcard;
    }
    TestSpec.parse = function (spec) {
        var pos = 0, tok = "sof", value = "";
        function err() {
            throw new SyntaxError("Invalid test spec: " + spec);
        }
        function next() {
            while (pos < spec.length && /\s/.test(spec.charAt(pos)))
                pos++;
            if (pos == spec.length)
                return tok = "eof";
            var next = spec.charAt(pos++);
            if (next == "(" && spec.slice(pos, pos + 4) == "...)") {
                pos += 4;
                return tok = "...";
            }
            if (/[\[\](),=]/.test(next))
                return tok = next;
            if (/[^()\[\],="\s]/.test(next)) {
                var name_1 = /[^()\[\],="\s]*/.exec(spec.slice(pos - 1));
                value = name_1[0];
                pos += name_1[0].length - 1;
                return tok = "name";
            }
            if (next == '"') {
                var content = /^"((?:[^\\"]|\\.)*)"/.exec(spec.slice(pos - 1)) || err();
                value = JSON.parse(content[0]);
                pos += content[0].length - 1;
                return tok = "name";
            }
            return err();
        }
        next();
        function parseSeq() {
            var seq = [];
            while (tok != "eof" && tok != ")") {
                seq.push(parse());
                if (tok == ",")
                    next();
            }
            return seq;
        }
        function parse() {
            var name = value, children = none, props = [], wildcard = false;
            if (tok != "name")
                err();
            next();
            if (tok == "[") {
                next();
                while (tok != "]") {
                    if (tok != "name")
                        err();
                    var prop = common_1.NodeProp[value], val = "";
                    if (!(prop instanceof common_1.NodeProp))
                        err();
                    next();
                    if (tok == "=") {
                        next();
                        if (tok != "name")
                            err();
                        val = value;
                        next();
                    }
                    props.push({ prop: prop, value: prop.deserialize(val) });
                }
                next();
            }
            if (tok == "(") {
                next();
                children = parseSeq();
                // @ts-ignore TypeScript doesn't understand that `next` may have mutated `tok` (#9998)
                if (tok != ")")
                    err();
                next();
            }
            else if (tok == "...") {
                wildcard = true;
                next();
            }
            return new TestSpec(name, props, children, wildcard);
        }
        var result = parseSeq();
        if (tok != "eof")
            err();
        return result;
    };
    TestSpec.prototype.matches = function (type) {
        if (type.name != this.name)
            return false;
        for (var _i = 0, _a = this.props; _i < _a.length; _i++) {
            var _b = _a[_i], prop = _b.prop, value = _b.value;
            if ((value || type.prop(prop)) && JSON.stringify(type.prop(prop)) != JSON.stringify(value))
                return false;
        }
        return true;
    };
    return TestSpec;
}());
function defaultIgnore(type) { return /\W/.test(type.name); }
function testTree(tree, expect, mayIgnore) {
    if (mayIgnore === void 0) { mayIgnore = defaultIgnore; }
    var specs = TestSpec.parse(expect);
    var stack = [specs], pos = [0];
    tree.iterate({
        enter: function (n) {
            if (!n.name)
                return;
            var last = stack.length - 1, index = pos[last], seq = stack[last];
            var next = index < seq.length ? seq[index] : null;
            if (next && next.matches(n.type)) {
                if (next.wildcard) {
                    pos[last]++;
                    return false;
                }
                pos.push(0);
                stack.push(next.children);
                return undefined;
            }
            else if (mayIgnore(n.type)) {
                return false;
            }
            else {
                var parent_1 = last > 0 ? stack[last - 1][pos[last - 1]].name : "tree";
                var after_1 = next ? next.name + (parent_1 == "tree" ? "" : " in " + parent_1) : "end of ".concat(parent_1);
                throw new Error("Expected ".concat(after_1, ", got ").concat(n.name, " at ").concat(n.to, " \n").concat(tree));
            }
        },
        leave: function (n) {
            if (!n.name)
                return;
            var last = stack.length - 1, index = pos[last], seq = stack[last];
            if (index < seq.length)
                throw new Error("Unexpected end of ".concat(n.name, ". Expected ").concat(seq.slice(index).map(function (s) { return s.name; }).join(", "), " at ").concat(n.from, "\n").concat(tree));
            pos.pop();
            stack.pop();
            pos[last - 1]++;
        }
    });
    if (pos[0] != specs.length)
        throw new Error("Unexpected end of tree. Expected ".concat(stack[0].slice(pos[0]).map(function (s) { return s.name; }).join(", "), " at ").concat(tree.length, "\n").concat(tree));
}
exports.testTree = testTree;
/**
 * Returns a line context for the given file.
 *
 * @param {string} file
 * @param {number} index
 */
function toLineContext(file, index) {
    var endEol = file.indexOf('\n', index + 80);
    var endIndex = endEol === -1 ? file.length : endEol;
    return file.substring(index, endIndex).split(/\n/).map(function (str) { return '  | ' + str; }).join('\n');
}
function fileTests(file, fileName, mayIgnore) {
    if (mayIgnore === void 0) { mayIgnore = defaultIgnore; }
    var caseExpr = /\s*#[ \t]*(.*)(?:\r\n|\r|\n)([^]*?)==+>([^]*?)(?:$|(?:\r\n|\r|\n)+(?=#))/gy;
    var tests = [];
    var lastIndex = 0;
    var _loop_1 = function () {
        var m = caseExpr.exec(file);
        if (!m)
            throw new Error("Unexpected file format in ".concat(fileName, " around\n\n").concat(toLineContext(file, lastIndex)));
        var text = m[2].trim(), expected = m[3].trim();
        var _a = /(.*?)(\{.*?\})?$/.exec(m[1]), name_2 = _a[1], configStr = _a[2];
        var config = configStr ? JSON.parse(configStr) : null;
        var strict = !/⚠|\.\.\./.test(expected);
        tests.push({
            name: name_2,
            text: text,
            expected: expected,
            configStr: configStr,
            config: config,
            strict: strict,
            run: function (parser) {
                if (parser.configure && (strict || config))
                    parser = parser.configure(__assign({ strict: strict }, config));
                testTree(parser.parse(text), expected, mayIgnore);
            }
        });
        lastIndex = m.index + m[0].length;
        if (lastIndex == file.length)
            return "break";
    };
    for (;;) {
        var state_1 = _loop_1();
        if (state_1 === "break")
            break;
    }
    return tests;
}
exports.fileTests = fileTests;
