// Copyright (c) Jupyter Development Team.
// Distributed under the terms of the Modified BSD License.
import { Sanitizer } from '@jupyterlab/apputils';
import { renderText } from '@jupyterlab/rendermime';
import { HoverBox } from '@jupyterlab/ui-components';
import { ElementExt } from '@lumino/domutils';
import { Signal } from '@lumino/signaling';
import { Widget } from '@lumino/widgets';
/**
 * The class name added to completer menu items.
 */
const ITEM_CLASS = 'jp-Completer-item';
/**
 * The class name added to an active completer menu item.
 */
const ACTIVE_CLASS = 'jp-mod-active';
/**
 * The class used by item listing which determines the height of the completer.
 */
const LIST_CLASS = 'jp-Completer-list';
/**
 * Class of the documentation panel.
 */
const DOC_PANEL_CLASS = 'jp-Completer-docpanel';
/**
 * A flag to indicate that event handlers are caught in the capture phase.
 */
const USE_CAPTURE = true;
/**
 * The number of colors defined for the completer type annotations.
 * These are listed in completer/style/index.css#102-152.
 */
const N_COLORS = 10;
/**
 * A widget that enables text completion.
 *
 * #### Notes
 * The completer is intended to be absolutely positioned on the
 * page and hover over any other content, so it should be attached directly
 * to `document.body`, or a node that is the full size of `document.body`.
 * Attaching it to other nodes may incorrectly locate the completer.
 */
export class Completer extends Widget {
    /**
     * Construct a text completer menu widget.
     */
    constructor(options) {
        var _a, _b, _c, _d;
        super({ node: document.createElement('div') });
        this._activeIndex = 0;
        this._editor = null;
        this._model = null;
        this._selected = new Signal(this);
        this._visibilityChanged = new Signal(this);
        this._indexChanged = new Signal(this);
        this._lastSubsetMatch = '';
        this._geometryLock = false;
        /**
         * Increasing this counter invalidates previous request to save geometry cache in animation callback.
         */
        this._geometryCounter = 0;
        this._docPanelExpanded = false;
        this._renderCounter = 0;
        this.sanitizer = (_a = options.sanitizer) !== null && _a !== void 0 ? _a : new Sanitizer();
        this._defaultRenderer = Completer.getDefaultRenderer(this.sanitizer);
        this._renderer = (_b = options.renderer) !== null && _b !== void 0 ? _b : this._defaultRenderer;
        this.model = (_c = options.model) !== null && _c !== void 0 ? _c : null;
        this.editor = (_d = options.editor) !== null && _d !== void 0 ? _d : null;
        this.addClass('jp-Completer');
        this._updateConstraints();
    }
    /**
     * Cache style constraints from CSS.
     */
    _updateConstraints() {
        const tempNode = document.createElement('div');
        tempNode.classList.add(LIST_CLASS);
        tempNode.style.visibility = 'hidden';
        tempNode.style.overflowY = 'scroll';
        document.body.appendChild(tempNode);
        const computedStyle = window.getComputedStyle(tempNode);
        this._maxHeight = parseInt(computedStyle.maxHeight, 10);
        this._minHeight = parseInt(computedStyle.minHeight, 10);
        this._scrollbarWidth = tempNode.offsetWidth - tempNode.clientWidth;
        document.body.removeChild(tempNode);
        const tempDocPanel = document.createElement('div');
        tempDocPanel.classList.add(DOC_PANEL_CLASS);
        this._docPanelWidth = Private.measureSize(tempDocPanel, 'inline-block').width;
    }
    /**
     * The active index.
     */
    get activeIndex() {
        return this._activeIndex;
    }
    /**
     * The editor used by the completion widget.
     */
    get editor() {
        return this._editor;
    }
    set editor(newValue) {
        this._editor = newValue;
    }
    /**
     * A signal emitted when a selection is made from the completer menu.
     */
    get selected() {
        return this._selected;
    }
    /**
     * A signal emitted when the completer widget's visibility changes.
     *
     * #### Notes
     * This signal is useful when there are multiple floating widgets that may
     * contend with the same space and ought to be mutually exclusive.
     */
    get visibilityChanged() {
        return this._visibilityChanged;
    }
    /**
     * A signal emitted when the active index changes.
     */
    get indexChanged() {
        return this._indexChanged;
    }
    /**
     * The model used by the completer widget.
     */
    get model() {
        return this._model;
    }
    set model(model) {
        if ((!model && !this._model) || model === this._model) {
            return;
        }
        if (this._model) {
            this._model.stateChanged.disconnect(this.onModelStateChanged, this);
            this._model.queryChanged.disconnect(this.onModelQueryChanged, this);
        }
        this._model = model;
        if (this._model) {
            this._model.stateChanged.connect(this.onModelStateChanged, this);
            this._model.queryChanged.connect(this.onModelQueryChanged, this);
        }
    }
    /**
     * Enable/disable the document panel.
     */
    set showDocsPanel(showDoc) {
        this._showDoc = showDoc;
    }
    get showDocsPanel() {
        return this._showDoc;
    }
    /**
     * Dispose of the resources held by the completer widget.
     */
    dispose() {
        this._sizeCache = undefined;
        this._model = null;
        super.dispose();
    }
    /**
     * Handle the DOM events for the widget.
     *
     * @param event - The DOM event sent to the widget.
     *
     * #### Notes
     * This method implements the DOM `EventListener` interface and is
     * called in response to events on the dock panel's node. It should
     * not be called directly by user code.
     */
    handleEvent(event) {
        if (this.isHidden || !this._editor) {
            return;
        }
        switch (event.type) {
            case 'keydown':
                this._evtKeydown(event);
                break;
            case 'mousedown':
                this._evtMousedown(event);
                break;
            case 'scroll':
                this._evtScroll(event);
                break;
            default:
                break;
        }
    }
    /**
     * Reset the widget.
     */
    reset() {
        this._activeIndex = 0;
        this._lastSubsetMatch = '';
        if (this._model) {
            this._model.reset(true);
        }
        // Clear size cache.
        this._sizeCache = undefined;
        this.node.scrollTop = 0;
    }
    /**
     * Emit the selected signal for the current active item and reset.
     */
    selectActive() {
        const active = this.node.querySelector(`.${ACTIVE_CLASS}`);
        if (!active) {
            this.reset();
            return;
        }
        this._selected.emit(active.getAttribute('data-value'));
        this.reset();
    }
    /**
     * Handle `after-attach` messages for the widget.
     */
    onAfterAttach(msg) {
        document.addEventListener('keydown', this, USE_CAPTURE);
        document.addEventListener('mousedown', this, USE_CAPTURE);
        document.addEventListener('scroll', this, USE_CAPTURE);
    }
    /**
     * Handle `before-detach` messages for the widget.
     */
    onBeforeDetach(msg) {
        document.removeEventListener('keydown', this, USE_CAPTURE);
        document.removeEventListener('mousedown', this, USE_CAPTURE);
        document.removeEventListener('scroll', this, USE_CAPTURE);
    }
    /**
     * Handle model state changes.
     */
    onModelStateChanged() {
        if (this.isAttached) {
            this._activeIndex = 0;
            this._indexChanged.emit(this._activeIndex);
            this.update();
        }
    }
    /**
     * Handle model query changes.
     */
    onModelQueryChanged(model, queryChange) {
        // If query was changed by the user typing, the filtered down items
        // may no longer reach/exceed the maxHeight of the completer widget,
        // hence size needs to be recalculated.
        if (this._sizeCache && queryChange.origin === 'editorUpdate') {
            const newItems = model.completionItems();
            const oldItems = this._sizeCache.items;
            // Only reset size if the number of items changed, or the longest item changed.
            const oldWidest = oldItems[this._findWidestItemIndex(oldItems)];
            const newWidest = newItems[this._findWidestItemIndex(newItems)];
            const heuristic = this._getPreferredItemWidthHeuristic();
            if (newItems.length !== this._sizeCache.items.length ||
                heuristic(oldWidest) !== heuristic(newWidest)) {
                this._sizeCache = undefined;
            }
        }
    }
    /**
     * Handle `update-request` messages.
     */
    onUpdateRequest(msg) {
        var _a;
        const model = this._model;
        if (!model) {
            return;
        }
        // If this is the first time the current completer session has loaded,
        // populate any initial subset match. This is being done before node
        // gets rendered to avoid rendering it twice.
        if (!model.query) {
            this._populateSubset();
        }
        let items = model.completionItems();
        // If there are no items, reset and bail.
        if (!items.length) {
            if (!this.isHidden) {
                this.reset();
                this.hide();
                this._visibilityChanged.emit(undefined);
            }
            return;
        }
        // Update constraints before any DOM modifications
        this._updateConstraints();
        // Do not trigger any geometry updates from async code when in lock.
        this._geometryLock = true;
        const node = this._createCompleterNode(model, items);
        let active = node.querySelectorAll(`.${ITEM_CLASS}`)[this._activeIndex];
        active.classList.add(ACTIVE_CLASS);
        // Add the documentation panel
        if (this._showDoc) {
            let docPanel = document.createElement('div');
            docPanel.className = DOC_PANEL_CLASS;
            this._docPanel = docPanel;
            node.appendChild(docPanel);
            this._docPanelExpanded = false;
        }
        const resolvedItem = (_a = this.model) === null || _a === void 0 ? void 0 : _a.resolveItem(this._activeIndex);
        this._updateDocPanel(resolvedItem);
        if (this.isHidden) {
            this.show();
            this._setGeometry();
            this._visibilityChanged.emit(undefined);
        }
        else {
            this._setGeometry();
        }
        this._geometryLock = false;
    }
    /**
     * Get cached dimensions of the completer box.
     */
    get sizeCache() {
        if (!this._sizeCache) {
            return;
        }
        return {
            width: this._sizeCache.width,
            height: this._sizeCache.height
        };
    }
    _createCompleterNode(model, items) {
        const current = ++this._renderCounter;
        // Clear the node.
        let node = this.node;
        node.textContent = '';
        // Compute an ordered list of all the types in the typeMap, this is computed
        // once by the model each time new data arrives for efficiency.
        let orderedTypes = model.orderedTypes();
        // Populate the completer items.
        let ul = document.createElement('ul');
        ul.className = LIST_CLASS;
        // Add first N items to fill the first "page" assuming that the completer
        // would reach its maximum allowed height.
        const first = this._renderer.createCompletionItemNode(items[0], orderedTypes);
        const renderedItems = [first];
        const firstItemSize = Private.measureSize(first, 'inline-grid');
        const pageSize = Math.max(Math.ceil(this._maxHeight / firstItemSize.height), 5);
        // We add one item in case if height heuristic is inacurate.
        const toRenderImmediately = Math.min(pageSize + 1, items.length);
        const start = performance.now();
        for (let i = 1; i < toRenderImmediately; i++) {
            const li = this._renderer.createCompletionItemNode(items[i], orderedTypes);
            renderedItems.push(li);
        }
        for (const li of renderedItems) {
            ul.appendChild(li);
        }
        if (pageSize < items.length) {
            // If the first "page" is completely filled, we can pre-calculate size:
            //  - height will equal maximum allowed height,
            //  - width will be estimated from the widest item.
            // If the page size is larger than the number of items, then there are
            // few items and the benefit from pre-computing the size is negligible.
            const widestItemIndex = this._findWidestItemIndex(items);
            const widestItem = widestItemIndex < renderedItems.length
                ? renderedItems[widestItemIndex]
                : this._renderer.createCompletionItemNode(items[widestItemIndex], orderedTypes);
            // The node needs to be cloned to avoid side-effect of detaching it.
            const widestItemSize = Private.measureSize(widestItem.cloneNode(true), 'inline-grid');
            this._sizeCache = {
                height: this._maxHeight,
                width: widestItemSize.width + this._scrollbarWidth,
                items: items
            };
        }
        if (toRenderImmediately < items.length) {
            // Render remaining items on idle in subsequent animation frames,
            // in chunks of size such that each frame would take about 16ms
            // allowing for 4ms of overhead, but keep the chunks no smaller
            // than 5 items at a time.
            const timePerItem = (performance.now() - start) / toRenderImmediately;
            const chunkSize = Math.max(5, Math.floor(12 / timePerItem));
            let alreadyRendered = toRenderImmediately;
            let previousChunkFinal = renderedItems[renderedItems.length - 1];
            const renderChunk = () => {
                if (alreadyRendered >= items.length) {
                    return;
                }
                // Add a filler so that the list with partially rendered items has the total
                // height equal to the (predicted) final height to avoid scrollbar jitter.
                const predictedMissingHeight = firstItemSize.height * (items.length - alreadyRendered);
                previousChunkFinal.style.marginBottom = `${predictedMissingHeight}px`;
                requestAnimationFrame(() => {
                    if (current != this._renderCounter) {
                        // Bail if rendering afresh was requested in the meantime.
                        return;
                    }
                    previousChunkFinal.style.marginBottom = '';
                    const limit = Math.min(items.length, alreadyRendered + chunkSize);
                    for (let i = alreadyRendered; i < limit; i++) {
                        const li = this._renderer.createCompletionItemNode(items[i], orderedTypes);
                        ul.appendChild(li);
                        previousChunkFinal = li;
                    }
                    alreadyRendered = limit;
                    renderChunk();
                });
            };
            renderChunk();
        }
        node.appendChild(ul);
        return node;
    }
    /**
     * Use preferred heuristic to find the index of the widest item.
     */
    _findWidestItemIndex(items) {
        const widthHeuristic = this._getPreferredItemWidthHeuristic();
        const widthHeuristics = items.map(widthHeuristic);
        return widthHeuristics.indexOf(Math.max(...widthHeuristics));
    }
    /**
     * Get item width heuristic function from renderer if available,
     * or the default one otherwise.
     */
    _getPreferredItemWidthHeuristic() {
        return this._renderer.itemWidthHeuristic
            ? this._renderer.itemWidthHeuristic.bind(this._renderer)
            : this._defaultRenderer.itemWidthHeuristic.bind(this._defaultRenderer);
    }
    /**
     * Cycle through the available completer items.
     *
     * #### Notes
     * When the user cycles all the way `down` to the last index, subsequent
     * `down` cycles will cycle to the first index. When the user cycles `up` to
     * the first item, subsequent `up` cycles will cycle to the last index.
     */
    _cycle(direction) {
        var _a;
        const items = this.node.querySelectorAll(`.${ITEM_CLASS}`);
        const index = this._activeIndex;
        let active = this.node.querySelector(`.${ACTIVE_CLASS}`);
        active.classList.remove(ACTIVE_CLASS);
        if (direction === 'up') {
            this._activeIndex = index === 0 ? items.length - 1 : index - 1;
        }
        else if (direction === 'down') {
            this._activeIndex = index < items.length - 1 ? index + 1 : 0;
        }
        else {
            // Measure the number of items on a page.
            const boxHeight = this.node.getBoundingClientRect().height;
            const itemHeight = active.getBoundingClientRect().height;
            const pageLength = Math.floor(boxHeight / itemHeight);
            // Update the index
            if (direction === 'pageUp') {
                this._activeIndex = index - pageLength;
            }
            else {
                this._activeIndex = index + pageLength;
            }
            // Clamp to the length of the list.
            this._activeIndex = Math.min(Math.max(0, this._activeIndex), items.length - 1);
        }
        active = items[this._activeIndex];
        active.classList.add(ACTIVE_CLASS);
        let completionList = this.node.querySelector(`.${LIST_CLASS}`);
        ElementExt.scrollIntoViewIfNeeded(completionList, active);
        this._indexChanged.emit(this._activeIndex);
        const resolvedItem = (_a = this.model) === null || _a === void 0 ? void 0 : _a.resolveItem(this._activeIndex);
        this._updateDocPanel(resolvedItem);
    }
    /**
     * Handle keydown events for the widget.
     */
    _evtKeydown(event) {
        if (this.isHidden || !this._editor) {
            return;
        }
        if (!this._editor.host.contains(event.target)) {
            this.reset();
            return;
        }
        switch (event.keyCode) {
            case 9: {
                // Tab key
                event.preventDefault();
                event.stopPropagation();
                event.stopImmediatePropagation();
                const model = this._model;
                if (!model) {
                    return;
                }
                // Autoinsert single completions on manual request (tab)
                const items = model.completionItems();
                if (items && items.length === 1) {
                    this._selected.emit(items[0].insertText || items[0].label);
                    this.reset();
                    return;
                }
                const populated = this._populateSubset();
                // If the common subset was found and set on `query`,
                // or if there is a `query` in the initialization options,
                // then emit a completion signal with that `query` (=subset match),
                // but only if the query has actually changed.
                // See: https://github.com/jupyterlab/jupyterlab/issues/10439#issuecomment-875189540
                if (model.query && model.query != this._lastSubsetMatch) {
                    model.subsetMatch = true;
                    this._selected.emit(model.query);
                    model.subsetMatch = false;
                    this._lastSubsetMatch = model.query;
                }
                // If the query changed, update rendering of the options.
                if (populated) {
                    this.update();
                }
                this._cycle(event.shiftKey ? 'up' : 'down');
                return;
            }
            case 27: // Esc key
                event.preventDefault();
                event.stopPropagation();
                event.stopImmediatePropagation();
                this.reset();
                return;
            case 33: // PageUp
            case 34: // PageDown
            case 38: // Up arrow key
            case 40: {
                // Down arrow key
                event.preventDefault();
                event.stopPropagation();
                event.stopImmediatePropagation();
                const cycle = Private.keyCodeMap[event.keyCode];
                this._cycle(cycle);
                return;
            }
            default:
                return;
        }
    }
    /**
     * Handle mousedown events for the widget.
     */
    _evtMousedown(event) {
        if (this.isHidden || !this._editor) {
            return;
        }
        if (Private.nonstandardClick(event)) {
            this.reset();
            return;
        }
        let target = event.target;
        while (target !== document.documentElement) {
            // If the user has made a selection, emit its value and reset the widget.
            if (target.classList.contains(ITEM_CLASS)) {
                event.preventDefault();
                event.stopPropagation();
                event.stopImmediatePropagation();
                this._selected.emit(target.getAttribute('data-value'));
                this.reset();
                return;
            }
            // If the mouse event happened anywhere else in the widget, bail.
            if (target === this.node) {
                event.preventDefault();
                event.stopPropagation();
                event.stopImmediatePropagation();
                return;
            }
            target = target.parentElement;
        }
        this.reset();
    }
    /**
     * Handle scroll events for the widget
     */
    _evtScroll(event) {
        if (this.isHidden || !this._editor) {
            return;
        }
        const { node } = this;
        // All scrolls except scrolls in the actual hover box node may cause the
        // referent editor that anchors the node to move, so the only scroll events
        // that can safely be ignored are ones that happen inside the hovering node.
        if (node.contains(event.target)) {
            return;
        }
        // Set the geometry of the node asynchronously.
        requestAnimationFrame(() => {
            this._setGeometry();
        });
    }
    /**
     * Populate the completer up to the longest initial subset of items.
     *
     * @returns `true` if a subset match was found and populated.
     */
    _populateSubset() {
        const { model } = this;
        if (!model) {
            return false;
        }
        const items = model.completionItems();
        const subset = Private.commonSubset(items.map(item => item.insertText || item.label));
        const { query } = model;
        // If a common subset exists and it is not the current query, highlight it.
        if (subset && subset !== query && subset.indexOf(query) === 0) {
            model.query = subset;
            return true;
        }
        return false;
    }
    /**
     * Set the visible dimensions of the widget.
     */
    _setGeometry() {
        const { node } = this;
        const model = this._model;
        const editor = this._editor;
        // This is an overly defensive test: `cursor` will always exist if
        // `original` exists, except in contrived tests. But since it is possible
        // to generate a runtime error, the check occurs here.
        if (!editor || !model || !model.original || !model.cursor) {
            return;
        }
        const start = model.cursor.start;
        const position = editor.getPositionAt(start);
        const anchor = editor.getCoordinateForPosition(position);
        const style = window.getComputedStyle(node);
        const borderLeft = parseInt(style.borderLeftWidth, 10) || 0;
        const paddingLeft = parseInt(style.paddingLeft, 10) || 0;
        // When the editor is attached to the main area, contain the completer hover box
        // to the full area available (rather than to the editor itself); the available
        // area excludes the toolbar, hence the first Widget child between MainAreaWidget
        // and editor is preferred. The difference is negligible in File Editor, but
        // substantial for Notebooks.
        const host = editor.host.closest('.jp-MainAreaWidget > .lm-Widget') ||
            editor.host;
        const items = model.completionItems();
        // Fast cache invalidation (only checks for length rather than length + width)
        if (this._sizeCache && this._sizeCache.items.length !== items.length) {
            this._sizeCache = undefined;
        }
        // Calculate the geometry of the completer.
        HoverBox.setGeometry({
            anchor,
            host: host,
            maxHeight: this._maxHeight,
            minHeight: this._minHeight,
            node: node,
            size: this._sizeCache,
            offset: { horizontal: borderLeft + paddingLeft },
            privilege: 'below',
            style: style,
            outOfViewDisplay: {
                top: 'stick-inside',
                bottom: 'stick-inside',
                left: 'stick-inside',
                right: 'stick-outside'
            }
        });
        const current = ++this._geometryCounter;
        if (!this._sizeCache) {
            // If size was not pre-calculated using heuristics, save the actual
            // size into cache once rendered.
            requestAnimationFrame(() => {
                if (current != this._geometryCounter) {
                    // Do not set size to cache if it may already be outdated.
                    return;
                }
                let rect = node.getBoundingClientRect();
                this._sizeCache = {
                    width: rect.width,
                    height: rect.height,
                    items: items
                };
            });
        }
    }
    /**
     * Update the display-state and contents of the documentation panel
     */
    _updateDocPanel(resolvedItem) {
        var _a, _b, _c;
        let docPanel = this._docPanel;
        if (!docPanel) {
            return;
        }
        this._toggleDocPanel(true);
        if (!resolvedItem) {
            this._toggleDocPanel(false);
            return;
        }
        docPanel.textContent = '';
        const loadingIndicator = (_c = (_b = (_a = this._renderer).createLoadingDocsIndicator) === null || _b === void 0 ? void 0 : _b.call(_a)) !== null && _c !== void 0 ? _c : this._defaultRenderer.createLoadingDocsIndicator();
        docPanel.appendChild(loadingIndicator);
        resolvedItem
            .then(activeItem => {
            var _a, _b, _c;
            if (!activeItem) {
                return;
            }
            if (!docPanel) {
                return;
            }
            if (activeItem.documentation) {
                const node = (_c = (_b = (_a = this._renderer).createDocumentationNode) === null || _b === void 0 ? void 0 : _b.call(_a, activeItem)) !== null && _c !== void 0 ? _c : this._defaultRenderer.createDocumentationNode(activeItem);
                docPanel.textContent = '';
                docPanel.appendChild(node);
            }
            else {
                this._toggleDocPanel(false);
            }
        })
            .catch(e => console.error(e));
    }
    _toggleDocPanel(show) {
        let docPanel = this._docPanel;
        if (!docPanel) {
            return;
        }
        if (show) {
            if (this._docPanelExpanded) {
                return;
            }
            docPanel.style.display = '';
            this._docPanelExpanded = true;
        }
        else {
            if (!this._docPanelExpanded) {
                return;
            }
            docPanel.style.display = 'none';
            this._docPanelExpanded = false;
        }
        const sizeCache = this._sizeCache;
        if (sizeCache) {
            sizeCache.width += this._docPanelWidth * (show ? +1 : -1);
            if (!this._geometryLock) {
                this._setGeometry();
            }
        }
    }
}
(function (Completer) {
    /**
     * The default implementation of an `IRenderer`.
     */
    class Renderer {
        constructor(options) {
            this.sanitizer = (options === null || options === void 0 ? void 0 : options.sanitizer) || new Sanitizer();
        }
        /**
         * Create an item node from an ICompletionItem for a text completer menu.
         */
        createCompletionItemNode(item, orderedTypes) {
            let wrapperNode = this._createWrapperNode(item.insertText || item.label);
            if (item.deprecated) {
                wrapperNode.classList.add('jp-Completer-deprecated');
            }
            return this._constructNode(wrapperNode, this._createLabelNode(item.label), !!item.type, item.type, orderedTypes, item.icon);
        }
        /**
         * Create a documentation node for documentation panel.
         */
        createDocumentationNode(activeItem) {
            const host = document.createElement('div');
            host.classList.add('jp-RenderedText');
            const sanitizer = this.sanitizer;
            const source = activeItem.documentation || '';
            renderText({ host, sanitizer, source }).catch(console.error);
            return host;
        }
        /**
         * Get a heuristic for the width of an item.
         */
        itemWidthHeuristic(item) {
            var _a;
            return (item.label.replace(/<\?mark>/g, '').length + (((_a = item.type) === null || _a === void 0 ? void 0 : _a.length) || 0));
        }
        /**
         * Create a loading bar for the documentation panel.
         */
        createLoadingDocsIndicator() {
            const loadingContainer = document.createElement('div');
            loadingContainer.classList.add('jp-Completer-loading-bar-container');
            const loadingBar = document.createElement('div');
            loadingBar.classList.add('jp-Completer-loading-bar');
            loadingContainer.append(loadingBar);
            return loadingContainer;
        }
        /**
         * Create base node with the value to be inserted.
         */
        _createWrapperNode(value) {
            const li = document.createElement('li');
            li.className = ITEM_CLASS;
            // Set the raw, un-marked up value as a data attribute.
            li.setAttribute('data-value', value);
            return li;
        }
        /**
         * Create match node to highlight potential prefix match within result.
         */
        _createLabelNode(result) {
            const matchNode = document.createElement('code');
            matchNode.className = 'jp-Completer-match';
            // Use innerHTML because search results include <mark> tags.
            matchNode.innerHTML = result;
            return matchNode;
        }
        /**
         * Attaches type and match nodes to base node.
         */
        _constructNode(li, matchNode, typesExist, type, orderedTypes, icon) {
            // Add the icon or type monogram
            if (icon) {
                const iconNode = icon.element({
                    className: 'jp-Completer-type jp-Completer-icon'
                });
                li.appendChild(iconNode);
            }
            else if (typesExist) {
                const typeNode = document.createElement('span');
                typeNode.textContent = (type[0] || '').toLowerCase();
                const colorIndex = (orderedTypes.indexOf(type) % N_COLORS) + 1;
                typeNode.className = 'jp-Completer-type jp-Completer-monogram';
                typeNode.setAttribute(`data-color-index`, colorIndex.toString());
                li.appendChild(typeNode);
            }
            else {
                // Create empty span to ensure consistent list styling.
                // Otherwise, in a list of two items,
                // if one item has an icon, but the other has type,
                // the icon grows out of its bounds.
                const dummyNode = document.createElement('span');
                dummyNode.className = 'jp-Completer-monogram';
                li.appendChild(dummyNode);
            }
            li.appendChild(matchNode);
            // If there is a type, add the type extension and title
            if (typesExist) {
                li.title = type;
                const typeExtendedNode = document.createElement('code');
                typeExtendedNode.className = 'jp-Completer-typeExtended';
                typeExtendedNode.textContent = type.toLocaleLowerCase();
                li.appendChild(typeExtendedNode);
            }
            else {
                // If no type is present on the right,
                // the highlighting of the completion item
                // doesn't cover the entire row.
                const dummyTypeExtendedNode = document.createElement('span');
                dummyTypeExtendedNode.className = 'jp-Completer-typeExtended';
                li.appendChild(dummyTypeExtendedNode);
            }
            return li;
        }
    }
    Completer.Renderer = Renderer;
    /**
     * Default renderer
     */
    let _defaultRenderer;
    /**
     * The default `IRenderer` instance.
     */
    function getDefaultRenderer(sanitizer) {
        if (!_defaultRenderer ||
            (sanitizer && _defaultRenderer.sanitizer !== sanitizer)) {
            _defaultRenderer = new Renderer({ sanitizer: sanitizer });
        }
        return _defaultRenderer;
    }
    Completer.getDefaultRenderer = getDefaultRenderer;
})(Completer || (Completer = {}));
/**
 * A namespace for completer widget private data.
 */
var Private;
(function (Private) {
    /**
     * Mapping from keyCodes to scrollTypes.
     */
    Private.keyCodeMap = {
        38: 'up',
        40: 'down',
        33: 'pageUp',
        34: 'pageDown'
    };
    /**
     * Returns the common subset string that a list of strings shares.
     */
    function commonSubset(values) {
        const len = values.length;
        let subset = '';
        if (len < 2) {
            return subset;
        }
        const strlen = values[0].length;
        for (let i = 0; i < strlen; i++) {
            const ch = values[0][i];
            for (let j = 1; j < len; j++) {
                if (values[j][i] !== ch) {
                    return subset;
                }
            }
            subset += ch;
        }
        return subset;
    }
    Private.commonSubset = commonSubset;
    /**
     * Returns true for any modified click event (i.e., not a left-click).
     */
    function nonstandardClick(event) {
        return (event.button !== 0 ||
            event.altKey ||
            event.ctrlKey ||
            event.shiftKey ||
            event.metaKey);
    }
    Private.nonstandardClick = nonstandardClick;
    /**
     * Measure size of provided HTML element without painting it.
     *
     * #### Notes
     * The provided element has to be detached (not connected to DOM),
     * or a side-effect of detaching it will occur.
     */
    function measureSize(element, display) {
        if (element.isConnected) {
            console.warn('Measuring connected elements with `measureSize` has side-effects');
        }
        element.style.visibility = 'hidden';
        element.style.display = display;
        document.body.appendChild(element);
        const size = element.getBoundingClientRect();
        document.body.removeChild(element);
        element.removeAttribute('style');
        return size;
    }
    Private.measureSize = measureSize;
})(Private || (Private = {}));
//# sourceMappingURL=widget.js.map