// Copyright (c) Jupyter Development Team.
// Distributed under the terms of the Modified BSD License.
/**
 * The reconciliator which is used to fetch and merge responses from multiple completion providers.
 */
export class ProviderReconciliator {
    /**
     * Creates an instance of ProviderReconciliator.
     */
    constructor(options) {
        this._resolveFactory = (provider, el) => provider.resolve
            ? (patch) => provider.resolve(el, this._context, patch)
            : undefined;
        /**
         * Counter to reject current provider response if a new fetch request is created.
         */
        this._fetching = 0;
        this._providers = options.providers;
        this._context = options.context;
        this._timeout = options.timeout;
    }
    /**
     * Fetch response from multiple providers, If a provider can not return
     * the response for a completer request before timeout,
     * the result of this provider will be ignored.
     *
     * @param {CompletionHandler.IRequest} request - The completion request.
     */
    async fetch(request) {
        const current = ++this._fetching;
        let promises = [];
        for (const provider of this._providers) {
            let promise;
            promise = provider.fetch(request, this._context).then(reply => {
                if (current !== this._fetching) {
                    return Promise.reject(void 0);
                }
                const items = reply.items.map(el => ({
                    ...el,
                    resolve: this._resolveFactory(provider, el)
                }));
                return { ...reply, items };
            });
            const timeoutPromise = new Promise(resolve => {
                return setTimeout(() => resolve(null), this._timeout);
            });
            promise = Promise.race([promise, timeoutPromise]);
            // Wrap promise and return error in case of failure.
            promises.push(promise.catch(p => p));
        }
        // TODO: maybe use `Promise.allSettled` once library is at es2020 instead of adding a catch.
        const combinedPromise = Promise.all(promises);
        return this._mergeCompletions(combinedPromise);
    }
    /**
     * Check if completer should make request to fetch completion responses
     * on user typing. If the provider with highest rank does not have
     * `shouldShowContinuousHint` method, a default one will be used.
     *
     * @param completerIsVisible - The visible status of completer widget.
     * @param changed - CodeMirror changed argument.
     */
    shouldShowContinuousHint(completerIsVisible, changed) {
        if (this._providers[0].shouldShowContinuousHint) {
            return this._providers[0].shouldShowContinuousHint(completerIsVisible, changed);
        }
        return this._defaultShouldShowContinuousHint(completerIsVisible, changed);
    }
    _alignPrefixes(replies, minStart, maxStart) {
        if (minStart != maxStart) {
            const editor = this._context.editor;
            if (!editor) {
                return replies;
            }
            const cursor = editor.getCursorPosition();
            const line = editor.getLine(cursor.line);
            if (!line) {
                return replies;
            }
            return replies.map(reply => {
                // No prefix to strip, return as-is.
                if (reply.start == maxStart) {
                    return reply;
                }
                let prefix = line.substring(reply.start, maxStart);
                return {
                    ...reply,
                    items: reply.items.map(item => {
                        let insertText = item.insertText || item.label;
                        item.insertText = insertText.startsWith(prefix)
                            ? insertText.slice(prefix.length)
                            : insertText;
                        return item;
                    })
                };
            });
        }
        return replies;
    }
    async _mergeCompletions(promises) {
        let replies = (await promises).filter(reply => {
            // Ignore it errors out.
            if (!reply || reply instanceof Error) {
                return false;
            }
            // Ignore if no matches.
            if (!reply.items.length) {
                return false;
            }
            // Otherwise keep.
            return true;
        });
        // Fast path for a single reply or no replies.
        if (replies.length == 0) {
            return null;
        }
        else if (replies.length == 1) {
            return replies[0];
        }
        const minEnd = Math.min(...replies.map(reply => reply.end));
        // If any of the replies uses a wider range, we need to align them
        // so that all responses use the same range.
        const starts = replies.map(reply => reply.start);
        const minStart = Math.min(...starts);
        const maxStart = Math.max(...starts);
        replies = this._alignPrefixes(replies, minStart, maxStart);
        const insertTextSet = new Set();
        const mergedItems = new Array();
        for (const reply of replies) {
            reply.items.forEach(item => {
                // IPython returns 'import' and 'import '; while the latter is more useful,
                // user should not see two suggestions with identical labels and nearly-identical
                // behaviour as they could not distinguish the two either way.
                let text = (item.insertText || item.label).trim();
                if (insertTextSet.has(text)) {
                    return;
                }
                insertTextSet.add(text);
                mergedItems.push(item);
            });
        }
        return {
            start: maxStart,
            end: minEnd,
            items: mergedItems
        };
    }
    _defaultShouldShowContinuousHint(completerIsVisible, changed) {
        return (!completerIsVisible &&
            (changed.sourceChange == null ||
                changed.sourceChange.some(delta => delta.insert != null && delta.insert.length > 0)));
    }
}
//# sourceMappingURL=reconciliator.js.map