import { CodeEditor } from '@jupyterlab/codeeditor';
import { ISharedText, SourceChange } from '@jupyter/ydoc';
import { IDataConnector } from '@jupyterlab/statedb';
import { LabIcon } from '@jupyterlab/ui-components';
import { IDisposable } from '@lumino/disposable';
import { Message } from '@lumino/messaging';
import { IProviderReconciliator } from './tokens';
import { Completer } from './widget';
/**
 * A completion handler for editors.
 */
export declare class CompletionHandler implements IDisposable {
    /**
     * Construct a new completion handler for a widget.
     */
    constructor(options: CompletionHandler.IOptions);
    /**
     * The completer widget managed by the handler.
     */
    readonly completer: Completer;
    set reconciliator(reconciliator: IProviderReconciliator);
    /**
     * The editor used by the completion handler.
     */
    get editor(): CodeEditor.IEditor | null | undefined;
    set editor(newValue: CodeEditor.IEditor | null | undefined);
    /**
     * Get whether the completion handler is disposed.
     */
    get isDisposed(): boolean;
    /**
     * Enable/disable continuous hinting mode.
     */
    set autoCompletion(value: boolean);
    get autoCompletion(): boolean;
    /**
     * Dispose of the resources used by the handler.
     */
    dispose(): void;
    /**
     * Invoke the handler and launch a completer.
     */
    invoke(): void;
    /**
     * Process a message sent to the completion handler.
     */
    processMessage(msg: Message): void;
    /**
     * Get the state of the text editor at the given position.
     */
    protected getState(editor: CodeEditor.IEditor, position: CodeEditor.IPosition): Completer.ITextState;
    /**
     * Handle a completion selected signal from the completion widget.
     */
    protected onCompletionSelected(completer: Completer, val: string): void;
    /**
     * Handle `invoke-request` messages.
     */
    protected onInvokeRequest(msg: Message): void;
    /**
     * Handle selection changed signal from an editor.
     *
     * #### Notes
     * If a sub-class reimplements this method, then that class must either call
     * its super method or it must take responsibility for adding and removing
     * the completer completable class to the editor host node.
     *
     * Despite the fact that the editor widget adds a class whenever there is a
     * primary selection, this method checks independently for two reasons:
     *
     * 1. The editor widget connects to the same signal to add that class, so
     *    there is no guarantee that the class will be added before this method
     *    is invoked so simply checking for the CSS class's existence is not an
     *    option. Secondarily, checking the editor state should be faster than
     *    querying the DOM in either case.
     * 2. Because this method adds a class that indicates whether completer
     *    functionality ought to be enabled, relying on the behavior of the
     *    `jp-mod-has-primary-selection` to filter out any editors that have
     *    a selection means the semantic meaning of `jp-mod-completer-enabled`
     *    is obscured because there may be cases where the enabled class is added
     *    even though the completer is not available.
     */
    protected onSelectionsChanged(): void;
    /**
     * Handle a text changed signal from an editor.
     */
    protected onTextChanged(str: ISharedText, changed: SourceChange): void;
    /**
     * Handle a visibility change signal from a completer widget.
     */
    protected onVisibilityChanged(completer: Completer): void;
    /**
     * Make a completion request.
     */
    private _makeRequest;
    /**
     * Updates model with text state and current cursor position.
     */
    private _updateModel;
    private _reconciliator;
    private _editor;
    private _enabled;
    private _isDisposed;
    private _autoCompletion;
}
/**
 * A namespace for cell completion handler statics.
 */
export declare namespace CompletionHandler {
    /**
     * The instantiation options for cell completion handlers.
     */
    interface IOptions {
        /**
         * The completion widget the handler will connect to.
         */
        completer: Completer;
        /**
         * The reconciliator that will fetch and merge completions from active providers.
         */
        reconciliator: IProviderReconciliator;
    }
    /**
     * Type alias for ICompletionItem list.
     * Implementers of this interface should be responsible for
     * deduping and sorting the items in the list.
     */
    type ICompletionItems = ReadonlyArray<ICompletionItem>;
    /**
     * Completion item object based off of LSP CompletionItem.
     * Compared to the old kernel completions interface, this enhances the completions UI to support:
     * - differentiation between inserted text and user facing text
     * - documentation for each completion item to be displayed adjacently
     * - deprecation styling
     * - custom icons
     * and other potential new features.
     */
    interface ICompletionItem {
        /**
         * User facing completion.
         * If insertText is not set, this will be inserted.
         */
        label: string;
        /**
         * Completion to be inserted.
         */
        insertText?: string;
        /**
         * Type of this completion item.
         */
        type?: string;
        /**
         * LabIcon object for icon to be rendered with completion type.
         */
        icon?: LabIcon;
        /**
         * A human-readable string with additional information
         * about this item, like type or symbol information.
         */
        documentation?: string;
        /**
         * Indicates if the item is deprecated.
         */
        deprecated?: boolean;
        /**
         * Method allowing to update fields asynchronously.
         */
        resolve?: (patch?: Completer.IPatch) => Promise<CompletionHandler.ICompletionItem>;
    }
    /**
     * Connector for completion items.
     *
     * @deprecated since v4 to add a new source of completions, register a completion provider;
     *   to customise how completions get merged, provide a custom reconciliator.
     */
    type ICompletionItemsConnector = IDataConnector<CompletionHandler.ICompletionItemsReply, void, CompletionHandler.IRequest>;
    /**
     * A reply to a completion items fetch request.
     */
    interface ICompletionItemsReply<T extends CompletionHandler.ICompletionItem = CompletionHandler.ICompletionItem> {
        /**
         * The starting index for the substring being replaced by completion.
         */
        start: number;
        /**
         * The end index for the substring being replaced by completion.
         */
        end: number;
        /**
         * A list of completion items. default to CompletionHandler.ICompletionItems
         */
        items: Array<T>;
    }
    /**
     * The details of a completion request.
     */
    interface IRequest {
        /**
         * The cursor offset position within the text being completed.
         */
        offset: number;
        /**
         * The text being completed.
         */
        text: string;
    }
    /**
     * A namespace for completion handler messages.
     */
    namespace Msg {
        /**
         * A singleton `'invoke-request'` message.
         */
        const InvokeRequest: Message;
    }
}
