import { ReactWidget } from '@jupyterlab/apputils';
import { ISettingRegistry } from '@jupyterlab/settingregistry';
import { ITranslator } from '@jupyterlab/translation';
import { IScore } from '@jupyterlab/ui-components';
import { Message } from '@lumino/messaging';
import { ISignal } from '@lumino/signaling';
/**
 * A list of plugins with editable settings.
 */
export declare class PluginList extends ReactWidget {
    /**
     * Create a new plugin list.
     */
    constructor(options: PluginList.IOptions);
    /**
     * The setting registry.
     */
    readonly registry: ISettingRegistry;
    /**
     * A signal emitted when a list user interaction happens.
     */
    get changed(): ISignal<this, void>;
    /**
     * The selection value of the plugin list.
     */
    get scrollTop(): number | undefined;
    get hasErrors(): boolean;
    get filter(): (item: ISettingRegistry.IPlugin) => string[] | null;
    /**
     * The selection value of the plugin list.
     */
    get selection(): string;
    set selection(selection: string);
    /**
     * Signal that fires when search filter is updated so that settings panel can filter results.
     */
    get updateFilterSignal(): ISignal<this, (plugin: ISettingRegistry.IPlugin) => string[] | null>;
    get handleSelectSignal(): ISignal<this, string>;
    /**
     * Handle `'update-request'` messages.
     */
    protected onUpdateRequest(msg: Message): void;
    /**
     * Handle the `'mousedown'` event for the plugin list.
     *
     * @param event - The DOM event sent to the widget
     */
    private _evtMousedown;
    /**
     * Check the plugin for a rendering hint's value.
     *
     * #### Notes
     * The order of priority for overridden hints is as follows, from most
     * important to least:
     * 1. Data set by the end user in a settings file.
     * 2. Data set by the plugin author as a schema default.
     * 3. Data set by the plugin author as a top-level key of the schema.
     */
    getHint(key: string, registry: ISettingRegistry, plugin: ISettingRegistry.IPlugin): string;
    /**
     * Function to recursively filter properties that match search results.
     * @param filter - Function to filter based on search results
     * @param props - Schema properties being filtered
     * @param definitions - Definitions to use for filling in references in properties
     * @param ref - Reference to a definition
     * @returns - String array of properties that match the search results.
     */
    getFilterString(filter: (item: string) => Partial<IScore> | null, props: ISettingRegistry.IProperty, definitions?: any, ref?: string): string[];
    /**
     * Updates the filter when the search bar value changes.
     * @param filter Filter function passed by search bar based on search value.
     */
    setFilter(filter: (item: string) => Partial<IScore> | null, query?: string): void;
    setError(id: string, error: boolean): void;
    mapPlugins(plugin: ISettingRegistry.IPlugin): JSX.Element;
    render(): JSX.Element;
    protected translator: ITranslator;
    private _changed;
    private _errors;
    private _filter;
    private _query;
    private _handleSelectSignal;
    private _updateFilterSignal;
    private _allPlugins;
    private _settings;
    private _confirm?;
    private _scrollTop;
    private _selection;
}
/**
 * A namespace for `PluginList` statics.
 */
export declare namespace PluginList {
    /**
     * The instantiation options for a plugin list.
     */
    interface IOptions {
        /**
         * A function that allows for asynchronously confirming a selection.
         *
         * #### Notes
         * If the promise returned by the function resolves, then the selection will
         * succeed and emit an event. If the promise rejects, the selection is not
         * made.
         */
        confirm?: (id: string) => Promise<void>;
        /**
         * The setting registry for the plugin list.
         */
        registry: ISettingRegistry;
        /**
         * List of plugins to skip
         */
        toSkip?: string[];
        /**
         * The setting registry for the plugin list.
         */
        translator?: ITranslator;
        /**
         * An optional initial query so the plugin list can filter on start.
         */
        query?: string;
    }
    /**
     * Sort a list of plugins by title and ID.
     */
    function sortPlugins(registry: ISettingRegistry): ISettingRegistry.IPlugin[];
}
