/* -----------------------------------------------------------------------------
| Copyright (c) Jupyter Development Team.
| Distributed under the terms of the Modified BSD License.
|----------------------------------------------------------------------------*/
import { nullTranslator } from '@jupyterlab/translation';
import { jupyterIcon, ReactWidget } from '@jupyterlab/ui-components';
import { JSONExt } from '@lumino/coreutils';
import { SplitPanel } from '@lumino/widgets';
import * as React from 'react';
import { PluginEditor } from './plugineditor';
import { PluginList } from './pluginlist';
/**
 * The ratio panes in the setting editor.
 */
const DEFAULT_LAYOUT = {
    sizes: [1, 3],
    container: {
        editor: 'raw',
        plugin: '',
        sizes: [1, 1]
    }
};
/**
 * An interface for modifying and saving application settings.
 */
export class JsonSettingEditor extends SplitPanel {
    /**
     * Create a new setting editor.
     */
    constructor(options) {
        super({
            orientation: 'horizontal',
            renderer: SplitPanel.defaultRenderer,
            spacing: 1
        });
        this._fetching = null;
        this._saving = false;
        this._state = JSONExt.deepCopy(DEFAULT_LAYOUT);
        this.translator = options.translator || nullTranslator;
        const trans = this.translator.load('jupyterlab');
        this.addClass('jp-SettingEditor');
        this.key = options.key;
        this.state = options.state;
        const { commands, editorFactory, rendermime } = options;
        const registry = (this.registry = options.registry);
        const instructions = (this._instructions = ReactWidget.create(React.createElement(React.Fragment, null,
            React.createElement("h2", null,
                React.createElement(jupyterIcon.react, { className: "jp-SettingEditorInstructions-icon", tag: "span", elementPosition: "center", height: "auto", width: "60px" }),
                React.createElement("span", { className: "jp-SettingEditorInstructions-title" }, trans.__('Settings'))),
            React.createElement("span", { className: "jp-SettingEditorInstructions-text" }, trans.__('Select a plugin from the list to view and edit its preferences.')))));
        instructions.addClass('jp-SettingEditorInstructions');
        const editor = (this._editor = new PluginEditor({
            commands,
            editorFactory,
            registry,
            rendermime,
            translator: this.translator
        }));
        const confirm = () => editor.confirm();
        const list = (this._list = new PluginList({
            confirm,
            registry,
            translator: this.translator
        }));
        const when = options.when;
        if (when) {
            this._when = Array.isArray(when) ? Promise.all(when) : when;
        }
        this.addWidget(list);
        this.addWidget(instructions);
        SplitPanel.setStretch(list, 0);
        SplitPanel.setStretch(instructions, 1);
        SplitPanel.setStretch(editor, 1);
        editor.stateChanged.connect(this._onStateChanged, this);
        list.changed.connect(this._onStateChanged, this);
        this.handleMoved.connect(this._onStateChanged, this);
    }
    /**
     * Whether the raw editor revert functionality is enabled.
     */
    get canRevertRaw() {
        return this._editor.raw.canRevert;
    }
    /**
     * Whether the raw editor save functionality is enabled.
     */
    get canSaveRaw() {
        return this._editor.raw.canSave;
    }
    /**
     * Emits when the commands passed in at instantiation change.
     */
    get commandsChanged() {
        return this._editor.raw.commandsChanged;
    }
    /**
     * The currently loaded settings.
     */
    get settings() {
        return this._editor.settings;
    }
    /**
     * The inspectable raw user editor source for the currently loaded settings.
     */
    get source() {
        return this._editor.raw.source;
    }
    /**
     * Dispose of the resources held by the setting editor.
     */
    dispose() {
        if (this.isDisposed) {
            return;
        }
        super.dispose();
        this._editor.dispose();
        this._instructions.dispose();
        this._list.dispose();
    }
    /**
     * Revert raw editor back to original settings.
     */
    revert() {
        this._editor.raw.revert();
    }
    /**
     * Save the contents of the raw editor.
     */
    save() {
        return this._editor.raw.save();
    }
    /**
     * Handle `'after-attach'` messages.
     */
    onAfterAttach(msg) {
        super.onAfterAttach(msg);
        this.hide();
        this._fetchState()
            .then(() => {
            this.show();
            this._setState();
        })
            .catch(reason => {
            console.error('Fetching setting editor state failed', reason);
            this.show();
            this._setState();
        });
    }
    /**
     * Handle `'close-request'` messages.
     */
    onCloseRequest(msg) {
        this._editor
            .confirm()
            .then(() => {
            super.onCloseRequest(msg);
            this.dispose();
        })
            .catch(() => {
            /* no op */
        });
    }
    /**
     * Get the state of the panel.
     */
    _fetchState() {
        if (this._fetching) {
            return this._fetching;
        }
        const { key, state } = this;
        const promises = [state.fetch(key), this._when];
        return (this._fetching = Promise.all(promises).then(([value]) => {
            this._fetching = null;
            if (this._saving) {
                return;
            }
            this._state = Private.normalizeState(value, this._state);
        }));
    }
    /**
     * Handle root level layout state changes.
     */
    async _onStateChanged() {
        this._state.sizes = this.relativeSizes();
        this._state.container = this._editor.state;
        this._state.container.plugin = this._list.selection;
        try {
            await this._saveState();
        }
        catch (error) {
            console.error('Saving setting editor state failed', error);
        }
        this._setState();
    }
    /**
     * Set the state of the setting editor.
     */
    async _saveState() {
        const { key, state } = this;
        const value = this._state;
        this._saving = true;
        try {
            await state.save(key, value);
            this._saving = false;
        }
        catch (error) {
            this._saving = false;
            throw error;
        }
    }
    /**
     * Set the layout sizes.
     */
    _setLayout() {
        const editor = this._editor;
        const state = this._state;
        editor.state = state.container;
        // Allow the message queue (which includes fit requests that might disrupt
        // setting relative sizes) to clear before setting sizes.
        requestAnimationFrame(() => {
            this.setRelativeSizes(state.sizes);
        });
    }
    /**
     * Set the presets of the setting editor.
     */
    _setState() {
        const editor = this._editor;
        const list = this._list;
        const { container } = this._state;
        if (!container.plugin) {
            editor.settings = null;
            list.selection = '';
            this._setLayout();
            return;
        }
        if (editor.settings && editor.settings.id === container.plugin) {
            this._setLayout();
            return;
        }
        const instructions = this._instructions;
        this.registry
            .load(container.plugin)
            .then(settings => {
            if (instructions.isAttached) {
                instructions.parent = null;
            }
            if (!editor.isAttached) {
                this.addWidget(editor);
            }
            editor.settings = settings;
            list.selection = container.plugin;
            this._setLayout();
        })
            .catch(reason => {
            console.error(`Loading ${container.plugin} settings failed.`, reason);
            list.selection = this._state.container.plugin = '';
            editor.settings = null;
            this._setLayout();
        });
    }
}
/**
 * A namespace for private module data.
 */
var Private;
(function (Private) {
    /**
     * Return a normalized restored layout state that defaults to the presets.
     */
    function normalizeState(saved, current) {
        if (!saved) {
            return JSONExt.deepCopy(DEFAULT_LAYOUT);
        }
        if (!('sizes' in saved) || !numberArray(saved.sizes)) {
            saved.sizes = JSONExt.deepCopy(DEFAULT_LAYOUT.sizes);
        }
        if (!('container' in saved)) {
            saved.container = JSONExt.deepCopy(DEFAULT_LAYOUT.container);
            return saved;
        }
        const container = 'container' in saved &&
            saved.container &&
            typeof saved.container === 'object'
            ? saved.container
            : {};
        saved.container = {
            plugin: typeof container.plugin === 'string'
                ? container.plugin
                : DEFAULT_LAYOUT.container.plugin,
            sizes: numberArray(container.sizes)
                ? container.sizes
                : JSONExt.deepCopy(DEFAULT_LAYOUT.container.sizes)
        };
        return saved;
    }
    Private.normalizeState = normalizeState;
    /**
     * Tests whether an array consists exclusively of numbers.
     */
    function numberArray(value) {
        return Array.isArray(value) && value.every(x => typeof x === 'number');
    }
})(Private || (Private = {}));
//# sourceMappingURL=jsonsettingeditor.js.map