/* -----------------------------------------------------------------------------
| Copyright (c) Jupyter Development Team.
| Distributed under the terms of the Modified BSD License.
|----------------------------------------------------------------------------*/
import { showErrorMessage } from '@jupyterlab/apputils';
import { caretDownIcon, caretRightIcon, FormComponent } from '@jupyterlab/ui-components';
import { JSONExt } from '@lumino/coreutils';
import { Debouncer } from '@lumino/polling';
import validatorAjv8 from '@rjsf/validator-ajv8';
import React from 'react';
/**
 * Indentation to use when saving the settings as JSON document.
 */
const JSON_INDENTATION = 4;
/**
 * A React component that prepares the settings for a
 * given plugin to be rendered in the FormEditor.
 */
export class SettingsFormEditor extends React.Component {
    constructor(props) {
        super(props);
        /**
         * Handler for the "Restore to defaults" button - clears all
         * modified settings then calls `setFormData` to restore the
         * values.
         */
        this.reset = async (event) => {
            event.stopPropagation();
            for (const field in this.props.settings.user) {
                await this.props.settings.remove(field);
            }
            this._formData = this.props.settings.composite;
            this.setState({ isModified: false });
        };
        this._onChange = (e) => {
            this.props.hasError(e.errors.length !== 0);
            this._formData = e.formData;
            if (e.errors.length === 0) {
                this.props.updateDirtyState(true);
                void this._debouncer.invoke();
            }
            this.props.onSelect(this.props.settings.id);
        };
        const { settings } = props;
        this._formData = settings.composite;
        this.state = {
            isModified: settings.isModified,
            uiSchema: {},
            filteredSchema: this.props.settings.schema,
            formContext: {
                defaultFormData: this.props.settings.default(),
                settings: this.props.settings
            }
        };
        this.handleChange = this.handleChange.bind(this);
        this._debouncer = new Debouncer(this.handleChange);
    }
    componentDidMount() {
        this._setUiSchema();
        this._setFilteredSchema();
    }
    componentDidUpdate(prevProps) {
        this._setUiSchema(prevProps.renderers[prevProps.settings.id]);
        this._setFilteredSchema(prevProps.filteredValues);
        if (prevProps.settings !== this.props.settings) {
            this.setState({
                formContext: {
                    settings: this.props.settings,
                    defaultFormData: this.props.settings.default()
                }
            });
        }
    }
    componentWillUnmount() {
        this._debouncer.dispose();
    }
    /**
     * Handler for edits made in the form editor.
     */
    handleChange() {
        // Prevent unnecessary save when opening settings that haven't been modified.
        if (!this.props.settings.isModified &&
            this.props.settings.isDefault(this._formData)) {
            this.props.updateDirtyState(false);
            return;
        }
        this.props.settings
            .save(JSON.stringify(this._formData, undefined, JSON_INDENTATION))
            .then(() => {
            this.props.updateDirtyState(false);
            this.setState({ isModified: this.props.settings.isModified });
        })
            .catch((reason) => {
            this.props.updateDirtyState(false);
            const trans = this.props.translator.load('jupyterlab');
            void showErrorMessage(trans.__('Error saving settings.'), reason);
        });
    }
    render() {
        const trans = this.props.translator.load('jupyterlab');
        const icon = this.props.isCollapsed ? caretRightIcon : caretDownIcon;
        return (React.createElement("div", null,
            React.createElement("div", { className: "jp-SettingsHeader", onClick: () => {
                    this.props.onCollapseChange(!this.props.isCollapsed);
                    this.props.onSelect(this.props.settings.id);
                } },
                React.createElement("header", { className: "jp-SettingsTitle" },
                    React.createElement(icon.react, { tag: "span", elementPosition: "center", className: "jp-SettingsTitle-caret" }),
                    React.createElement("h2", null, this.props.settings.schema.title),
                    React.createElement("div", { className: "jp-SettingsHeader-description" }, this.props.settings.schema.description)),
                this.state.isModified && (React.createElement("button", { className: "jp-RestoreButton", onClick: this.reset }, trans.__('Restore to Defaults')))),
            !this.props.isCollapsed && (React.createElement(FormComponent, { validator: validatorAjv8, schema: this.state.filteredSchema, formData: this._formData, uiSchema: this.state.uiSchema, fields: this.props.renderers[this.props.settings.id], formContext: this.state.formContext, liveValidate: true, idPrefix: `jp-SettingsEditor-${this.props.settings.id}`, onChange: this._onChange, translator: this.props.translator }))));
    }
    _setUiSchema(prevRenderers) {
        var _a;
        const renderers = this.props.renderers[this.props.settings.id];
        if (!JSONExt.deepEqual(Object.keys(prevRenderers !== null && prevRenderers !== void 0 ? prevRenderers : {}).sort(), Object.keys(renderers !== null && renderers !== void 0 ? renderers : {}).sort())) {
            /**
             * Construct uiSchema to pass any custom renderers to the form editor.
             */
            const uiSchema = {};
            for (const id in this.props.renderers[this.props.settings.id]) {
                if (Object.keys((_a = this.props.settings.schema.properties) !== null && _a !== void 0 ? _a : {}).includes(id)) {
                    uiSchema[id] = {
                        'ui:field': id
                    };
                }
            }
            this.setState({ uiSchema });
        }
    }
    _setFilteredSchema(prevFilteredValues) {
        var _a, _b, _c, _d;
        if (prevFilteredValues === undefined ||
            !JSONExt.deepEqual(prevFilteredValues, this.props.filteredValues)) {
            /**
             * Only show fields that match search value.
             */
            const filteredSchema = JSONExt.deepCopy(this.props.settings.schema);
            if ((_b = (_a = this.props.filteredValues) === null || _a === void 0 ? void 0 : _a.length) !== null && _b !== void 0 ? _b : 0 > 0) {
                for (const field in filteredSchema.properties) {
                    if (!((_c = this.props.filteredValues) === null || _c === void 0 ? void 0 : _c.includes((_d = filteredSchema.properties[field].title) !== null && _d !== void 0 ? _d : field))) {
                        delete filteredSchema.properties[field];
                    }
                }
            }
            this.setState({ filteredSchema });
        }
    }
}
//# sourceMappingURL=SettingsFormEditor.js.map