/**
 * @packageDocumentation
 * @module running
 */
import { ITranslator } from '@jupyterlab/translation';
import { LabIcon, SidePanel } from '@jupyterlab/ui-components';
import { Token } from '@lumino/coreutils';
import { IDisposable } from '@lumino/disposable';
import { ISignal } from '@lumino/signaling';
/**
 * The running sessions token.
 */
export declare const IRunningSessionManagers: Token<IRunningSessionManagers>;
/**
 * The running interface.
 */
export interface IRunningSessionManagers {
    /**
     * Add a running item manager.
     *
     * @param manager - The running item manager.
     *
     */
    add(manager: IRunningSessions.IManager): IDisposable;
    /**
     * Signal emitted when a new manager is added.
     */
    added: ISignal<IRunningSessionManagers, IRunningSessions.IManager>;
    /**
     * Return an array of managers.
     */
    items(): ReadonlyArray<IRunningSessions.IManager>;
}
export declare class RunningSessionManagers implements IRunningSessionManagers {
    /**
     * Signal emitted when a new manager is added.
     */
    get added(): ISignal<this, IRunningSessions.IManager>;
    /**
     * Add a running item manager.
     *
     * @param manager - The running item manager.
     *
     */
    add(manager: IRunningSessions.IManager): IDisposable;
    /**
     * Return an iterator of launcher items.
     */
    items(): ReadonlyArray<IRunningSessions.IManager>;
    private _added;
    private _managers;
}
/**
 * A class that exposes the running terminal and kernel sessions.
 */
export declare class RunningSessions extends SidePanel {
    /**
     * Construct a new running widget.
     */
    constructor(managers: IRunningSessionManagers, translator?: ITranslator);
    /**
     * Dispose the resources held by the widget
     */
    dispose(): void;
    /**
     * Add a section for a new manager.
     *
     * @param managers Managers
     * @param manager New manager
     */
    protected addSection(_: unknown, manager: IRunningSessions.IManager): void;
    protected managers: IRunningSessionManagers;
    protected translator: ITranslator;
}
/**
 * The namespace for the `IRunningSessions` class statics.
 */
export declare namespace IRunningSessions {
    /**
     * A manager of running items grouped under a single section.
     */
    interface IManager {
        /**
         * Name that is shown to the user in plural.
         */
        name: string;
        /**
         * Called when the shutdown all button is pressed.
         */
        shutdownAll(): void;
        /**
         * List the running models.
         */
        running(): IRunningItem[];
        /**
         * Force a refresh of the running models.
         */
        refreshRunning(): void;
        /**
         * A signal that should be emitted when the item list has changed.
         */
        runningChanged: ISignal<any, any>;
        /**
         * A string used to describe the shutdown action.
         */
        shutdownLabel?: string;
        /**
         * A string used to describe the shutdown all action.
         */
        shutdownAllLabel?: string;
        /**
         * A string used as the body text in the shutdown all confirmation dialog.
         */
        shutdownAllConfirmationText?: string;
        /**
         * The icon to show for shutting down an individual item in this section.
         */
        shutdownItemIcon?: LabIcon;
    }
    /**
     * A running item.
     */
    interface IRunningItem {
        /**
         * Optional child nodes that belong to a top-level running item.
         */
        children?: IRunningItem[];
        /**
         * Optional CSS class name to add to the running item.
         */
        className?: string;
        /**
         * Optional context hint to add to the `data-context` attribute of an item.
         */
        context?: string;
        /**
         * Called when the running item is clicked.
         */
        open?: () => void;
        /**
         * Called when the shutdown button is pressed on a particular item.
         */
        shutdown?: () => void;
        /**
         * The `LabIcon` to use as the icon for the running item or the string
         * `src` URL.
         */
        icon: () => LabIcon | string;
        /**
         * Called to determine the label for each item.
         */
        label: () => string;
        /**
         * Called to determine the `title` attribute for each item, which is
         * revealed on hover.
         */
        labelTitle?: () => string;
        /**
         * Called to determine the `detail` attribute, which is shown optionally in
         * a column after the label.
         */
        detail?: () => string;
    }
}
