import { IChangedArgs } from '@jupyterlab/coreutils';
import { IObservableList } from '@jupyterlab/observables';
import { IDisposable } from '@lumino/disposable';
import { Message } from '@lumino/messaging';
import { ISignal, Signal } from '@lumino/signaling';
import { PanelLayout, Widget } from '@lumino/widgets';
/**
 * Windowed list abstract model.
 */
export declare abstract class WindowedListModel implements WindowedList.IModel {
    /**
     * Constructor
     *
     * @param options Constructor options
     */
    constructor(options?: WindowedList.IModelOptions);
    /**
     * Provide a best guess for the widget size at position index
     *
     * #### Notes
     *
     * This function should be very light to compute especially when
     * returning the default size.
     * The default value should be constant (i.e. two calls with `null` should
     * return the same value). But it can change for a given `index`.
     *
     * @param index Widget position
     * @returns Estimated widget size
     */
    abstract estimateWidgetSize: (index: number) => number;
    /**
     * Widget factory for the list items.
     *
     * Caching the resulting widgets should be done by the callee.
     *
     * @param index List index
     * @returns The widget at the given position
     */
    abstract widgetRenderer: (index: number) => Widget;
    /**
     * List widget height
     */
    get height(): number;
    set height(h: number);
    /**
     * Test whether the model is disposed.
     */
    get isDisposed(): boolean;
    /**
     * Items list to be rendered
     */
    get itemsList(): IObservableList<any> | null;
    set itemsList(v: IObservableList<any> | null);
    /**
     * Number of widgets to render in addition to those
     * visible in the viewport.
     */
    get overscanCount(): number;
    set overscanCount(newValue: number);
    /**
     * Viewport scroll offset.
     */
    get scrollOffset(): number;
    set scrollOffset(offset: number);
    /**
     * Total number of widgets in the list
     */
    get widgetCount(): number;
    set widgetCount(newValue: number);
    /**
     * Whether windowing is active or not.
     *
     * This is true by default.
     */
    get windowingActive(): boolean;
    set windowingActive(newValue: boolean);
    /**
     * A signal emitted when any model state changes.
     */
    get stateChanged(): ISignal<WindowedListModel, IChangedArgs<any, any, 'count' | 'index' | 'list' | 'overscanCount' | 'windowingActive'>>;
    /**
     * Dispose the model.
     */
    dispose(): void;
    /**
     * Get the total list size.
     *
     * @returns Total estimated size
     */
    getEstimatedTotalSize(): number;
    /**
     * Get the scroll offset to display an item in the viewport.
     *
     * @param index Item index
     * @param align Where to align the item in the viewport
     * @returns The needed scroll offset
     */
    getOffsetForIndexAndAlignment(index: number, align?: WindowedList.ScrollToAlign): number;
    /**
     * Compute the items range to display.
     *
     * It returns ``null`` if the range does not need to be updated.
     *
     * @returns The current items range to display
     */
    getRangeToRender(): WindowedList.WindowIndex | null;
    /**
     * Return the viewport top position and height for range spanning from
     * ``startIndex`` to ``stopIndex``.
     *
     * @param startIndex First item in viewport index
     * @param stopIndex Last item in viewport index
     * @returns The viewport top position and its height
     */
    getSpan(startIndex: number, stopIndex: number): [number, number];
    /**
     * WindowedListModel caches offsets and measurements for each index for performance purposes.
     * This method clears that cached data for all items after (and including) the specified index.
     *
     * The list will automatically re-render after the index is reset.
     *
     * @param index
     */
    resetAfterIndex(index: number): void;
    /**
     * Update item sizes.
     *
     * This should be called when the real item sizes has been
     * measured.
     *
     * @param sizes New sizes per item index
     * @returns Whether some sizes changed or not
     */
    setWidgetSize(sizes: {
        index: number;
        size: number;
    }[]): boolean;
    /**
     * Callback on list changes
     *
     * @param list List items
     * @param changes List change
     */
    protected onListChanged(list: IObservableList<Widget>, changes: IObservableList.IChangedArgs<Widget>): void;
    private _getItemMetadata;
    private _findNearestItem;
    private _findNearestItemBinarySearch;
    private _findNearestItemExponentialSearch;
    private _getRangeToRender;
    private _getStartIndexForOffset;
    private _getStopIndexForStartIndex;
    /**
     * Default widget size estimation
     */
    protected _estimatedWidgetSize: number;
    protected _stateChanged: Signal<WindowedListModel, IChangedArgs<any, any, "list" | "index" | "count" | "overscanCount" | "windowingActive">>;
    private _currentWindow;
    private _height;
    private _isDisposed;
    private _itemsList;
    private _lastMeasuredIndex;
    private _overscanCount;
    private _scrollOffset;
    private _widgetCount;
    private _widgetSizers;
    private _windowingActive;
}
/**
 * Windowed list widget
 */
export declare class WindowedList<T extends WindowedList.IModel = WindowedList.IModel> extends Widget {
    /**
     * Default widget size
     */
    static readonly DEFAULT_WIDGET_SIZE = 50;
    /**
     * Constructor
     *
     * @param options Constructor options
     */
    constructor(options: WindowedList.IOptions<T>);
    /**
     * Widget layout
     */
    get layout(): WindowedLayout;
    /**
     * Viewport
     */
    get viewportNode(): HTMLDivElement;
    /**
     * Windowed list view model
     */
    protected get viewModel(): T;
    /**
     * Callback on event.
     *
     * @param event Event
     */
    handleEvent(event: Event): void;
    /**
     * Scroll to the specified offset `scrollTop`.
     *
     * @param scrollOffset Offset to scroll
     *
     * @deprecated since v4 This is an internal helper. Prefer calling `scrollToItem`.
     */
    scrollTo(scrollOffset: number): void;
    /**
     * Scroll to the specified item.
     *
     * By default, the List will scroll as little as possible to ensure the item is visible. You can control the alignment of the item though by specifying a second alignment parameter. Acceptable values are:
     *
     *   auto (default) - Scroll as little as possible to ensure the item is visible. (If the item is already visible, it won't scroll at all.)
     *   smart - If the item is already visible, don't scroll at all. If it is less than one viewport away, scroll as little as possible so that it becomes visible. If it is more than one viewport away, scroll so that it is centered within the list.
     *   center - Center align the item within the list.
     *   end - Align the item to the end of the list
     *   start - Align the item to the beginning of the list
     *
     * @param index Item index to scroll to
     * @param align Type of alignment
     */
    scrollToItem(index: number, align?: WindowedList.ScrollToAlign): Promise<void>;
    /**
     * A message handler invoked on an `'after-attach'` message.
     */
    protected onAfterAttach(msg: Message): void;
    /**
     * A message handler invoked on an `'before-detach'` message.
     */
    protected onBeforeDetach(msg: Message): void;
    /**
     * Callback on scroll event
     *
     * @param event Scroll event
     */
    protected onScroll(event: Event): void;
    /**
     * A message handler invoked on an `'resize-request'` message.
     */
    protected onResize(msg: Widget.ResizeMessage): void;
    /**
     * Callback on view model change
     *
     * @param model Windowed list model
     * @param changes Change
     */
    protected onStateChanged(model: WindowedList.IModel, changes: IChangedArgs<number | boolean, number | boolean, string>): void;
    /**
     * A message handler invoked on an `'update-request'` message.
     *
     * #### Notes
     * The default implementation of this handler is a no-op.
     */
    protected onUpdateRequest(msg: Message): void;
    private _addListeners;
    private _removeListeners;
    private _update;
    private _onWidgetResize;
    private _resetScrollToItem;
    protected _viewModel: T;
    private _innerElement;
    private _isScrolling;
    private _windowElement;
    private _resetScrollToItemTimeout;
    private _resizeObserver;
    private _scrollRepaint;
    private _scrollToItem;
    private _scrollUpdateWasRequested;
}
/**
 * Customized layout for windowed list container.
 */
export declare class WindowedLayout extends PanelLayout {
    /**
     * Constructor
     */
    constructor();
    /**
     * Specialized parent type definition
     */
    get parent(): WindowedList | null;
    set parent(value: WindowedList | null);
    /**
     * Attach a widget to the parent's DOM node.
     *
     * @param index - The current index of the widget in the layout.
     *
     * @param widget - The widget to attach to the parent.
     *
     * #### Notes
     * This method is called automatically by the panel layout at the
     * appropriate time. It should not be called directly by user code.
     *
     * The default implementation adds the widgets's node to the parent's
     * node at the proper location, and sends the appropriate attach
     * messages to the widget if the parent is attached to the DOM.
     *
     * Subclasses may reimplement this method to control how the widget's
     * node is added to the parent's node.
     */
    protected attachWidget(index: number, widget: Widget): void;
    /**
     * Detach a widget from the parent's DOM node.
     *
     * @param index - The previous index of the widget in the layout.
     *
     * @param widget - The widget to detach from the parent.
     *
     * #### Notes
     * This method is called automatically by the panel layout at the
     * appropriate time. It should not be called directly by user code.
     *
     * The default implementation removes the widget's node from the
     * parent's node, and sends the appropriate detach messages to the
     * widget if the parent is attached to the DOM.
     *
     * Subclasses may reimplement this method to control how the widget's
     * node is removed from the parent's node.
     */
    protected detachWidget(index: number, widget: Widget): void;
    /**
     * Move a widget in the parent's DOM node.
     *
     * @param fromIndex - The previous index of the widget in the layout.
     *
     * @param toIndex - The current index of the widget in the layout.
     *
     * @param widget - The widget to move in the parent.
     *
     * #### Notes
     * This method is called automatically by the panel layout at the
     * appropriate time. It should not be called directly by user code.
     *
     * The default implementation moves the widget's node to the proper
     * location in the parent's node and sends the appropriate attach and
     * detach messages to the widget if the parent is attached to the DOM.
     *
     * Subclasses may reimplement this method to control how the widget's
     * node is moved in the parent's node.
     */
    protected moveWidget(fromIndex: number, toIndex: number, widget: Widget): void;
    /**
     * A message handler invoked on an `'update-request'` message.
     *
     * #### Notes
     * This is a reimplementation of the base class method,
     * and is a no-op.
     */
    protected onUpdateRequest(msg: Message): void;
}
/**
 * A namespace for windowed list
 */
export declare namespace WindowedList {
    /**
     * Windowed list model interface
     */
    interface IModel extends IDisposable {
        /**
         * Provide a best guess for the widget size at position index
         *
         * #### Notes
         *
         * This function should be very light to compute especially when
         * returning the default size.
         * The default value should be constant (i.e. two calls with `null` should
         * return the same value). But it can change for a given `index`.
         *
         * @param index Widget position
         * @returns Estimated widget size
         */
        estimateWidgetSize: (index: number) => number;
        /**
         * Get the total list size.
         *
         * @returns Total estimated size
         */
        getEstimatedTotalSize(): number;
        /**
         * Get the scroll offset to display an item in the viewport.
         *
         * @param index Item index
         * @param align Where to align the item in the viewport
         * @returns The needed scroll offset
         */
        getOffsetForIndexAndAlignment(index: number, align: ScrollToAlign): number;
        /**
         * Compute the items range to display.
         *
         * It returns ``null`` if the range does not need to be updated.
         *
         * @returns The current items range to display
         */
        getRangeToRender(): WindowIndex | null;
        /**
         * Return the viewport top position and height for range spanning from
         * ``startIndex`` to ``stopIndex``.
         *
         * @param start First item in viewport index
         * @param stop Last item in viewport index
         * @returns The viewport top position and its height
         */
        getSpan(start: number, stop: number): [number, number];
        /**
         * List widget height
         */
        height: number;
        /**
         * Items list to be rendered
         */
        itemsList: {
            length: number;
            changed: ISignal<any, IObservableList.IChangedArgs<any>>;
        } | null;
        /**
         * Number of widgets to render in addition to those
         * visible in the viewport.
         */
        overscanCount: number;
        /**
         * WindowedListModel caches offsets and measurements for each index for performance purposes.
         * This method clears that cached data for all items after (and including) the specified index.
         *
         * The list will automatically re-render after the index is reset.
         *
         * @param index
         */
        resetAfterIndex(index: number): void;
        /**
         * Viewport scroll offset.
         */
        scrollOffset: number;
        /**
         * Update item sizes.
         *
         * This should be called when the real item sizes has been
         * measured.
         *
         * @param sizes New sizes per item index
         * @returns Whether some sizes changed or not
         */
        setWidgetSize(sizes: {
            index: number;
            size: number;
        }[]): boolean;
        /**
         * A signal emitted when any model state changes.
         */
        readonly stateChanged: ISignal<IModel, IChangedArgs<any, any, 'count' | 'index' | 'list' | 'overscanCount' | 'windowingActive'>>;
        /**
         * Total number of widgets in the list
         */
        widgetCount: number;
        /**
         * Whether windowing is active or not.
         */
        windowingActive: boolean;
        /**
         * Widget factory for the list items.
         *
         * Caching the resulting widgets should be done by the callee.
         *
         * @param index List index
         * @returns The widget at the given position
         */
        widgetRenderer: (index: number) => Widget;
    }
    /**
     * Windowed list model constructor options
     */
    interface IModelOptions {
        /**
         * Total number of widgets in the list.
         *
         * #### Notes
         * If an observable list is provided this will be ignored.
         */
        count?: number;
        /**
         * Dynamic list of items
         */
        itemsList?: IObservableList<any>;
        /**
         * Number of widgets to render in addition to those
         * visible in the viewport.
         */
        overscanCount?: number;
        /**
         * Whether windowing is active or not.
         *
         * This is true by default.
         */
        windowingActive?: boolean;
    }
    /**
     * Windowed list view constructor options
     */
    interface IOptions<T extends WindowedList.IModel = WindowedList.IModel> {
        /**
         * Windowed list model to display
         */
        model: T;
        /**
         * Windowed list layout
         */
        layout?: WindowedLayout;
    }
    /**
     * Item list metadata
     */
    type ItemMetadata = {
        /**
         * Item vertical offset in the container
         */
        offset: number;
        /**
         * Item height
         */
        size: number;
        /**
         * Whether the size is an estimation or a measurement.
         */
        measured?: boolean;
    };
    /**
     * Type of scroll
     */
    type ScrollToAlign = 'auto' | 'smart' | 'center' | 'start' | 'end';
    /**
     * Widget range in view port
     */
    type WindowIndex = [number, number, number, number];
}
