import { ReactNode, FormEvent, ElementType, Ref, Component, RefObject, ComponentType } from 'react';
import { StrictRJSFSchema, RJSFSchema, FormContextType, ValidatorType, UiSchema, RegistryFieldsType, TemplatesType, RegistryWidgetsType, RJSFValidationError, CustomValidator, ErrorSchema, ErrorTransformer, Registry, IdSchema, SchemaUtilsType, ValidationData, PathSchema } from '@rjsf/utils';

/** The properties that are passed to the `Form` */
interface FormProps<T = any, S extends StrictRJSFSchema = RJSFSchema, F extends FormContextType = any> {
    /** The JSON schema object for the form */
    schema: S;
    /** An implementation of the `ValidatorType` interface that is needed for form validation to work */
    validator: ValidatorType<T, S, F>;
    /** The optional children for the form, if provided, it will replace the default `SubmitButton` */
    children?: ReactNode;
    /** The uiSchema for the form */
    uiSchema?: UiSchema<T, S, F>;
    /** The data for the form, used to prefill a form with existing data */
    formData?: T;
    /** You can provide a `formContext` object to the form, which is passed down to all fields and widgets. Useful for
     * implementing context aware fields and widgets.
     *
     * NOTE: Setting `{readonlyAsDisabled: false}` on the formContext will make the antd theme treat readOnly fields as
     * disabled.
     */
    formContext?: F;
    /** To avoid collisions with existing ids in the DOM, it is possible to change the prefix used for ids;
     * Default is `root`
     */
    idPrefix?: string;
    /** To avoid using a path separator that is present in field names, it is possible to change the separator used for
     * ids (Default is `_`)
     */
    idSeparator?: string;
    /** It's possible to disable the whole form by setting the `disabled` prop. The `disabled` prop is then forwarded down
     * to each field of the form. If you just want to disable some fields, see the `ui:disabled` parameter in `uiSchema`
     */
    disabled?: boolean;
    /** It's possible to make the whole form read-only by setting the `readonly` prop. The `readonly` prop is then
     * forwarded down to each field of the form. If you just want to make some fields read-only, see the `ui:readonly`
     * parameter in `uiSchema`
     */
    readonly?: boolean;
    /** The dictionary of registered fields in the form */
    fields?: RegistryFieldsType<T, S, F>;
    /** The dictionary of registered templates in the form; Partial allows a subset to be provided beyond the defaults */
    templates?: Partial<Omit<TemplatesType<T, S, F>, 'ButtonTemplates'>> & {
        ButtonTemplates?: Partial<TemplatesType<T, S, F>['ButtonTemplates']>;
    };
    /** The dictionary of registered widgets in the form */
    widgets?: RegistryWidgetsType<T, S, F>;
    /** If you plan on being notified every time the form data are updated, you can pass an `onChange` handler, which will
     * receive the same args as `onSubmit` any time a value is updated in the form. Can also return the `id` of the field
     * that caused the change
     */
    onChange?: (data: IChangeEvent<T, S, F>, id?: string) => void;
    /** To react when submitted form data are invalid, pass an `onError` handler. It will be passed the list of
     * encountered errors
     */
    onError?: (errors: RJSFValidationError[]) => void;
    /** You can pass a function as the `onSubmit` prop of your `Form` component to listen to when the form is submitted
     * and its data are valid. It will be passed a result object having a `formData` attribute, which is the valid form
     * data you're usually after. The original event will also be passed as a second parameter
     */
    onSubmit?: (data: IChangeEvent<T, S, F>, event: FormEvent<any>) => void;
    /** Sometimes you may want to trigger events or modify external state when a field has been touched, so you can pass
     * an `onBlur` handler, which will receive the id of the input that was blurred and the field value
     */
    onBlur?: (id: string, data: any) => void;
    /** Sometimes you may want to trigger events or modify external state when a field has been focused, so you can pass
     * an `onFocus` handler, which will receive the id of the input that is focused and the field value
     */
    onFocus?: (id: string, data: any) => void;
    /** The value of this prop will be passed to the `accept-charset` HTML attribute on the form */
    acceptcharset?: string;
    /** The value of this prop will be passed to the `action` HTML attribute on the form
     *
     * NOTE: this just renders the `action` attribute in the HTML markup. There is no real network request being sent to
     * this `action` on submit. Instead, react-jsonschema-form catches the submit event with `event.preventDefault()`
     * and then calls the `onSubmit` function, where you could send a request programmatically with `fetch` or similar.
     */
    action?: string;
    /** The value of this prop will be passed to the `autocomplete` HTML attribute on the form */
    autoComplete?: string;
    /** The value of this prop will be passed to the `class` HTML attribute on the form */
    className?: string;
    /** The value of this prop will be passed to the `enctype` HTML attribute on the form */
    enctype?: string;
    /** The value of this prop will be passed to the `id` HTML attribute on the form */
    id?: string;
    /** The value of this prop will be passed to the `name` HTML attribute on the form */
    name?: string;
    /** The value of this prop will be passed to the `method` HTML attribute on the form */
    method?: string;
    /** It's possible to change the default `form` tag name to a different HTML tag, which can be helpful if you are
     * nesting forms. However, native browser form behaviour, such as submitting when the `Enter` key is pressed, may no
     * longer work
     */
    tagName?: ElementType;
    /** The value of this prop will be passed to the `target` HTML attribute on the form */
    target?: string;
    /** Formerly the `validate` prop; Takes a function that specifies custom validation rules for the form */
    customValidate?: CustomValidator<T, S, F>;
    /** This prop allows passing in custom errors that are augmented with the existing JSON Schema errors on the form; it
     * can be used to implement asynchronous validation
     */
    extraErrors?: ErrorSchema<T>;
    /** If set to true, turns off HTML5 validation on the form; Set to `false` by default */
    noHtml5Validate?: boolean;
    /** If set to true, turns off all validation. Set to `false` by default
     *
     * @deprecated - In a future release, this switch may be replaced by making `validator` prop optional
     */
    noValidate?: boolean;
    /** If set to true, the form will perform validation and show any validation errors whenever the form data is changed,
     * rather than just on submit
     */
    liveValidate?: boolean;
    /** If `omitExtraData` and `liveOmit` are both set to true, then extra form data values that are not in any form field
     * will be removed whenever `onChange` is called. Set to `false` by default
     */
    liveOmit?: boolean;
    /** If set to true, then extra form data values that are not in any form field will be removed whenever `onSubmit` is
     * called. Set to `false` by default.
     */
    omitExtraData?: boolean;
    /** When this prop is set to `top` or 'bottom', a list of errors (or the custom error list defined in the `ErrorList`) will also
     * show. When set to false, only inline input validation errors will be shown. Set to `top` by default
     */
    showErrorList?: false | 'top' | 'bottom';
    /** A function can be passed to this prop in order to make modifications to the default errors resulting from JSON
     * Schema validation
     */
    transformErrors?: ErrorTransformer<T, S, F>;
    /** If set to true, then the first field with an error will receive the focus when the form is submitted with errors
     */
    focusOnFirstError?: boolean;
    /** Optional string translation function, if provided, allows users to change the translation of the RJSF internal
     * strings. Some strings contain replaceable parameter values as indicated by `%1`, `%2`, etc. The number after the
     * `%` indicates the order of the parameter. The ordering of parameters is important because some languages may choose
     * to put the second parameter before the first in its translation.
     */
    translateString?: Registry['translateString'];
    /**
     * _internalFormWrapper is currently used by the semantic-ui theme to provide a custom wrapper around `<Form />`
     * that supports the proper rendering of those themes. To use this prop, one must pass a component that takes two
     * props: `children` and `as`. That component, at minimum, should render the `children` inside of a <form /> tag
     * unless `as` is provided, in which case, use the `as` prop in place of `<form />`.
     * i.e.:
     * ```
     * export default function InternalForm({ children, as }) {
     *   const FormTag = as || 'form';
     *   return <FormTag>{children}</FormTag>;
     * }
     * ```
     *
     * Use at your own risk as this prop is private and may change at any time without notice.
     */
    _internalFormWrapper?: ElementType;
    /** Support receiving a React ref to the Form
     */
    ref?: Ref<Form<T, S, F>>;
}
/** The data that is contained within the state for the `Form` */
interface FormState<T = any, S extends StrictRJSFSchema = RJSFSchema, F extends FormContextType = any> {
    /** The JSON schema object for the form */
    schema: S;
    /** The uiSchema for the form */
    uiSchema: UiSchema<T, S, F>;
    /** The `IdSchema` for the form, computed from the `schema`, the `rootFieldId`, the `formData` and the `idPrefix` and
     * `idSeparator` props.
     */
    idSchema: IdSchema<T>;
    /** The schemaUtils implementation used by the `Form`, created from the `validator` and the `schema` */
    schemaUtils: SchemaUtilsType<T, S, F>;
    /** The current data for the form, computed from the `formData` prop and the changes made by the user */
    formData?: T;
    /** Flag indicating whether the form is in edit mode, true when `formData` is passed to the form, otherwise false */
    edit: boolean;
    /** The current list of errors for the form, includes `extraErrors` */
    errors: RJSFValidationError[];
    /** The current errors, in `ErrorSchema` format, for the form, includes `extraErrors` */
    errorSchema: ErrorSchema<T>;
    /** The current list of errors for the form directly from schema validation, does NOT include `extraErrors` */
    schemaValidationErrors: RJSFValidationError[];
    /** The current errors, in `ErrorSchema` format, for the form directly from schema validation, does NOT include
     * `extraErrors`
     */
    schemaValidationErrorSchema: ErrorSchema<T>;
}
/** The event data passed when changes have been made to the form, includes everything from the `FormState` except
 * the schema validation errors. An additional `status` is added when returned from `onSubmit`
 */
interface IChangeEvent<T = any, S extends StrictRJSFSchema = RJSFSchema, F extends FormContextType = any> extends Omit<FormState<T, S, F>, 'schemaValidationErrors' | 'schemaValidationErrorSchema'> {
    /** The status of the form when submitted */
    status?: 'submitted';
}
/** The `Form` component renders the outer form and all the fields defined in the `schema` */
declare class Form<T = any, S extends StrictRJSFSchema = RJSFSchema, F extends FormContextType = any> extends Component<FormProps<T, S, F>, FormState<T, S, F>> {
    /** The ref used to hold the `form` element, this needs to be `any` because `tagName` or `_internalFormWrapper` can
     * provide any possible type here
     */
    formElement: RefObject<any>;
    /** Constructs the `Form` from the `props`. Will setup the initial state from the props. It will also call the
     * `onChange` handler if the initially provided `formData` is modified to add missing default values as part of the
     * state construction.
     *
     * @param props - The initial props for the `Form`
     */
    constructor(props: FormProps<T, S, F>);
    /** React lifecycle method that gets called before new props are provided, updates the state based on new props. It
     * will also call the`onChange` handler if the `formData` is modified to add missing default values as part of the
     * state construction.
     *
     * @param nextProps - The new set of props about to be applied to the `Form`
     */
    UNSAFE_componentWillReceiveProps(nextProps: FormProps<T, S, F>): void;
    /** Extracts the updated state from the given `props` and `inputFormData`. As part of this process, the
     * `inputFormData` is first processed to add any missing required defaults. After that, the data is run through the
     * validation process IF required by the `props`.
     *
     * @param props - The props passed to the `Form`
     * @param inputFormData - The new or current data for the `Form`
     * @returns - The new state for the `Form`
     */
    getStateFromProps(props: FormProps<T, S, F>, inputFormData?: T): FormState<T, S, F>;
    /** React lifecycle method that is used to determine whether component should be updated.
     *
     * @param nextProps - The next version of the props
     * @param nextState - The next version of the state
     * @returns - True if the component should be updated, false otherwise
     */
    shouldComponentUpdate(nextProps: FormProps<T, S, F>, nextState: FormState<T, S, F>): boolean;
    /** Validates the `formData` against the `schema` using the `altSchemaUtils` (if provided otherwise it uses the
     * `schemaUtils` in the state), returning the results.
     *
     * @param formData - The new form data to validate
     * @param schema - The schema used to validate against
     * @param altSchemaUtils - The alternate schemaUtils to use for validation
     */
    validate(formData: T | undefined, schema?: S, altSchemaUtils?: SchemaUtilsType<T, S, F>): ValidationData<T>;
    /** Renders any errors contained in the `state` in using the `ErrorList`, if not disabled by `showErrorList`. */
    renderErrors(registry: Registry<T, S, F>): JSX.Element | null;
    /** Returns the `formData` with only the elements specified in the `fields` list
     *
     * @param formData - The data for the `Form`
     * @param fields - The fields to keep while filtering
     */
    getUsedFormData: (formData: T | undefined, fields: string[][]) => T | undefined;
    /** Returns the list of field names from inspecting the `pathSchema` as well as using the `formData`
     *
     * @param pathSchema - The `PathSchema` object for the form
     * @param [formData] - The form data to use while checking for empty objects/arrays
     */
    getFieldNames: (pathSchema: PathSchema<T>, formData?: T) => string[][];
    /** Function to handle changes made to a field in the `Form`. This handler receives an entirely new copy of the
     * `formData` along with a new `ErrorSchema`. It will first update the `formData` with any missing default fields and
     * then, if `omitExtraData` and `liveOmit` are turned on, the `formData` will be filterer to remove any extra data not
     * in a form field. Then, the resulting formData will be validated if required. The state will be updated with the new
     * updated (potentially filtered) `formData`, any errors that resulted from validation. Finally the `onChange`
     * callback will be called if specified with the updated state.
     *
     * @param formData - The new form data from a change to a field
     * @param newErrorSchema - The new `ErrorSchema` based on the field change
     * @param id - The id of the field that caused the change
     */
    onChange: (formData: T | undefined, newErrorSchema?: ErrorSchema<T>, id?: string) => void;
    /** Callback function to handle when a field on the form is blurred. Calls the `onBlur` callback for the `Form` if it
     * was provided.
     *
     * @param id - The unique `id` of the field that was blurred
     * @param data - The data associated with the field that was blurred
     */
    onBlur: (id: string, data: any) => void;
    /** Callback function to handle when a field on the form is focused. Calls the `onFocus` callback for the `Form` if it
     * was provided.
     *
     * @param id - The unique `id` of the field that was focused
     * @param data - The data associated with the field that was focused
     */
    onFocus: (id: string, data: any) => void;
    /** Callback function to handle when the form is submitted. First, it prevents the default event behavior. Nothing
     * happens if the target and currentTarget of the event are not the same. It will omit any extra data in the
     * `formData` in the state if `omitExtraData` is true. It will validate the resulting `formData`, reporting errors
     * via the `onError()` callback unless validation is disabled. Finally it will add in any `extraErrors` and then call
     * back the `onSubmit` callback if it was provided.
     *
     * @param event - The submit HTML form event
     */
    onSubmit: (event: FormEvent<any>) => void;
    /** Returns the registry for the form */
    getRegistry(): Registry<T, S, F>;
    /** Provides a function that can be used to programmatically submit the `Form` */
    submit(): void;
    /** Attempts to focus on the field associated with the `error`. Uses the `property` field to compute path of the error
     * field, then, using the `idPrefix` and `idSeparator` converts that path into an id. Then the input element with that
     * id is attempted to be found using the `formElement` ref. If it is located, then it is focused.
     *
     * @param error - The error on which to focus
     */
    focusOnError(error: RJSFValidationError): void;
    /** Programmatically validate the form. If `onError` is provided, then it will be called with the list of errors the
     * same way as would happen on form submission.
     *
     * @returns - True if the form is valid, false otherwise.
     */
    validateForm(): boolean;
    /** Renders the `Form` fields inside the <form> | `tagName` or `_internalFormWrapper`, rendering any errors if
     * needed along with the submit button or any children of the form.
     */
    render(): JSX.Element;
}

/** The properties for the `withTheme` function, essentially a subset of properties from the `FormProps` that can be
 * overridden while creating a theme
 */
type ThemeProps<T = any, S extends StrictRJSFSchema = RJSFSchema, F extends FormContextType = any> = Pick<FormProps<T, S, F>, 'fields' | 'templates' | 'widgets' | '_internalFormWrapper'>;
/** A Higher-Order component that creates a wrapper around a `Form` with the overrides from the `WithThemeProps` */
declare function withTheme<T = any, S extends StrictRJSFSchema = RJSFSchema, F extends FormContextType = any>(themeProps: ThemeProps<T, S, F>): ComponentType<FormProps<T, S, F>>;

/** The default registry consists of all the fields, templates and widgets provided in the core implementation,
 * plus an empty `rootSchema` and `formContext. We omit schemaUtils here because it cannot be defaulted without a
 * rootSchema and validator. It will be added into the computed registry later in the Form.
 */
declare function getDefaultRegistry<T = any, S extends StrictRJSFSchema = RJSFSchema, F extends FormContextType = any>(): Omit<Registry<T, S, F>, 'schemaUtils'>;

export { FormProps, FormState, IChangeEvent, ThemeProps, Form as default, getDefaultRegistry, withTheme };
