// Copyright (c) Jupyter Development Team.
// Distributed under the terms of the Modified BSD License.
import { PageConfig } from '@jupyterlab/coreutils';
import { Base64ModelFactory } from '@jupyterlab/docregistry';
import { ServiceManager } from '@jupyterlab/services';
import { Token } from '@lumino/coreutils';
import { JupyterFrontEnd } from './frontend';
import { createRendermimePlugins } from './mimerenderers';
import { LabShell } from './shell';
import { LabStatus } from './status';
/**
 * JupyterLab is the main application class. It is instantiated once and shared.
 */
export class JupyterLab extends JupyterFrontEnd {
    /**
     * Construct a new JupyterLab object.
     */
    constructor(options = { shell: new LabShell() }) {
        super({
            ...options,
            shell: options.shell || new LabShell(),
            serviceManager: options.serviceManager ||
                new ServiceManager({
                    standby: () => {
                        return !this._info.isConnected || 'when-hidden';
                    }
                })
        });
        /**
         * The name of the JupyterLab application.
         */
        this.name = PageConfig.getOption('appName') || 'JupyterLab';
        /**
         * A namespace/prefix plugins may use to denote their provenance.
         */
        this.namespace = PageConfig.getOption('appNamespace') || this.name;
        /**
         * A list of all errors encountered when registering plugins.
         */
        this.registerPluginErrors = [];
        /**
         * The application busy and dirty status signals and flags.
         */
        this.status = new LabStatus(this);
        /**
         * The version of the JupyterLab application.
         */
        this.version = PageConfig.getOption('appVersion') || 'unknown';
        this._info = JupyterLab.defaultInfo;
        this.restored = this.shell.restored
            .then(() => undefined)
            .catch(() => undefined);
        // Create an IInfo dictionary from the options to override the defaults.
        const info = Object.keys(JupyterLab.defaultInfo).reduce((acc, val) => {
            if (val in options) {
                acc[val] = JSON.parse(JSON.stringify(options[val]));
            }
            return acc;
        }, {});
        // Populate application info.
        this._info = { ...JupyterLab.defaultInfo, ...info };
        // Populate application paths override the defaults if necessary.
        const defaultURLs = JupyterLab.defaultPaths.urls;
        const defaultDirs = JupyterLab.defaultPaths.directories;
        const optionURLs = (options.paths && options.paths.urls) || {};
        const optionDirs = (options.paths && options.paths.directories) || {};
        this._paths = {
            urls: Object.keys(defaultURLs).reduce((acc, key) => {
                if (key in optionURLs) {
                    const value = optionURLs[key];
                    acc[key] = value;
                }
                else {
                    acc[key] = defaultURLs[key];
                }
                return acc;
            }, {}),
            directories: Object.keys(JupyterLab.defaultPaths.directories).reduce((acc, key) => {
                if (key in optionDirs) {
                    const value = optionDirs[key];
                    acc[key] = value;
                }
                else {
                    acc[key] = defaultDirs[key];
                }
                return acc;
            }, {})
        };
        if (this._info.devMode) {
            this.shell.addClass('jp-mod-devMode');
        }
        // Add initial model factory.
        this.docRegistry.addModelFactory(new Base64ModelFactory());
        if (options.mimeExtensions) {
            for (const plugin of createRendermimePlugins(options.mimeExtensions)) {
                this.registerPlugin(plugin);
            }
        }
    }
    /**
     * The JupyterLab application information dictionary.
     */
    get info() {
        return this._info;
    }
    /**
     * The JupyterLab application paths dictionary.
     */
    get paths() {
        return this._paths;
    }
    /**
     * Register plugins from a plugin module.
     *
     * @param mod - The plugin module to register.
     */
    registerPluginModule(mod) {
        let data = mod.default;
        // Handle commonjs exports.
        if (!mod.hasOwnProperty('__esModule')) {
            data = mod;
        }
        if (!Array.isArray(data)) {
            data = [data];
        }
        data.forEach(item => {
            try {
                this.registerPlugin(item);
            }
            catch (error) {
                this.registerPluginErrors.push(error);
            }
        });
    }
    /**
     * Register the plugins from multiple plugin modules.
     *
     * @param mods - The plugin modules to register.
     */
    registerPluginModules(mods) {
        mods.forEach(mod => {
            this.registerPluginModule(mod);
        });
    }
}
/**
 * The namespace for `JupyterLab` class statics.
 */
(function (JupyterLab) {
    /**
     * The layout restorer token.
     */
    JupyterLab.IInfo = new Token('@jupyterlab/application:IInfo');
    /**
     * The default JupyterLab application info.
     */
    JupyterLab.defaultInfo = {
        devMode: PageConfig.getOption('devMode').toLowerCase() === 'true',
        deferred: { patterns: [], matches: [] },
        disabled: { patterns: [], matches: [] },
        mimeExtensions: [],
        filesCached: PageConfig.getOption('cacheFiles').toLowerCase() === 'true',
        isConnected: true
    };
    /**
     * The default JupyterLab application paths.
     */
    JupyterLab.defaultPaths = {
        urls: {
            base: PageConfig.getOption('baseUrl'),
            notFound: PageConfig.getOption('notFoundUrl'),
            app: PageConfig.getOption('appUrl'),
            doc: PageConfig.getOption('docUrl'),
            static: PageConfig.getOption('staticUrl'),
            settings: PageConfig.getOption('settingsUrl'),
            themes: PageConfig.getOption('themesUrl'),
            translations: PageConfig.getOption('translationsApiUrl'),
            hubHost: PageConfig.getOption('hubHost') || undefined,
            hubPrefix: PageConfig.getOption('hubPrefix') || undefined,
            hubUser: PageConfig.getOption('hubUser') || undefined,
            hubServerName: PageConfig.getOption('hubServerName') || undefined
        },
        directories: {
            appSettings: PageConfig.getOption('appSettingsDir'),
            schemas: PageConfig.getOption('schemasDir'),
            static: PageConfig.getOption('staticDir'),
            templates: PageConfig.getOption('templatesDir'),
            themes: PageConfig.getOption('themesDir'),
            userSettings: PageConfig.getOption('userSettingsDir'),
            serverRoot: PageConfig.getOption('serverRoot'),
            workspaces: PageConfig.getOption('workspacesDir')
        }
    };
})(JupyterLab || (JupyterLab = {}));
//# sourceMappingURL=lab.js.map