// Copyright (c) Jupyter Development Team.
// Distributed under the terms of the Modified BSD License.
import { PathExt } from '@jupyterlab/coreutils';
import { closeIcon, consoleIcon, jupyterIcon, notebookIcon } from '@jupyterlab/ui-components';
import { Throttler } from '@lumino/polling';
import { Signal } from '@lumino/signaling';
import { CommandIDs } from '.';
const ITEM_CLASS = 'jp-mod-kernel';
/**
 * Add the running kernel manager (notebooks & consoles) to the running panel.
 */
export async function addKernelRunningSessionManager(managers, translator, app) {
    const { commands, contextMenu, serviceManager } = app;
    const { kernels, kernelspecs, sessions } = serviceManager;
    const { runningChanged, RunningKernel } = Private;
    const throttler = new Throttler(() => runningChanged.emit(undefined), 100);
    const trans = translator.load('jupyterlab');
    // Throttle signal emissions from the kernel and session managers.
    kernels.runningChanged.connect(() => void throttler.invoke());
    sessions.runningChanged.connect(() => void throttler.invoke());
    // Wait until the relevant services are ready.
    await Promise.all([kernels.ready, kernelspecs.ready, sessions.ready]);
    // Add the kernels pane to the running sidebar.
    managers.add({
        name: trans.__('Kernels'),
        running: () => Array.from(kernels.running()).map(kernel => {
            var _a;
            return new RunningKernel({
                commands,
                kernel,
                kernels,
                sessions,
                spec: (_a = kernelspecs.specs) === null || _a === void 0 ? void 0 : _a.kernelspecs[kernel.name],
                trans
            });
        }),
        shutdownAll: () => kernels.shutdownAll(),
        refreshRunning: () => Promise.all([kernels.refreshRunning(), sessions.refreshRunning()]),
        runningChanged,
        shutdownLabel: trans.__('Shut Down Kernel'),
        shutdownAllLabel: trans.__('Shut Down All'),
        shutdownAllConfirmationText: trans.__('Are you sure you want to permanently shut down all running kernels?')
    });
    // Add running kernels commands to the registry.
    const test = (node) => node.classList.contains(ITEM_CLASS);
    commands.addCommand(CommandIDs.kernelNewConsole, {
        icon: consoleIcon,
        label: trans.__('New Console for Kernel'),
        execute: args => {
            var _a;
            const node = app.contextMenuHitTest(test);
            const id = (_a = args.id) !== null && _a !== void 0 ? _a : node === null || node === void 0 ? void 0 : node.dataset['context'];
            if (id) {
                return commands.execute('console:create', { kernelPreference: { id } });
            }
        }
    });
    commands.addCommand(CommandIDs.kernelNewNotebook, {
        icon: notebookIcon,
        label: trans.__('New Notebook for Kernel'),
        execute: args => {
            var _a;
            const node = app.contextMenuHitTest(test);
            const id = (_a = args.id) !== null && _a !== void 0 ? _a : node === null || node === void 0 ? void 0 : node.dataset['context'];
            if (id) {
                return commands.execute('notebook:create-new', { kernelId: id });
            }
        }
    });
    commands.addCommand(CommandIDs.kernelOpenSession, {
        icon: args => args.type === 'console'
            ? consoleIcon
            : args.type === 'notebook'
                ? notebookIcon
                : undefined,
        isEnabled: ({ path, type }) => !!type || path !== undefined,
        label: ({ name, path }) => name ||
            PathExt.basename(path || trans.__('Unknown Session')),
        execute: ({ path, type }) => {
            if (!type || path === undefined) {
                return;
            }
            const command = type === 'console' ? 'console:open' : 'docmanager:open';
            return commands.execute(command, { path });
        }
    });
    commands.addCommand(CommandIDs.kernelShutDown, {
        icon: closeIcon,
        label: trans.__('Shut Down Kernel'),
        execute: args => {
            var _a;
            const node = app.contextMenuHitTest(test);
            const id = (_a = args.id) !== null && _a !== void 0 ? _a : node === null || node === void 0 ? void 0 : node.dataset['context'];
            if (id) {
                return kernels.shutdown(id);
            }
        }
    });
    const sessionsItems = [];
    // Populate connected sessions submenu when context menu is opened.
    contextMenu.opened.connect(async () => {
        var _a, _b, _c;
        const submenu = (_b = (_a = contextMenu.menu.items.find(item => {
            var _a;
            return item.type === 'submenu' &&
                ((_a = item.submenu) === null || _a === void 0 ? void 0 : _a.id) === 'jp-contextmenu-connected-sessions';
        })) === null || _a === void 0 ? void 0 : _a.submenu) !== null && _b !== void 0 ? _b : null;
        if (!submenu) {
            // Bail early if the connected session menu is not found
            return;
        }
        // Empty the connected sessions submenu.
        sessionsItems.forEach(item => item.dispose());
        sessionsItems.length = 0;
        submenu.clearItems();
        const node = app.contextMenuHitTest(test);
        const id = node === null || node === void 0 ? void 0 : node.dataset['context'];
        if (!id) {
            return;
        }
        // Populate the submenu with sessions connected to this kernel.
        const command = CommandIDs.kernelOpenSession;
        for (const session of sessions.running()) {
            if (id === ((_c = session.kernel) === null || _c === void 0 ? void 0 : _c.id)) {
                const { name, path, type } = session;
                sessionsItems.push(submenu.addItem({ command, args: { name, path, type } }));
            }
        }
    });
}
var Private;
(function (Private) {
    class RunningKernel {
        constructor(options) {
            this.className = ITEM_CLASS;
            this.commands = options.commands;
            this.kernel = options.kernel;
            this.context = this.kernel.id;
            this.kernels = options.kernels;
            this.sessions = options.sessions;
            this.spec = options.spec || null;
            this.trans = options.trans;
        }
        get children() {
            var _a;
            const children = [];
            const open = CommandIDs.kernelOpenSession;
            const { commands } = this;
            for (const session of this.sessions.running()) {
                if (this.kernel.id === ((_a = session.kernel) === null || _a === void 0 ? void 0 : _a.id)) {
                    const { name, path, type } = session;
                    children.push({
                        className: ITEM_CLASS,
                        context: this.kernel.id,
                        open: () => void commands.execute(open, { name, path, type }),
                        icon: () => type === 'console'
                            ? consoleIcon
                            : type === 'notebook'
                                ? notebookIcon
                                : jupyterIcon,
                        label: () => name,
                        labelTitle: () => path
                    });
                }
            }
            return children;
        }
        shutdown() {
            return this.kernels.shutdown(this.kernel.id);
        }
        icon() {
            const { spec } = this;
            if (!spec || !spec.resources) {
                return jupyterIcon;
            }
            return (spec.resources['logo-svg'] ||
                spec.resources['logo-64x64'] ||
                spec.resources['logo-32x32']);
        }
        label() {
            const { kernel, spec } = this;
            return (spec === null || spec === void 0 ? void 0 : spec.display_name) || kernel.name;
        }
        labelTitle() {
            var _a;
            const { trans } = this;
            const { id } = this.kernel;
            const title = [`${this.label()}: ${id}`];
            for (const session of this.sessions.running()) {
                if (this.kernel.id === ((_a = session.kernel) === null || _a === void 0 ? void 0 : _a.id)) {
                    const { path, type } = session;
                    title.push(trans.__(`%1\nPath: %2`, type, path));
                }
            }
            return title.join('\n\n');
        }
    }
    Private.RunningKernel = RunningKernel;
    Private.runningChanged = new Signal({});
})(Private || (Private = {}));
//# sourceMappingURL=kernels.js.map