// Copyright (c) Jupyter Development Team.
// Distributed under the terms of the Modified BSD License.
import { nullTranslator } from '@jupyterlab/translation';
import { bugIcon, SidePanel } from '@jupyterlab/ui-components';
import { Widget } from '@lumino/widgets';
import { Breakpoints as BreakpointsPanel } from './panels/breakpoints';
import { Callstack as CallstackPanel } from './panels/callstack';
import { Sources as SourcesPanel } from './panels/sources';
import { KernelSources as KernelSourcesPanel } from './panels/kernelSources';
import { Variables as VariablesPanel } from './panels/variables';
/**
 * A debugger sidebar.
 */
export class DebuggerSidebar extends SidePanel {
    /**
     * Instantiate a new Debugger.Sidebar
     *
     * @param options The instantiation options for a Debugger.Sidebar
     */
    constructor(options) {
        const translator = options.translator || nullTranslator;
        super({ translator });
        this.id = 'jp-debugger-sidebar';
        this.title.icon = bugIcon;
        this.addClass('jp-DebuggerSidebar');
        const { callstackCommands, breakpointsCommands, editorServices, service, themeManager } = options;
        const model = service.model;
        this.variables = new VariablesPanel({
            model: model.variables,
            commands: callstackCommands.registry,
            service,
            themeManager,
            translator
        });
        this.callstack = new CallstackPanel({
            commands: callstackCommands,
            model: model.callstack,
            translator
        });
        this.breakpoints = new BreakpointsPanel({
            service,
            commands: breakpointsCommands,
            model: model.breakpoints,
            translator
        });
        this.sources = new SourcesPanel({
            model: model.sources,
            service,
            editorServices,
            translator
        });
        this.kernelSources = new KernelSourcesPanel({
            model: model.kernelSources,
            service,
            translator
        });
        const header = new DebuggerSidebar.Header();
        this.header.addWidget(header);
        model.titleChanged.connect((_, title) => {
            header.title.label = title;
        });
        this.content.addClass('jp-DebuggerSidebar-body');
        this.addWidget(this.variables);
        this.addWidget(this.callstack);
        this.addWidget(this.breakpoints);
        this.addWidget(this.sources);
        this.addWidget(this.kernelSources);
    }
}
/**
 * A namespace for DebuggerSidebar statics
 */
(function (DebuggerSidebar) {
    /**
     * The header for a debugger sidebar.
     */
    class Header extends Widget {
        /**
         * Instantiate a new sidebar header.
         */
        constructor() {
            super({ node: Private.createHeader() });
            this.title.changed.connect(_ => {
                this.node.textContent = this.title.label;
            });
        }
    }
    DebuggerSidebar.Header = Header;
})(DebuggerSidebar || (DebuggerSidebar = {}));
/**
 * A namespace for private module data.
 */
var Private;
(function (Private) {
    /**
     * Create a sidebar header node.
     */
    function createHeader() {
        const title = document.createElement('h2');
        title.textContent = '-';
        title.classList.add('jp-text-truncated');
        return title;
    }
    Private.createHeader = createHeader;
})(Private || (Private = {}));
//# sourceMappingURL=sidebar.js.map