// Copyright (c) Jupyter Development Team.
// Distributed under the terms of the Modified BSD License.
import { nullTranslator } from '@jupyterlab/translation';
import { Signal } from '@lumino/signaling';
import { caretDownEmptyIcon, ReactWidget, searchIcon } from '@jupyterlab/ui-components';
import { ArrayExt } from '@lumino/algorithm';
import React, { useCallback, useEffect, useState } from 'react';
import { convertType } from '.';
import { Debugger } from '../../debugger';
const BUTTONS_CLASS = 'jp-DebuggerVariables-buttons';
/**
 * The body for tree of variables.
 */
export class VariablesBodyTree extends ReactWidget {
    /**
     * Instantiate a new Body for the tree of variables.
     *
     * @param options The instantiation options for a VariablesBodyTree.
     */
    constructor(options) {
        super();
        this._scope = '';
        this._scopes = [];
        this._filter = new Set();
        this._commands = options.commands;
        this._service = options.service;
        this._translator = options.translator;
        this._hoverChanged = new Signal(this);
        const model = (this.model = options.model);
        model.changed.connect(this._updateScopes, this);
        this.addClass('jp-DebuggerVariables-body');
    }
    /**
     * Render the VariablesBodyTree.
     */
    render() {
        var _a;
        const scope = (_a = this._scopes.find(scope => scope.name === this._scope)) !== null && _a !== void 0 ? _a : this._scopes[0];
        const handleSelectVariable = (variable) => {
            this.model.selectedVariable = variable;
        };
        const collapserIcon = (React.createElement(caretDownEmptyIcon.react, { stylesheet: "menuItem", tag: "span" }));
        return scope ? (React.createElement(React.Fragment, null,
            React.createElement(VariablesBranch, { key: scope.name, commands: this._commands, service: this._service, data: scope.variables, filter: this._filter, translator: this._translator, handleSelectVariable: handleSelectVariable, onHoverChanged: (data) => {
                    this._hoverChanged.emit(data);
                }, collapserIcon: collapserIcon }),
            React.createElement(TreeButtons, { commands: this._commands, service: this._service, hoverChanged: this._hoverChanged, handleSelectVariable: handleSelectVariable }))) : (React.createElement("div", null));
    }
    /**
     * Set the variable filter list.
     */
    set filter(filter) {
        this._filter = filter;
        this.update();
    }
    /**
     * Set the current scope
     */
    set scope(scope) {
        this._scope = scope;
        this.update();
    }
    /**
     * Update the scopes and the tree of variables.
     *
     * @param model The variables model.
     */
    _updateScopes(model) {
        if (ArrayExt.shallowEqual(this._scopes, model.scopes)) {
            return;
        }
        this._scopes = model.scopes;
        this.update();
    }
}
/**
 * The singleton buttons bar shown by the variables.
 */
const TreeButtons = (props) => {
    var _a;
    const { commands, service, translator, handleSelectVariable } = props;
    const trans = (translator !== null && translator !== void 0 ? translator : nullTranslator).load('jupyterlab');
    const [buttonsTop, setButtonsTop] = useState(0);
    const [variable, setVariable] = useState(null);
    let stateRefreshLock = 0;
    // Empty dependency array is to only register once per lifetime.
    const handleHover = useCallback((_, data) => {
        const current = ++stateRefreshLock;
        if (!data.variable) {
            // Handle mouse leave.
            if (current !== stateRefreshLock) {
                return;
            }
            const target = data.target;
            if (target &&
                // Note: Element, not HTMLElement to permit entering <svg> icon.
                target instanceof Element &&
                target.closest(`.${BUTTONS_CLASS}`)) {
                // Allow to enter the buttons.
                return;
            }
            setVariable(null);
        }
        else {
            // Handle mouse over.
            setVariable(data.variable);
            requestAnimationFrame(() => {
                if (current !== stateRefreshLock || !data.target) {
                    return;
                }
                setButtonsTop(data.target.offsetTop);
            });
        }
    }, []);
    useEffect(() => {
        props.hoverChanged.connect(handleHover);
        return () => {
            props.hoverChanged.disconnect(handleHover);
        };
    }, [handleHover]);
    return (React.createElement("div", { className: BUTTONS_CLASS, style: 
        // Positioning and hiding is implemented using compositor-only
        // properties (transform and opacity) for performance.
        {
            transform: `translateY(${buttonsTop}px)`,
            opacity: !variable ||
                // Do not show buttons display for special entries, defined in debugpy:
                // https://github.com/microsoft/debugpy/blob/cf0d684566edc339545b161da7c3dfc48af7c7d5/src/debugpy/_vendored/pydevd/_pydevd_bundle/pydevd_utils.py#L359
                [
                    'special variables',
                    'protected variables',
                    'function variables',
                    'class variables'
                ].includes(variable.name)
                ? 0
                : 1
        } },
        React.createElement("button", { className: "jp-DebuggerVariables-renderVariable", disabled: !variable ||
                !service.model.hasRichVariableRendering ||
                !commands.isEnabled(Debugger.CommandIDs.renderMimeVariable, {
                    name: variable.name,
                    frameID: (_a = service.model.callstack.frame) === null || _a === void 0 ? void 0 : _a.id
                }), onClick: e => {
                var _a;
                if (!variable || !handleSelectVariable) {
                    return;
                }
                e.stopPropagation();
                handleSelectVariable(variable);
                commands
                    .execute(Debugger.CommandIDs.renderMimeVariable, {
                    name: variable.name,
                    frameID: (_a = service.model.callstack.frame) === null || _a === void 0 ? void 0 : _a.id
                })
                    .catch(reason => {
                    console.error(`Failed to render variable ${variable === null || variable === void 0 ? void 0 : variable.name}`, reason);
                });
            }, title: trans.__('Render variable: %1', variable === null || variable === void 0 ? void 0 : variable.name) },
            React.createElement(searchIcon.react, { stylesheet: "menuItem", tag: "span" }))));
};
/**
 * A React component to display a list of variables.
 *
 * @param {object} props The component props.
 * @param props.data An array of variables.
 * @param props.service The debugger service.
 * @param props.filter Optional variable filter list.
 */
const VariablesBranch = (props) => {
    const { commands, data, service, filter, translator, handleSelectVariable, onHoverChanged, collapserIcon } = props;
    const [variables, setVariables] = useState(data);
    useEffect(() => {
        setVariables(data);
    }, [data]);
    return (React.createElement("ul", { className: "jp-DebuggerVariables-branch" }, variables
        .filter(variable => !(filter || new Set()).has(variable.evaluateName || ''))
        .map(variable => {
        const key = `${variable.name}-${variable.evaluateName}-${variable.type}-${variable.value}-${variable.variablesReference}`;
        return (React.createElement(VariableComponent, { key: key, commands: commands, data: variable, service: service, filter: filter, translator: translator, onSelect: handleSelectVariable, onHoverChanged: onHoverChanged, collapserIcon: collapserIcon }));
    })));
};
function _prepareDetail(variable) {
    const detail = convertType(variable);
    if (variable.type === 'float' && isNaN(detail)) {
        // silence React warning:
        // `Received NaN for the `children` attribute. If this is expected, cast the value to a string`
        return 'NaN';
    }
    return detail;
}
/**
 * A React component to display one node variable in tree.
 *
 * @param {object} props The component props.
 * @param props.data An array of variables.
 * @param props.service The debugger service.
 * @param props.filter Optional variable filter list.
 */
const VariableComponent = (props) => {
    const { commands, data, service, filter, translator, onSelect, onHoverChanged, collapserIcon } = props;
    const [variable] = useState(data);
    const [expanded, setExpanded] = useState();
    const [variables, setVariables] = useState();
    const onSelection = onSelect !== null && onSelect !== void 0 ? onSelect : (() => void 0);
    const expandable = variable.variablesReference !== 0 || variable.type === 'function';
    const onVariableClicked = async (e) => {
        if (!expandable) {
            return;
        }
        e.stopPropagation();
        const variables = await service.inspectVariable(variable.variablesReference);
        setExpanded(!expanded);
        setVariables(variables);
    };
    return (React.createElement("li", { onClick: (e) => onVariableClicked(e), onMouseDown: e => {
            e.stopPropagation();
            onSelection(variable);
        }, onMouseOver: (event) => {
            if (onHoverChanged) {
                onHoverChanged({ target: event.currentTarget, variable });
                event.stopPropagation();
            }
        }, onMouseLeave: (event) => {
            if (onHoverChanged) {
                onHoverChanged({
                    target: event.relatedTarget,
                    variable: null
                });
                event.stopPropagation();
            }
        } },
        React.createElement("span", { className: 'jp-DebuggerVariables-collapser' +
                (expanded ? ' jp-mod-expanded' : '') }, 
        // note: using React.cloneElement due to high typestyle cost
        expandable ? React.cloneElement(collapserIcon) : null),
        React.createElement("span", { className: "jp-DebuggerVariables-name" }, variable.name),
        React.createElement("span", { className: "jp-DebuggerVariables-detail" }, _prepareDetail(variable)),
        expanded && variables && (React.createElement(VariablesBranch, { key: variable.name, commands: commands, data: variables, service: service, filter: filter, translator: translator, handleSelectVariable: onSelect, onHoverChanged: onHoverChanged, collapserIcon: collapserIcon }))));
};
//# sourceMappingURL=tree.js.map