import { CommandRegistry } from '@lumino/commands';
import { DataModel } from '@lumino/datagrid';
import { Panel } from '@lumino/widgets';
import { IThemeManager } from '@jupyterlab/apputils';
import { ITranslator } from '@jupyterlab/translation';
import { IDebugger } from '../../tokens';
/**
 * A class wrapping the underlying variables datagrid.
 */
export declare class Grid extends Panel {
    /**
     * Instantiate a new VariablesGrid.
     *
     * @param options The instantiation options for a VariablesGrid.
     */
    constructor(options: Grid.IOptions);
    /**
     * Set the variable filter list.
     *
     * @param filter The variable filter to apply.
     */
    set filter(filter: Set<string>);
    /**
     * Set the scope for the variables data model.
     *
     * @param scope The scopes for the variables
     */
    set scope(scope: string);
    /**
     * Get the data model for the data grid.
     */
    get dataModel(): GridModel;
    /**
     * Handle `after-attach` messages.
     *
     * @param message - The `after-attach` message.
     */
    protected onAfterAttach(message: any): void;
    /**
     * Update the computed style for the data grid on theme change.
     */
    private _updateStyles;
    private _grid;
    protected model: IDebugger.Model.IVariables;
}
/**
 * A namespace for VariablesGrid `statics`.
 */
declare namespace Grid {
    /**
     * Instantiation options for `VariablesGrid`.
     */
    interface IOptions {
        /**
         * The commands registry.
         */
        commands: CommandRegistry;
        /**
         * The variables model.
         */
        model: IDebugger.Model.IVariables;
        /**
         * An optional application theme manager to detect theme changes.
         */
        themeManager?: IThemeManager | null;
        /**
         * The application language translator.
         */
        translator?: ITranslator;
    }
}
/**
 * A data grid model for variables.
 */
export declare class GridModel extends DataModel {
    /**
     * Create gird model
     * @param translator optional translator
     */
    constructor(translator?: ITranslator);
    /**
     * Set the variable filter list.
     */
    set filter(filter: Set<string>);
    /**
     * Get the current scope for the variables.
     */
    get scope(): string;
    /**
     * Set the variable scope
     */
    set scope(scope: string);
    /**
     * Get the row count for a particular region in the data grid.
     *
     * @param region The datagrid region.
     */
    rowCount(region: DataModel.RowRegion): number;
    /**
     * Get the column count for a particular region in the data grid.
     *
     * @param region The datagrid region.
     */
    columnCount(region: DataModel.ColumnRegion): number;
    /**
     * Get the data count for a particular region, row and column in the data grid.
     *
     * @param region The datagrid region.
     * @param row The datagrid row
     * @param column The datagrid column
     */
    data(region: DataModel.CellRegion, row: number, column: number): any;
    /**
     * Get the variable reference for a given row
     *
     * @param row The row in the datagrid.
     */
    getVariableReference(row: number): number;
    /**
     * Get the variable name for a given row
     *
     * @param row The row in the datagrid.
     */
    getVariableName(row: number): string;
    /**
     * Set the datagrid model data from the list of variables.
     *
     * @param scopes The list of variables.
     */
    setData(scopes: IDebugger.IScope[]): void;
    /**
     * Clear all the data in the datagrid.
     */
    private _clearData;
    private _filter;
    private _scope;
    private _trans;
    private _data;
}
export {};
