// Copyright (c) Jupyter Development Team.
// Distributed under the terms of the Modified BSD License.
import { Panel } from '@lumino/widgets';
import { PromiseDelegate } from '@lumino/coreutils';
/**
 * A data grid that displays variables in a debugger session.
 */
export class VariablesBodyGrid extends Panel {
    /**
     * Instantiate a new VariablesBodyGrid.
     *
     * @param options The instantiation options for a VariablesBodyGrid.
     */
    constructor(options) {
        super();
        this._model = options.model;
        this._options = options;
        this.addClass('jp-DebuggerVariables-body');
    }
    /**
     * Wait until actually displaying the grid to trigger initialization.
     */
    onBeforeShow(msg) {
        if (!this._grid) {
            void this.initialize();
        }
    }
    /**
     * Load the grid panel implementation.
     */
    async initialize() {
        const { model, commands, themeManager, scopes, translator } = this._options;
        const { Grid } = await Private.ensureGridPanel();
        this._grid = new Grid({ commands, model, themeManager, translator });
        this._grid.addClass('jp-DebuggerVariables-grid');
        this._model.changed.connect((model) => {
            this._update();
        }, this);
        this._grid.dataModel.setData(scopes !== null && scopes !== void 0 ? scopes : []);
        this.addWidget(this._grid);
    }
    /**
     * Set the variable filter list.
     *
     * @param filter The variable filter to apply.
     */
    set filter(filter) {
        this._grid.dataModel.filter = filter;
        this._update();
    }
    /**
     * Set the current scope.
     *
     * @param scope The current scope for the variables.
     */
    set scope(scope) {
        this._grid.dataModel.scope = scope;
        this._update();
    }
    /**
     * Update the underlying data model
     */
    _update() {
        var _a;
        this._grid.dataModel.setData((_a = this._model.scopes) !== null && _a !== void 0 ? _a : []);
    }
}
/**
 * A private namespace for managing lazy loading of the underlying grid panel.
 */
var Private;
(function (Private) {
    let gridPanelLoaded = null;
    /**
     * Lazily load the datagrid module when the first grid is requested.
     */
    async function ensureGridPanel() {
        if (gridPanelLoaded == null) {
            gridPanelLoaded = new PromiseDelegate();
            gridPanelLoaded.resolve(await import('./gridpanel'));
        }
        return gridPanelLoaded.promise;
    }
    Private.ensureGridPanel = ensureGridPanel;
})(Private || (Private = {}));
//# sourceMappingURL=grid.js.map