// Copyright (c) Jupyter Development Team.
// Distributed under the terms of the Modified BSD License.
import { Dialog, showDialog } from '@jupyterlab/apputils';
import { nullTranslator } from '@jupyterlab/translation';
import { PanelWithToolbar, ToolbarButton } from '@jupyterlab/ui-components';
import { Signal } from '@lumino/signaling';
import { closeAllIcon, exceptionIcon } from '../../icons';
import { BreakpointsBody } from './body';
import { PauseOnExceptionsWidget } from './pauseonexceptions';
/**
 * A Panel to show a list of breakpoints.
 */
export class Breakpoints extends PanelWithToolbar {
    /**
     * Instantiate a new Breakpoints Panel.
     *
     * @param options The instantiation options for a Breakpoints Panel.
     */
    constructor(options) {
        var _a;
        super(options);
        this.clicked = new Signal(this);
        const { model, service, commands } = options;
        const trans = ((_a = options.translator) !== null && _a !== void 0 ? _a : nullTranslator).load('jupyterlab');
        this.title.label = trans.__('Breakpoints');
        const body = new BreakpointsBody(model);
        this.toolbar.addItem('pauseOnException', new PauseOnExceptionsWidget({
            service: service,
            commands: commands,
            icon: exceptionIcon,
            tooltip: trans.__('Pause on exception filter')
        }));
        this.toolbar.addItem('closeAll', new ToolbarButton({
            icon: closeAllIcon,
            onClick: async () => {
                if (model.breakpoints.size === 0) {
                    return;
                }
                const result = await showDialog({
                    title: trans.__('Remove All Breakpoints'),
                    body: trans.__('Are you sure you want to remove all breakpoints?'),
                    buttons: [
                        Dialog.okButton({ label: trans.__('Remove breakpoints') }),
                        Dialog.cancelButton()
                    ],
                    hasClose: true
                });
                if (result.button.accept) {
                    return service.clearBreakpoints();
                }
            },
            tooltip: trans.__('Remove All Breakpoints')
        }));
        this.addWidget(body);
        this.addClass('jp-DebuggerBreakpoints');
    }
}
//# sourceMappingURL=index.js.map