// Copyright (c) Jupyter Development Team.
// Distributed under the terms of the Modified BSD License.
import { StateEffect, StateField } from '@codemirror/state';
import { Decoration, EditorView } from '@codemirror/view';
import { GenericSearchProvider, TextSearchEngine } from '@jupyterlab/documentsearch';
import { Signal } from '@lumino/signaling';
/**
 * Search provider for editors.
 */
export class EditorSearchProvider {
    /**
     * Constructor
     */
    constructor() {
        /**
         * Current match index
         */
        this.currentIndex = null;
        /**
         * Current search query
         */
        this.query = null;
        this._isActive = true;
        this._inSelection = null;
        this._isDisposed = false;
        this._cmHandler = null;
        this.currentIndex = null;
        this._stateChanged = new Signal(this);
    }
    /**
     * CodeMirror search highlighter
     */
    get cmHandler() {
        if (!this._cmHandler) {
            this._cmHandler = new CodeMirrorSearchHighlighter(this.editor);
        }
        return this._cmHandler;
    }
    /**
     * Changed signal to be emitted when search matches change.
     */
    get stateChanged() {
        return this._stateChanged;
    }
    /**
     * Current match index
     */
    get currentMatchIndex() {
        return this.isActive ? this.currentIndex : null;
    }
    /**
     * Whether the cell search is active.
     *
     * This is used when applying search only on selected cells.
     */
    get isActive() {
        return this._isActive;
    }
    /**
     * Whether the search provider is disposed or not.
     */
    get isDisposed() {
        return this._isDisposed;
    }
    /**
     * Number of matches in the cell.
     */
    get matchesCount() {
        return this.isActive ? this.cmHandler.matches.length : 0;
    }
    /**
     * Clear currently highlighted match
     */
    clearHighlight() {
        this.currentIndex = null;
        this.cmHandler.clearHighlight();
        return Promise.resolve();
    }
    /**
     * Dispose the search provider
     */
    dispose() {
        if (this._isDisposed) {
            return;
        }
        this._isDisposed = true;
        Signal.clearData(this);
        if (this.isActive) {
            this.endQuery().catch(reason => {
                console.error(`Failed to end search query on cells.`, reason);
            });
        }
    }
    /**
     * Set `isActive` status.
     *
     * #### Notes
     * It will start or end the search
     *
     * @param v New value
     */
    async setIsActive(v) {
        if (this._isActive === v) {
            return;
        }
        this._isActive = v;
        if (this._isActive) {
            if (this.query !== null) {
                await this.startQuery(this.query, this.filters);
            }
        }
        else {
            await this.endQuery();
        }
    }
    /**
     * Set whether search should be limitted to specified selection.
     */
    async setSearchSelection(selection) {
        if (this._inSelection === selection) {
            return;
        }
        this._inSelection = selection;
        await this.updateCodeMirror(this.model.sharedModel.getSource());
        this._stateChanged.emit();
    }
    /**
     * Initialize the search using the provided options. Should update the UI
     * to highlight all matches and "select" the first match.
     *
     * @param query A RegExp to be use to perform the search
     * @param filters Filter parameters to pass to provider
     */
    async startQuery(query, filters) {
        this.query = query;
        this.filters = filters;
        // Search input
        const content = this.model.sharedModel.getSource();
        await this.updateCodeMirror(content);
        this.model.sharedModel.changed.connect(this.onSharedModelChanged, this);
    }
    /**
     * Stop the search and clean any UI elements.
     */
    async endQuery() {
        await this.clearHighlight();
        await this.cmHandler.endQuery();
        this.currentIndex = null;
    }
    /**
     * Highlight the next match.
     *
     * @returns The next match if there is one.
     */
    async highlightNext(loop = true, fromCursor = false) {
        if (this.matchesCount === 0 || !this.isActive) {
            this.currentIndex = null;
        }
        else {
            // This starts from the cursor position if `fromCursor` is true
            let match = await this.cmHandler.highlightNext(fromCursor);
            if (match) {
                this.currentIndex = this.cmHandler.currentIndex;
            }
            else {
                // Note: the loop logic is only used in single-editor (e.g. file editor)
                // provider sub-classes, notebook has it's own loop logic and ignores
                // `currentIndex` as set here.
                this.currentIndex = loop ? 0 : null;
            }
            return match;
        }
        return Promise.resolve(this.getCurrentMatch());
    }
    /**
     * Highlight the previous match.
     *
     * @returns The previous match if there is one.
     */
    async highlightPrevious(loop = true, fromCursor = false) {
        if (this.matchesCount === 0 || !this.isActive) {
            this.currentIndex = null;
        }
        else {
            // This starts from the cursor position if `fromCursor` is true
            let match = await this.cmHandler.highlightPrevious(fromCursor);
            if (match) {
                this.currentIndex = this.cmHandler.currentIndex;
            }
            else {
                this.currentIndex = loop ? this.matchesCount - 1 : null;
            }
            return match;
        }
        return Promise.resolve(this.getCurrentMatch());
    }
    /**
     * Replace the currently selected match with the provided text.
     *
     * If no match is selected, it won't do anything.
     *
     * The caller of this method is expected to call `highlightNext` if after
     * calling `replaceCurrentMatch()` attribute `this.currentIndex` is null.
     * It is necesary to let the caller handle highlighting because this
     * method is used in composition pattern (search engine of notebook cells)
     * and highligthing on the composer (notebook) level needs to switch to next
     * engine (cell) with matches.
     *
     * @param newText The replacement text.
     * @returns Whether a replace occurred.
     */
    replaceCurrentMatch(newText, loop, options) {
        if (!this.isActive) {
            return Promise.resolve(false);
        }
        let occurred = false;
        if (this.currentIndex !== null &&
            this.currentIndex < this.cmHandler.matches.length) {
            const match = this.getCurrentMatch();
            // If cursor there is no match selected, highlight the next match
            if (!match) {
                this.currentIndex = null;
            }
            else {
                this.cmHandler.matches.splice(this.currentIndex, 1);
                this.currentIndex =
                    this.currentIndex < this.cmHandler.matches.length
                        ? Math.max(this.currentIndex - 1, 0)
                        : null;
                const substitutedText = (options === null || options === void 0 ? void 0 : options.regularExpression)
                    ? match.text.replace(this.query, newText)
                    : newText;
                const insertText = (options === null || options === void 0 ? void 0 : options.preserveCase)
                    ? GenericSearchProvider.preserveCase(match.text, substitutedText)
                    : substitutedText;
                this.model.sharedModel.updateSource(match.position, match.position + match.text.length, insertText);
                occurred = true;
            }
        }
        return Promise.resolve(occurred);
    }
    /**
     * Replace all matches in the cell source with the provided text
     *
     * @param newText The replacement text.
     * @returns Whether a replace occurred.
     */
    replaceAllMatches(newText, options) {
        if (!this.isActive) {
            return Promise.resolve(false);
        }
        let occurred = this.cmHandler.matches.length > 0;
        let src = this.model.sharedModel.getSource();
        let lastEnd = 0;
        const finalSrc = this.cmHandler.matches.reduce((agg, match) => {
            const start = match.position;
            const end = start + match.text.length;
            const substitutedText = (options === null || options === void 0 ? void 0 : options.regularExpression)
                ? match.text.replace(this.query, newText)
                : newText;
            const insertText = (options === null || options === void 0 ? void 0 : options.preserveCase)
                ? GenericSearchProvider.preserveCase(match.text, substitutedText)
                : substitutedText;
            const newStep = `${agg}${src.slice(lastEnd, start)}${insertText}`;
            lastEnd = end;
            return newStep;
        }, '');
        if (occurred) {
            this.cmHandler.matches = [];
            this.currentIndex = null;
            this.model.sharedModel.setSource(`${finalSrc}${src.slice(lastEnd)}`);
        }
        return Promise.resolve(occurred);
    }
    /**
     * Get the current match if it exists.
     *
     * @returns The current match
     */
    getCurrentMatch() {
        if (this.currentIndex === null) {
            return undefined;
        }
        else {
            let match = undefined;
            if (this.currentIndex < this.cmHandler.matches.length) {
                match = this.cmHandler.matches[this.currentIndex];
            }
            return match;
        }
    }
    /**
     * Callback on source change
     *
     * @param emitter Source of the change
     * @param changes Source change
     */
    async onSharedModelChanged(emitter, changes) {
        if (changes.sourceChange) {
            await this.updateCodeMirror(emitter.getSource());
            this._stateChanged.emit();
        }
    }
    /**
     * Update matches
     */
    async updateCodeMirror(content) {
        if (this.query !== null && this.isActive) {
            const allMatches = await TextSearchEngine.search(this.query, content);
            if (this._inSelection) {
                const editor = this.editor;
                const start = editor.getOffsetAt(this._inSelection.start);
                const end = editor.getOffsetAt(this._inSelection.end);
                this.cmHandler.matches = allMatches.filter(match => match.position >= start && match.position <= end);
                // A special case to always have a current match when in line selection mode.
                if (this.cmHandler.currentIndex === null &&
                    this.cmHandler.matches.length > 0) {
                    await this.cmHandler.highlightNext(true, false);
                }
                this.currentIndex = this.cmHandler.currentIndex;
            }
            else {
                this.cmHandler.matches = allMatches;
            }
        }
        else {
            this.cmHandler.matches = [];
        }
    }
}
/**
 * Helper class to highlight texts in a code mirror editor.
 *
 * Highlighted texts (aka `matches`) must be provided through
 * the `matches` attributes.
 *
 * **NOTES:**
 * - to retain the selection visibility `drawSelection` extension is needed.
 * - highlighting starts from the cursor (if editor is focused, cursor moved,
 *   or `fromCursor` argument is set to `true`), or from last "current" match
 *   otherwise.
 * - `currentIndex` is the (readonly) source of truth for the current match.
 */
export class CodeMirrorSearchHighlighter {
    /**
     * Constructor
     *
     * @param editor The CodeMirror editor
     */
    constructor(editor) {
        this._current = null;
        this._cm = editor;
        this._matches = new Array();
        this._currentIndex = null;
        this._highlightEffect = StateEffect.define({
            map: (value, mapping) => {
                const transform = (v) => ({
                    text: v.text,
                    position: mapping.mapPos(v.position)
                });
                return {
                    matches: value.matches.map(transform),
                    currentMatch: value.currentMatch
                        ? transform(value.currentMatch)
                        : null
                };
            }
        });
        this._highlightMark = Decoration.mark({ class: 'cm-searching' });
        this._currentMark = Decoration.mark({ class: 'jp-current-match' });
        this._highlightField = StateField.define({
            create: () => {
                return Decoration.none;
            },
            update: (highlights, transaction) => {
                highlights = highlights.map(transaction.changes);
                for (let ef of transaction.effects) {
                    if (ef.is(this._highlightEffect)) {
                        const e = ef;
                        if (e.value.matches.length) {
                            highlights = highlights.update({
                                add: e.value.matches.map(m => this._highlightMark.range(m.position, m.position + m.text.length)),
                                // filter out old marks
                                filter: () => false
                            });
                            highlights = highlights.update({
                                add: e.value.currentMatch
                                    ? [
                                        this._currentMark.range(e.value.currentMatch.position, e.value.currentMatch.position +
                                            e.value.currentMatch.text.length)
                                    ]
                                    : []
                            });
                        }
                        else {
                            highlights = Decoration.none;
                        }
                    }
                }
                return highlights;
            },
            provide: f => EditorView.decorations.from(f)
        });
    }
    /**
     * The current index of the selected match.
     */
    get currentIndex() {
        return this._currentIndex;
    }
    /**
     * The list of matches
     */
    get matches() {
        return this._matches;
    }
    set matches(v) {
        this._matches = v;
        if (this._currentIndex !== null &&
            this._currentIndex > this._matches.length) {
            this._currentIndex = this._matches.length > 0 ? 0 : null;
        }
        this._highlightCurrentMatch(true);
    }
    /**
     * Clear all highlighted matches
     */
    clearHighlight() {
        this._currentIndex = null;
        this._highlightCurrentMatch();
    }
    /**
     * Clear the highlighted matches.
     */
    endQuery() {
        this._currentIndex = null;
        this._matches = [];
        if (this._cm) {
            this._cm.editor.dispatch({
                effects: this._highlightEffect.of({ matches: [], currentMatch: null })
            });
        }
        return Promise.resolve();
    }
    /**
     * Highlight the next match
     *
     * @returns The next match if available
     */
    highlightNext(fromCursor = false, doNotModifySelection = false) {
        this._currentIndex = this._findNext(false, fromCursor);
        this._highlightCurrentMatch(doNotModifySelection);
        return Promise.resolve(this._currentIndex !== null
            ? this._matches[this._currentIndex]
            : undefined);
    }
    /**
     * Highlight the previous match
     *
     * @returns The previous match if available
     */
    highlightPrevious(fromCursor = false) {
        this._currentIndex = this._findNext(true, fromCursor);
        this._highlightCurrentMatch();
        return Promise.resolve(this._currentIndex !== null
            ? this._matches[this._currentIndex]
            : undefined);
    }
    /**
     * Set the editor
     *
     * @param editor Editor
     */
    setEditor(editor) {
        if (this._cm) {
            throw new Error('CodeMirrorEditor already set.');
        }
        else {
            this._cm = editor;
            if (this._currentIndex !== null) {
                this._highlightCurrentMatch();
            }
            this._refresh();
        }
    }
    _selectCurrentMatch() {
        const match = this._current;
        if (!match) {
            return;
        }
        if (!this._cm) {
            return;
        }
        const selection = this._cm.editor.state.selection.main;
        if (selection.from === match.position &&
            selection.to === match.position + match.text.length) {
            return;
        }
        const cursor = {
            anchor: match.position,
            head: match.position + match.text.length
        };
        this._cm.editor.dispatch({
            selection: cursor,
            scrollIntoView: true
        });
    }
    _highlightCurrentMatch(doNotModifySelection = false) {
        if (!this._cm) {
            // no-op
            return;
        }
        // Highlight the current index
        if (this._currentIndex !== null) {
            const match = this.matches[this._currentIndex];
            this._current = match;
            // Do not change selection/scroll if user is selecting
            if (!doNotModifySelection) {
                if (this._cm.hasFocus()) {
                    // If editor is focused we actually set the cursor on the match.
                    this._selectCurrentMatch();
                }
                else {
                    // otherwise we just scroll to preserve the selection.
                    this._cm.editor.dispatch({
                        effects: EditorView.scrollIntoView(match.position)
                    });
                }
            }
        }
        else {
            this._current = null;
        }
        this._refresh();
    }
    _refresh() {
        if (!this._cm) {
            // no-op
            return;
        }
        let effects = [
            this._highlightEffect.of({
                matches: this.matches,
                currentMatch: this._current
            })
        ];
        if (!this._cm.state.field(this._highlightField, false)) {
            effects.push(StateEffect.appendConfig.of([this._highlightField]));
            // set cursor on active match when editor gets focused
            const focusExtension = EditorView.domEventHandlers({
                focus: () => {
                    this._selectCurrentMatch();
                }
            });
            effects.push(StateEffect.appendConfig.of([focusExtension]));
        }
        this._cm.editor.dispatch({ effects });
    }
    _findNext(reverse, fromCursor = false) {
        if (this._matches.length === 0) {
            // No-op
            return null;
        }
        let lastPosition = 0;
        if (this._cm.hasFocus() || fromCursor) {
            const cursor = this._cm.state.selection.main;
            lastPosition = reverse ? cursor.anchor : cursor.head;
        }
        else if (this._current) {
            lastPosition = reverse
                ? this._current.position
                : this._current.position + this._current.text.length;
        }
        if (lastPosition === 0 && reverse && this.currentIndex === null) {
            // The default position is (0, 0) but we want to start from the end in that case
            lastPosition = this._cm.doc.length;
        }
        const position = lastPosition;
        let found = Utils.findNext(this._matches, position, 0, this._matches.length - 1);
        if (found === null) {
            // Don't loop
            return reverse ? this._matches.length - 1 : null;
        }
        if (reverse) {
            found -= 1;
            if (found < 0) {
                // Don't loop
                return null;
            }
        }
        return found;
    }
}
/**
 * Helpers namespace
 */
var Utils;
(function (Utils) {
    /**
     * Find the closest match at `position` just after it.
     *
     * #### Notes
     * Search is done using a binary search algorithm
     *
     * @param matches List of matches
     * @param position Searched position
     * @param lowerBound Lower range index
     * @param higherBound High range index
     * @returns The next match or null if none exists
     */
    function findNext(matches, position, lowerBound = 0, higherBound = Infinity) {
        higherBound = Math.min(matches.length - 1, higherBound);
        while (lowerBound <= higherBound) {
            let middle = Math.floor(0.5 * (lowerBound + higherBound));
            const currentPosition = matches[middle].position;
            if (currentPosition < position) {
                lowerBound = middle + 1;
                if (lowerBound < matches.length &&
                    matches[lowerBound].position > position) {
                    return lowerBound;
                }
            }
            else if (currentPosition > position) {
                higherBound = middle - 1;
                if (higherBound > 0 && matches[higherBound].position < position) {
                    return middle;
                }
            }
            else {
                return middle;
            }
        }
        // Next could be the first item
        const first = lowerBound > 0 ? lowerBound - 1 : 0;
        const match = matches[first];
        return match.position >= position ? first : null;
    }
    Utils.findNext = findNext;
})(Utils || (Utils = {}));
//# sourceMappingURL=searchprovider.js.map