import { ISearchMatch } from '@jupyterlab/documentsearch';
import { CodeMirrorEditor } from './editor';
import { CodeEditor } from '@jupyterlab/codeeditor';
import { IBaseSearchProvider, IFilters, IReplaceOptions } from '@jupyterlab/documentsearch';
import { ISignal, Signal } from '@lumino/signaling';
import { ISharedText, SourceChange } from '@jupyter/ydoc';
/**
 * Search provider for editors.
 */
export declare abstract class EditorSearchProvider<T extends CodeEditor.IModel = CodeEditor.IModel> implements IBaseSearchProvider {
    /**
     * Constructor
     */
    constructor();
    /**
     * CodeMirror search highlighter
     */
    protected get cmHandler(): CodeMirrorSearchHighlighter;
    /**
     * Text editor
     */
    protected abstract get editor(): CodeEditor.IEditor | null;
    /**
     * Editor content model
     */
    protected abstract get model(): T;
    /**
     * Changed signal to be emitted when search matches change.
     */
    get stateChanged(): ISignal<IBaseSearchProvider, void>;
    /**
     * Current match index
     */
    get currentMatchIndex(): number | null;
    /**
     * Whether the cell search is active.
     *
     * This is used when applying search only on selected cells.
     */
    get isActive(): boolean;
    /**
     * Whether the search provider is disposed or not.
     */
    get isDisposed(): boolean;
    /**
     * Number of matches in the cell.
     */
    get matchesCount(): number;
    /**
     * Clear currently highlighted match
     */
    clearHighlight(): Promise<void>;
    /**
     * Dispose the search provider
     */
    dispose(): void;
    /**
     * Set `isActive` status.
     *
     * #### Notes
     * It will start or end the search
     *
     * @param v New value
     */
    setIsActive(v: boolean): Promise<void>;
    /**
     * Set whether search should be limitted to specified selection.
     */
    setSearchSelection(selection: CodeEditor.IRange | null): Promise<void>;
    /**
     * Initialize the search using the provided options. Should update the UI
     * to highlight all matches and "select" the first match.
     *
     * @param query A RegExp to be use to perform the search
     * @param filters Filter parameters to pass to provider
     */
    startQuery(query: RegExp | null, filters?: IFilters): Promise<void>;
    /**
     * Stop the search and clean any UI elements.
     */
    endQuery(): Promise<void>;
    /**
     * Highlight the next match.
     *
     * @returns The next match if there is one.
     */
    highlightNext(loop?: boolean, fromCursor?: boolean): Promise<ISearchMatch | undefined>;
    /**
     * Highlight the previous match.
     *
     * @returns The previous match if there is one.
     */
    highlightPrevious(loop?: boolean, fromCursor?: boolean): Promise<ISearchMatch | undefined>;
    /**
     * Replace the currently selected match with the provided text.
     *
     * If no match is selected, it won't do anything.
     *
     * The caller of this method is expected to call `highlightNext` if after
     * calling `replaceCurrentMatch()` attribute `this.currentIndex` is null.
     * It is necesary to let the caller handle highlighting because this
     * method is used in composition pattern (search engine of notebook cells)
     * and highligthing on the composer (notebook) level needs to switch to next
     * engine (cell) with matches.
     *
     * @param newText The replacement text.
     * @returns Whether a replace occurred.
     */
    replaceCurrentMatch(newText: string, loop?: boolean, options?: IReplaceOptions): Promise<boolean>;
    /**
     * Replace all matches in the cell source with the provided text
     *
     * @param newText The replacement text.
     * @returns Whether a replace occurred.
     */
    replaceAllMatches(newText: string, options?: IReplaceOptions): Promise<boolean>;
    /**
     * Get the current match if it exists.
     *
     * @returns The current match
     */
    getCurrentMatch(): ISearchMatch | undefined;
    /**
     * Callback on source change
     *
     * @param emitter Source of the change
     * @param changes Source change
     */
    protected onSharedModelChanged(emitter: ISharedText, changes: SourceChange): Promise<void>;
    /**
     * Update matches
     */
    protected updateCodeMirror(content: string): Promise<void>;
    /**
     * Current match index
     */
    protected currentIndex: number | null;
    /**
     * Current search filters
     */
    protected filters: IFilters | undefined;
    /**
     * Current search query
     */
    protected query: RegExp | null;
    protected _stateChanged: Signal<IBaseSearchProvider, void>;
    private _isActive;
    private _inSelection;
    private _isDisposed;
    private _cmHandler;
}
/**
 * Helper class to highlight texts in a code mirror editor.
 *
 * Highlighted texts (aka `matches`) must be provided through
 * the `matches` attributes.
 *
 * **NOTES:**
 * - to retain the selection visibility `drawSelection` extension is needed.
 * - highlighting starts from the cursor (if editor is focused, cursor moved,
 *   or `fromCursor` argument is set to `true`), or from last "current" match
 *   otherwise.
 * - `currentIndex` is the (readonly) source of truth for the current match.
 */
export declare class CodeMirrorSearchHighlighter {
    /**
     * Constructor
     *
     * @param editor The CodeMirror editor
     */
    constructor(editor: CodeMirrorEditor | null);
    /**
     * The current index of the selected match.
     */
    get currentIndex(): number | null;
    /**
     * The list of matches
     */
    get matches(): ISearchMatch[];
    set matches(v: ISearchMatch[]);
    private _current;
    /**
     * Clear all highlighted matches
     */
    clearHighlight(): void;
    /**
     * Clear the highlighted matches.
     */
    endQuery(): Promise<void>;
    /**
     * Highlight the next match
     *
     * @returns The next match if available
     */
    highlightNext(fromCursor?: boolean, doNotModifySelection?: boolean): Promise<ISearchMatch | undefined>;
    /**
     * Highlight the previous match
     *
     * @returns The previous match if available
     */
    highlightPrevious(fromCursor?: boolean): Promise<ISearchMatch | undefined>;
    /**
     * Set the editor
     *
     * @param editor Editor
     */
    setEditor(editor: CodeMirrorEditor): void;
    private _selectCurrentMatch;
    private _highlightCurrentMatch;
    private _refresh;
    private _findNext;
    private _cm;
    private _currentIndex;
    private _matches;
    private _highlightEffect;
    private _highlightMark;
    private _currentMark;
    private _highlightField;
}
