// Copyright (c) Jupyter Development Team.
// Distributed under the terms of the Modified BSD License.
import { closeBrackets } from '@codemirror/autocomplete';
import { defaultKeymap, indentLess } from '@codemirror/commands';
import { bracketMatching, foldGutter, indentOnInput, indentUnit } from '@codemirror/language';
import { Compartment, EditorState, StateEffect } from '@codemirror/state';
import { drawSelection, EditorView, highlightActiveLine, highlightTrailingWhitespace, highlightWhitespace, keymap, lineNumbers, scrollPastEnd } from '@codemirror/view';
import { nullTranslator } from '@jupyterlab/translation';
import { JSONExt } from '@lumino/coreutils';
import { Signal } from '@lumino/signaling';
import { StateCommands } from './commands';
import { customTheme, rulers } from './extensions';
/**
 * The class name added to read only editor widgets.
 */
const READ_ONLY_CLASS = 'jp-mod-readOnly';
/**
 * Editor configuration handler
 *
 * It stores the editor configuration and the editor extensions.
 * It also allows to inject new extensions into an editor.
 */
export class ExtensionsHandler {
    constructor({ baseConfiguration, config, defaultExtensions } = {}) {
        this._configChanged = new Signal(this);
        this._disposed = new Signal(this);
        this._isDisposed = false;
        this._immutables = new Set();
        this._baseConfig = baseConfiguration !== null && baseConfiguration !== void 0 ? baseConfiguration : {};
        this._config = config !== null && config !== void 0 ? config : {};
        this._configurableBuilderMap = new Map(defaultExtensions);
        const configurables = Object.keys(this._config).concat(Object.keys(this._baseConfig));
        this._immutables = new Set([...this._configurableBuilderMap.keys()].filter(key => !configurables.includes(key)));
    }
    /**
     * Signal triggered when the editor configuration changes.
     * It provides the mapping of the new configuration (only those that changed).
     *
     * It should result in a call to `IExtensionsHandler.reconfigureExtensions`.
     */
    get configChanged() {
        return this._configChanged;
    }
    /**
     * A signal emitted when the object is disposed.
     */
    get disposed() {
        return this._disposed;
    }
    /**
     * Tests whether the object is disposed.
     */
    get isDisposed() {
        return this._isDisposed;
    }
    /**
     * Dispose of the resources held by the object.
     */
    dispose() {
        if (this.isDisposed) {
            return;
        }
        this._isDisposed = true;
        this._disposed.emit();
        Signal.clearData(this);
    }
    /**
     * Get a config option for the editor.
     */
    getOption(option) {
        var _a;
        return (_a = this._config[option]) !== null && _a !== void 0 ? _a : this._baseConfig[option];
    }
    /**
     * Whether the option exists or not.
     */
    hasOption(option) {
        return (Object.keys(this._config).includes(option) ||
            Object.keys(this._baseConfig).includes(option));
    }
    /**
     * Set a config option for the editor.
     *
     * You will need to reconfigure the editor extensions by listening
     * to `IExtensionsHandler.configChanged`.
     */
    setOption(option, value) {
        // Don't bother setting the option if it is already the same.
        if (this._config[option] !== value) {
            this._config[option] = value;
            this._configChanged.emit({ [option]: value });
        }
    }
    /**
     * Set a base config option for the editor.
     *
     * You will need to reconfigure the editor extensions by listening
     * to `IExtensionsHandler.configChanged`.
     */
    setBaseOptions(options) {
        const changed = this._getChangedOptions(options, this._baseConfig);
        if (changed.length > 0) {
            this._baseConfig = options;
            const customizedKeys = Object.keys(this._config);
            const notOverridden = changed.filter(k => !customizedKeys.includes(k));
            if (notOverridden.length > 0) {
                this._configChanged.emit(notOverridden.reduce((agg, key) => {
                    agg[key] = this._baseConfig[key];
                    return agg;
                }, {}));
            }
        }
    }
    /**
     * Set config options for the editor.
     *
     * You will need to reconfigure the editor extensions by listening
     * to `EditorHandler.configChanged`.
     *
     * This method is preferred when setting several options. The
     * options are set within an operation, which only performs
     * the costly update at the end, and not after every option
     * is set.
     */
    setOptions(options) {
        const changed = this._getChangedOptions(options, this._config);
        if (changed.length > 0) {
            this._config = { ...options };
            this._configChanged.emit(changed.reduce((agg, key) => {
                var _a;
                agg[key] = (_a = this._config[key]) !== null && _a !== void 0 ? _a : this._baseConfig[key];
                return agg;
            }, {}));
        }
    }
    /**
     * Reconfigures the extension mapped with key with the provided value.
     *
     * @param view Editor view
     * @param key Parameter unique key
     * @param value Parameter value to be applied
     */
    reconfigureExtension(view, key, value) {
        const effect = this.getEffect(view.state, key, value);
        if (effect) {
            view.dispatch({
                effects: [effect]
            });
        }
    }
    /**
     * Reconfigures all the extensions mapped with the options from the
     * provided partial configuration.
     *
     * @param view Editor view
     * @param configuration Editor configuration
     */
    reconfigureExtensions(view, configuration) {
        const effects = Object.keys(configuration)
            .filter(key => this.has(key))
            .map(key => this.getEffect(view.state, key, configuration[key]));
        view.dispatch({
            effects: effects.filter(effect => effect !== null)
        });
    }
    /**
     * Appends extensions to the top-level configuration of the
     * editor.
     *
     * Injected extension cannot be removed.
     *
     * ### Notes
     * You should prefer registering a IEditorExtensionFactory instead
     * of this feature.
     *
     * @alpha
     * @param view Editor view
     * @param extension Editor extension to inject
     */
    injectExtension(view, extension) {
        view.dispatch({
            effects: StateEffect.appendConfig.of(extension)
        });
    }
    /**
     * Returns the list of initial extensions of an editor
     * based on the configuration.
     *
     * @returns The initial editor extensions
     */
    getInitialExtensions() {
        const configuration = { ...this._baseConfig, ...this._config };
        const extensions = [...this._immutables]
            .map(key => { var _a; return (_a = this.get(key)) === null || _a === void 0 ? void 0 : _a.instance(undefined); })
            .filter(ext => ext);
        for (const k of Object.keys(configuration)) {
            const builder = this.get(k);
            if (builder) {
                const value = configuration[k];
                extensions.push(builder.instance(value));
            }
        }
        return extensions;
    }
    /**
     * Get a extension builder
     * @param key Extension unique identifier
     * @returns The extension builder
     */
    get(key) {
        return this._configurableBuilderMap.get(key);
    }
    /**
     * Whether the editor has an extension for the identifier.
     *
     * @param key Extension unique identifier
     * @returns Extension existence
     */
    has(key) {
        return this._configurableBuilderMap.has(key);
    }
    getEffect(state, key, value) {
        var _a;
        const builder = this.get(key);
        return (_a = builder === null || builder === void 0 ? void 0 : builder.reconfigure(value)) !== null && _a !== void 0 ? _a : null;
    }
    _getChangedOptions(newConfig, oldConfig) {
        const changed = new Array();
        const newKeys = new Array();
        for (const [key, value] of Object.entries(newConfig)) {
            newKeys.push(key);
            if (oldConfig[key] !== value) {
                changed.push(key);
            }
        }
        // Add removed old keys
        changed.push(...Object.keys(oldConfig).filter(k => !newKeys.includes(k)));
        return changed;
    }
}
/**
 * CodeMirror extensions registry
 */
export class EditorExtensionRegistry {
    constructor() {
        this.configurationBuilder = new Map();
        this.configurationSchema = {};
        this.defaultOptions = {};
        this.handlers = new Set();
        this.immutableExtensions = new Set();
        this._baseConfiguration = {};
    }
    /**
     * Base editor configuration
     *
     * This is the default configuration optionally modified by the user;
     * e.g. through user settings.
     */
    get baseConfiguration() {
        return { ...this.defaultOptions, ...this._baseConfiguration };
    }
    set baseConfiguration(v) {
        if (!JSONExt.deepEqual(v, this._baseConfiguration)) {
            this._baseConfiguration = v;
            for (const handler of this.handlers) {
                handler.setBaseOptions(this.baseConfiguration);
            }
        }
    }
    /**
     * Default editor configuration
     *
     * This is the default configuration as defined when extensions
     * are registered.
     */
    get defaultConfiguration() {
        // Only options with schema should be JSON serializable
        // So we cannot use `JSONExt.deepCopy` on the default options.
        return Object.freeze({ ...this.defaultOptions });
    }
    /**
     * Editor configuration JSON schema
     */
    get settingsSchema() {
        return Object.freeze(JSONExt.deepCopy(this.configurationSchema));
    }
    /**
     * Add a default editor extension
     *
     * @template T Extension parameter type
     * @param factory Extension factory
     */
    addExtension(factory) {
        var _a;
        if (this.configurationBuilder.has(factory.name)) {
            throw new Error(`Extension named ${factory.name} is already registered.`);
        }
        this.configurationBuilder.set(factory.name, factory);
        if (typeof factory.default != 'undefined') {
            this.defaultOptions[factory.name] = factory.default;
        }
        if (factory.schema) {
            this.configurationSchema[factory.name] = {
                default: (_a = factory.default) !== null && _a !== void 0 ? _a : null,
                ...factory.schema
            };
            this.defaultOptions[factory.name] =
                this.configurationSchema[factory.name].default;
        }
    }
    /**
     * Create a new extensions handler for an editor
     *
     * @param options Extensions options and initial editor configuration
     */
    createNew(options) {
        const configuration = new Array();
        for (const [key, builder] of this.configurationBuilder.entries()) {
            const extension = builder.factory(options);
            if (extension) {
                configuration.push([key, extension]);
            }
        }
        const handler = new ExtensionsHandler({
            baseConfiguration: this.baseConfiguration,
            config: options.config,
            defaultExtensions: configuration
        });
        this.handlers.add(handler);
        handler.disposed.connect(() => {
            this.handlers.delete(handler);
        });
        return handler;
    }
}
/**
 * Editor extension registry namespace
 */
(function (EditorExtensionRegistry) {
    /**
     * Dynamically configurable editor extension.
     */
    class ConfigurableExtension {
        /**
         * Create a dynamic editor extension.
         *
         * @param builder Extension builder
         */
        constructor(builder) {
            this._compartment = new Compartment();
            this._builder = builder;
        }
        /**
         * Create an editor extension for the provided value.
         *
         * @param value Editor extension parameter value
         * @returns The editor extension
         */
        instance(value) {
            return this._compartment.of(this._builder(value));
        }
        /**
         * Reconfigure an editor extension.
         *
         * @param value Editor extension value
         * @returns Editor state effect
         */
        reconfigure(value) {
            return this._compartment.reconfigure(this._builder(value));
        }
    }
    /**
     * Immutable editor extension class
     */
    class ImmutableExtension {
        /**
         * Create an immutable editor extension.
         *
         * @param extension Extension
         */
        constructor(extension) {
            this._extension = extension;
        }
        /**
         * Create an editor extension.
         *
         * @returns The editor extension
         */
        instance() {
            return this._extension;
        }
        /**
         * Reconfigure an editor extension.
         *
         * This is a no-op
         */
        reconfigure() {
            // This is a no-op
            return null;
        }
    }
    /**
     * Creates a dynamically configurable editor extension.
     *
     * @param builder Extension builder
     * @return The extension
     */
    function createConfigurableExtension(builder) {
        return new ConfigurableExtension(builder);
    }
    EditorExtensionRegistry.createConfigurableExtension = createConfigurableExtension;
    /**
     * Creates a configurable extension returning
     * one of two extensions depending on a boolean value.
     *
     * @param truthy Extension to apply when the parameter is true
     * @param falsy Extension to apply when the parameter is false
     * @return The extension
     */
    function createConditionalExtension(truthy, falsy = []) {
        return new ConfigurableExtension(value => value ? truthy : falsy);
    }
    EditorExtensionRegistry.createConditionalExtension = createConditionalExtension;
    /**
     * Creates an immutable extension.
     *
     * @param extension Immutable extension
     * @return The extension
     */
    function createImmutableExtension(extension) {
        return new ImmutableExtension(extension);
    }
    EditorExtensionRegistry.createImmutableExtension = createImmutableExtension;
    /**
     * Get the default editor extensions.
     *
     * @returns CodeMirror 6 extension factories
     */
    function getDefaultExtensions(options = {}) {
        const { themes, translator } = options;
        const trans = (translator !== null && translator !== void 0 ? translator : nullTranslator).load('jupyter');
        const extensions = [
            Object.freeze({
                name: 'autoClosingBrackets',
                default: false,
                factory: () => createConditionalExtension(closeBrackets()),
                schema: {
                    type: 'boolean',
                    title: trans.__('Auto Closing Brackets')
                }
            }),
            Object.freeze({
                name: 'codeFolding',
                default: false,
                factory: () => createConditionalExtension(foldGutter()),
                schema: {
                    type: 'boolean',
                    title: trans.__('Code Folding')
                }
            }),
            Object.freeze({
                name: 'cursorBlinkRate',
                default: 1200,
                factory: () => createConfigurableExtension((value) => drawSelection({ cursorBlinkRate: value })),
                schema: {
                    type: 'number',
                    title: trans.__('Cursor blinking rate'),
                    description: trans.__('Half-period in milliseconds used for cursor blinking. The default blink rate is 1200ms. By setting this to zero, blinking can be disabled.')
                }
            }),
            Object.freeze({
                name: 'highlightActiveLine',
                default: false,
                factory: () => createConditionalExtension(highlightActiveLine()),
                schema: {
                    type: 'boolean',
                    title: trans.__('Highlight the active line')
                }
            }),
            Object.freeze({
                name: 'highlightTrailingWhitespace',
                default: false,
                factory: () => createConditionalExtension(highlightTrailingWhitespace()),
                schema: {
                    type: 'boolean',
                    title: trans.__('Highlight trailing white spaces')
                }
            }),
            Object.freeze({
                name: 'highlightWhitespace',
                default: false,
                factory: () => createConditionalExtension(highlightWhitespace()),
                schema: {
                    type: 'boolean',
                    title: trans.__('Highlight white spaces')
                }
            }),
            Object.freeze({
                name: 'indentUnit',
                default: '2',
                factory: () => createConfigurableExtension((value) => value == 'Tab'
                    ? indentUnit.of('\t')
                    : indentUnit.of(' '.repeat(parseInt(value, 10)))),
                schema: {
                    type: 'string',
                    title: trans.__('Indentation unit'),
                    description: trans.__('The indentation is a `Tab` or the number of spaces. This defaults to 2 spaces.'),
                    enum: ['Tab', '1', '2', '4', '8']
                }
            }),
            // Default keyboard shortcuts
            // TODO at some point we may want to get this configurable
            Object.freeze({
                name: 'keymap',
                default: [
                    ...defaultKeymap,
                    {
                        key: 'Tab',
                        run: StateCommands.indentMoreOrInsertTab,
                        shift: indentLess
                    }
                ],
                factory: () => createConfigurableExtension(value => keymap.of(value))
            }),
            Object.freeze({
                name: 'lineNumbers',
                default: true,
                factory: () => createConditionalExtension(lineNumbers()),
                schema: {
                    type: 'boolean',
                    title: trans.__('Line Numbers')
                }
            }),
            Object.freeze({
                name: 'lineWrap',
                factory: () => createConditionalExtension(EditorView.lineWrapping),
                default: true,
                schema: {
                    type: 'boolean',
                    title: trans.__('Line Wrap')
                }
            }),
            Object.freeze({
                name: 'matchBrackets',
                default: true,
                factory: () => createConditionalExtension(bracketMatching()),
                schema: {
                    type: 'boolean',
                    title: trans.__('Match Brackets')
                }
            }),
            Object.freeze({
                name: 'readOnly',
                default: false,
                factory: () => createConfigurableExtension((value) => [
                    EditorState.readOnly.of(value),
                    value
                        ? EditorView.editorAttributes.of({ class: READ_ONLY_CLASS })
                        : []
                ])
            }),
            Object.freeze({
                name: 'rulers',
                default: [],
                factory: () => createConfigurableExtension((value) => value.length > 0 ? rulers(value) : []),
                schema: {
                    type: 'array',
                    title: trans.__('Rulers'),
                    items: {
                        type: 'number',
                        minimum: 0
                    }
                }
            }),
            Object.freeze({
                name: 'scrollPastEnd',
                default: false,
                factory: (options) => options.inline ? null : createConditionalExtension(scrollPastEnd())
            }),
            Object.freeze({
                name: 'smartIndent',
                default: true,
                factory: () => createConditionalExtension(indentOnInput()),
                schema: {
                    type: 'boolean',
                    title: trans.__('Smart Indentation')
                }
            }),
            Object.freeze({
                name: 'tabSize',
                default: 4,
                factory: () => createConfigurableExtension((value) => EditorState.tabSize.of(value)),
                schema: {
                    type: 'number',
                    title: trans.__('Tab size')
                }
            }),
            Object.freeze({
                name: 'allowMultipleSelections',
                default: true,
                factory: () => createConfigurableExtension((value) => EditorState.allowMultipleSelections.of(value)),
                schema: {
                    type: 'boolean',
                    title: trans.__('Multiple selections')
                }
            }),
            Object.freeze({
                name: 'customStyles',
                factory: () => createConfigurableExtension(config => customTheme(config)),
                default: {
                    fontFamily: null,
                    fontSize: null,
                    lineHeight: null
                },
                schema: {
                    title: trans.__('Custom editor styles'),
                    type: 'object',
                    properties: {
                        fontFamily: {
                            type: ['string', 'null'],
                            title: trans.__('Font Family')
                        },
                        fontSize: {
                            type: ['number', 'null'],
                            minimum: 1,
                            maximum: 100,
                            title: trans.__('Font Size')
                        },
                        lineHeight: {
                            type: ['number', 'null'],
                            title: trans.__('Line Height')
                        }
                    },
                    additionalProperties: false
                }
            })
        ];
        if (themes) {
            extensions.push(Object.freeze({
                name: 'theme',
                default: 'jupyter',
                factory: () => createConfigurableExtension(value => themes.getTheme(value)),
                schema: {
                    type: 'string',
                    title: trans.__('Theme'),
                    description: trans.__('CodeMirror theme')
                }
            }));
        }
        if (translator) {
            extensions.push(Object.freeze({
                name: 'translation',
                // The list of internal strings is available at https://codemirror.net/examples/translate/
                default: {
                    // @codemirror/view
                    'Control character': trans.__('Control character'),
                    // @codemirror/commands
                    'Selection deleted': trans.__('Selection deleted'),
                    // @codemirror/language
                    'Folded lines': trans.__('Folded lines'),
                    'Unfolded lines': trans.__('Unfolded lines'),
                    to: trans.__('to'),
                    'folded code': trans.__('folded code'),
                    unfold: trans.__('unfold'),
                    'Fold line': trans.__('Fold line'),
                    'Unfold line': trans.__('Unfold line'),
                    // @codemirror/search
                    'Go to line': trans.__('Go to line'),
                    go: trans.__('go'),
                    Find: trans.__('Find'),
                    Replace: trans.__('Replace'),
                    next: trans.__('next'),
                    previous: trans.__('previous'),
                    all: trans.__('all'),
                    'match case': trans.__('match case'),
                    replace: trans.__('replace'),
                    'replace all': trans.__('replace all'),
                    close: trans.__('close'),
                    'current match': trans.__('current match'),
                    'replaced $ matches': trans.__('replaced $ matches'),
                    'replaced match on line $': trans.__('replaced match on line $'),
                    'on line': trans.__('on line'),
                    // @codemirror/autocomplete
                    Completions: trans.__('Completions'),
                    // @codemirror/lint
                    Diagnostics: trans.__('Diagnostics'),
                    'No diagnostics': trans.__('No diagnostics')
                },
                factory: () => createConfigurableExtension(value => EditorState.phrases.of(value))
            }));
        }
        return extensions;
    }
    EditorExtensionRegistry.getDefaultExtensions = getDefaultExtensions;
})(EditorExtensionRegistry || (EditorExtensionRegistry = {}));
//# sourceMappingURL=extension.js.map