(function (global, factory) {
    typeof exports === 'object' && typeof module !== 'undefined' ? factory(exports, require('@lumino/signaling')) :
    typeof define === 'function' && define.amd ? define(['exports', '@lumino/signaling'], factory) :
    (global = typeof globalThis !== 'undefined' ? globalThis : global || self, factory(global.lumino_disposable = {}, global.lumino_signaling));
})(this, (function (exports, signaling) { 'use strict';

    // Copyright (c) Jupyter Development Team.
    /**
     * A disposable object which delegates to a callback function.
     */
    class DisposableDelegate {
        /**
         * Construct a new disposable delegate.
         *
         * @param fn - The callback function to invoke on dispose.
         */
        constructor(fn) {
            this._fn = fn;
        }
        /**
         * Test whether the delegate has been disposed.
         */
        get isDisposed() {
            return !this._fn;
        }
        /**
         * Dispose of the delegate and invoke the callback function.
         */
        dispose() {
            if (!this._fn) {
                return;
            }
            let fn = this._fn;
            this._fn = null;
            fn();
        }
    }
    /**
     * An observable disposable object which delegates to a callback function.
     */
    class ObservableDisposableDelegate extends DisposableDelegate {
        constructor() {
            super(...arguments);
            this._disposed = new signaling.Signal(this);
        }
        /**
         * A signal emitted when the delegate is disposed.
         */
        get disposed() {
            return this._disposed;
        }
        /**
         * Dispose of the delegate and invoke the callback function.
         */
        dispose() {
            if (this.isDisposed) {
                return;
            }
            super.dispose();
            this._disposed.emit(undefined);
            signaling.Signal.clearData(this);
        }
    }
    /**
     * An object which manages a collection of disposable items.
     */
    class DisposableSet {
        constructor() {
            this._isDisposed = false;
            this._items = new Set();
        }
        /**
         * Test whether the set has been disposed.
         */
        get isDisposed() {
            return this._isDisposed;
        }
        /**
         * Dispose of the set and the items it contains.
         *
         * #### Notes
         * Items are disposed in the order they are added to the set.
         */
        dispose() {
            if (this._isDisposed) {
                return;
            }
            this._isDisposed = true;
            this._items.forEach(item => {
                item.dispose();
            });
            this._items.clear();
        }
        /**
         * Test whether the set contains a specific item.
         *
         * @param item - The item of interest.
         *
         * @returns `true` if the set contains the item, `false` otherwise.
         */
        contains(item) {
            return this._items.has(item);
        }
        /**
         * Add a disposable item to the set.
         *
         * @param item - The item to add to the set.
         *
         * #### Notes
         * If the item is already contained in the set, this is a no-op.
         */
        add(item) {
            this._items.add(item);
        }
        /**
         * Remove a disposable item from the set.
         *
         * @param item - The item to remove from the set.
         *
         * #### Notes
         * If the item is not contained in the set, this is a no-op.
         */
        remove(item) {
            this._items.delete(item);
        }
        /**
         * Remove all items from the set.
         */
        clear() {
            this._items.clear();
        }
    }
    /**
     * The namespace for the `DisposableSet` class statics.
     */
    (function (DisposableSet) {
        /**
         * Create a disposable set from an iterable of items.
         *
         * @param items - The iterable object of interest.
         *
         * @returns A new disposable initialized with the given items.
         */
        function from(items) {
            let set = new DisposableSet();
            for (const item of items) {
                set.add(item);
            }
            return set;
        }
        DisposableSet.from = from;
    })(DisposableSet || (DisposableSet = {}));
    /**
     * An observable object which manages a collection of disposable items.
     */
    class ObservableDisposableSet extends DisposableSet {
        constructor() {
            super(...arguments);
            this._disposed = new signaling.Signal(this);
        }
        /**
         * A signal emitted when the set is disposed.
         */
        get disposed() {
            return this._disposed;
        }
        /**
         * Dispose of the set and the items it contains.
         *
         * #### Notes
         * Items are disposed in the order they are added to the set.
         */
        dispose() {
            if (this.isDisposed) {
                return;
            }
            super.dispose();
            this._disposed.emit(undefined);
            signaling.Signal.clearData(this);
        }
    }
    /**
     * The namespace for the `ObservableDisposableSet` class statics.
     */
    (function (ObservableDisposableSet) {
        /**
         * Create an observable disposable set from an iterable of items.
         *
         * @param items - The iterable object of interest.
         *
         * @returns A new disposable initialized with the given items.
         */
        function from(items) {
            let set = new ObservableDisposableSet();
            for (const item of items) {
                set.add(item);
            }
            return set;
        }
        ObservableDisposableSet.from = from;
    })(ObservableDisposableSet || (ObservableDisposableSet = {}));

    exports.DisposableDelegate = DisposableDelegate;
    exports.DisposableSet = DisposableSet;
    exports.ObservableDisposableDelegate = ObservableDisposableDelegate;
    exports.ObservableDisposableSet = ObservableDisposableSet;

}));
//# sourceMappingURL=index.js.map
