import { ServerConnection } from '@jupyterlab/services';
import { TranslationBundle } from '@jupyterlab/translation';
import { LabIcon, VDomModel, VDomRenderer } from '@jupyterlab/ui-components';
import { ReadonlyJSONObject } from '@lumino/coreutils';
import { ISignal } from '@lumino/signaling';
import { VirtualElement } from '@lumino/virtualdom';
import { Panel, SplitPanel, TabBar, Widget } from '@lumino/widgets';
import * as React from 'react';
/**
 * A license viewer
 */
export declare class Licenses extends SplitPanel {
    readonly model: Licenses.Model;
    constructor(options: Licenses.IOptions);
    /**
     * Handle disposing of the widget
     */
    dispose(): void;
    /**
     * Initialize the left area for filters and bundles
     */
    protected initLeftPanel(): void;
    /**
     * Initialize the filters
     */
    protected initFilters(): void;
    /**
     * Initialize the listing of available bundles
     */
    protected initBundles(): void;
    /**
     * Initialize the listing of packages within the current bundle
     */
    protected initGrid(): void;
    /**
     * Initialize the full text of the current package
     */
    protected initLicenseText(): void;
    /**
     * Event handler for updating the model with the current bundle
     */
    protected onBundleSelected(): void;
    /**
     * Update the bundle tabs.
     */
    protected _updateBundles(): void;
    /**
     * An area for selecting licenses by bundle and filters
     */
    protected _leftPanel: Panel;
    /**
     * Filters on visible licenses
     */
    protected _filters: Licenses.Filters;
    /**
     * Tabs reflecting available bundles
     */
    protected _bundles: TabBar<Widget>;
    /**
     * A grid of the current bundle's packages' license metadata
     */
    protected _grid: Licenses.Grid;
    /**
     * The currently-selected package's full license text
     */
    protected _licenseText: Licenses.FullText;
}
/** A namespace for license components */
export declare namespace Licenses {
    /** The information about a license report format  */
    interface IReportFormat {
        title: string;
        icon: LabIcon;
        id: string;
    }
    /**
     * License report formats understood by the server (once lower-cased)
     */
    const REPORT_FORMATS: Record<string, IReportFormat>;
    /**
     * The default format (most human-readable)
     */
    const DEFAULT_FORMAT = "markdown";
    /**
     * Options for instantiating a license viewer
     */
    interface IOptions {
        model: Model;
    }
    /**
     * Options for instantiating a license model
     */
    interface IModelOptions extends ICreateArgs {
        licensesUrl: string;
        serverSettings?: ServerConnection.ISettings;
        trans: TranslationBundle;
    }
    /**
     * The JSON response from the API
     */
    interface ILicenseResponse {
        bundles: {
            [key: string]: ILicenseBundle;
        };
    }
    /**
     * A top-level report of the licenses for all code included in a bundle
     *
     * ### Note
     *
     * This is roughly informed by the terms defined in the SPDX spec, though is not
     * an SPDX Document, since there seem to be several (incompatible) specs
     * in that repo.
     *
     * @see https://github.com/spdx/spdx-spec/blob/development/v2.2.1/schemas/spdx-schema.json
     **/
    interface ILicenseBundle extends ReadonlyJSONObject {
        packages: IPackageLicenseInfo[];
    }
    /**
     * A best-effort single bundled package's information.
     *
     * ### Note
     *
     * This is roughly informed by SPDX `packages` and `hasExtractedLicenseInfos`,
     * as making it conformant would vastly complicate the structure.
     *
     * @see https://github.com/spdx/spdx-spec/blob/development/v2.2.1/schemas/spdx-schema.json
     **/
    interface IPackageLicenseInfo extends ReadonlyJSONObject {
        /**
         * the name of the package as it appears in package.json
         */
        name: string;
        /**
         * the version of the package, or an empty string if unknown
         */
        versionInfo: string;
        /**
         * an SPDX license identifier or LicenseRef, or an empty string if unknown
         */
        licenseId: string;
        /**
         * the verbatim extracted text of the license, or an empty string if unknown
         */
        extractedText: string;
    }
    /**
     * The format information for a download
     */
    interface IDownloadOptions {
        format: string;
    }
    /**
     * The fields which can be filtered
     */
    type TFilterKey = 'name' | 'versionInfo' | 'licenseId';
    interface ICreateArgs {
        currentBundleName?: string | null;
        packageFilter?: Partial<IPackageLicenseInfo> | null;
        currentPackageIndex?: number | null;
    }
    /**
     * A model for license data
     */
    class Model extends VDomModel implements ICreateArgs {
        constructor(options: IModelOptions);
        /**
         * Handle the initial request for the licenses from the server.
         */
        initLicenses(): Promise<void>;
        /**
         * Create a temporary download link, and emulate clicking it to trigger a named
         * file download.
         */
        download(options: IDownloadOptions): Promise<void>;
        /**
         * A promise that resolves when the licenses from the server change
         */
        get selectedPackageChanged(): ISignal<Model, void>;
        /**
         * A promise that resolves when the trackable data changes
         */
        get trackerDataChanged(): ISignal<Model, void>;
        /**
         * The names of the license bundles available
         */
        get bundleNames(): string[];
        /**
         * The current license bundle
         */
        get currentBundleName(): string | null;
        /**
         * Set the current license bundle, and reset the selected index
         */
        set currentBundleName(currentBundleName: string | null);
        /**
         * A promise that resolves when the licenses are available from the server
         */
        get licensesReady(): Promise<void>;
        /**
         * All the license bundles, keyed by the distributing packages
         */
        get bundles(): null | {
            [key: string]: ILicenseBundle;
        };
        /**
         * The index of the currently-selected package within its license bundle
         */
        get currentPackageIndex(): number | null;
        /**
         * Update the currently-selected package within its license bundle
         */
        set currentPackageIndex(currentPackageIndex: number | null);
        /**
         * The license data for the currently-selected package
         */
        get currentPackage(): IPackageLicenseInfo | null;
        /**
         * A translation bundle
         */
        get trans(): TranslationBundle;
        get title(): string;
        /**
         * The current package filter
         */
        get packageFilter(): Partial<IPackageLicenseInfo>;
        set packageFilter(packageFilter: Partial<IPackageLicenseInfo>);
        /**
         * Get filtered packages from current bundle where at least one token of each
         * key is present.
         */
        getFilteredPackages(allRows: IPackageLicenseInfo[]): IPackageLicenseInfo[];
        private _selectedPackageChanged;
        private _trackerDataChanged;
        private _serverResponse;
        private _licensesUrl;
        private _serverSettings;
        private _currentBundleName;
        private _trans;
        private _currentPackageIndex;
        private _licensesReady;
        private _packageFilter;
    }
    /**
     * A filter form for limiting the packages displayed
     */
    class Filters extends VDomRenderer<Model> {
        constructor(model: Model);
        protected render(): JSX.Element;
        /**
         * Render a filter input
         */
        protected renderFilter: (key: TFilterKey) => JSX.Element;
        /**
         * Handle a filter input changing
         */
        protected onFilterInput: (evt: React.ChangeEvent<HTMLInputElement>) => void;
    }
    /**
     * A fancy bundle renderer with the package count
     */
    class BundleTabRenderer extends TabBar.Renderer {
        /**
         * A model of the state of license viewing as well as the underlying data
         */
        model: Model;
        readonly closeIconSelector = ".lm-TabBar-tabCloseIcon";
        constructor(model: Model);
        /**
         * Render a full bundle
         */
        renderTab(data: TabBar.IRenderData<Widget>): VirtualElement;
        /**
         * Render the package count
         */
        renderCountBadge(data: TabBar.IRenderData<Widget>): VirtualElement;
    }
    /**
     * A grid of licenses
     */
    class Grid extends VDomRenderer<Licenses.Model> {
        constructor(model: Licenses.Model);
        /**
         * Render a grid of package license information
         */
        protected render(): JSX.Element;
        /**
         * Render a single package's license information
         */
        protected renderRow: (row: Licenses.IPackageLicenseInfo, index: number) => JSX.Element;
    }
    /**
     * A package's full license text
     */
    class FullText extends VDomRenderer<Model> {
        constructor(model: Model);
        /**
         * Render the license text, or a null state if no package is selected
         */
        protected render(): JSX.Element[];
    }
}
